/**
//@HEADER
// ************************************************************************
//
//                   Trios: Trilinos I/O Support
//                 Copyright 2011 Sandia Corporation
//
// Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
// the U.S. Government retains certain rights in this software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//Questions? Contact Ron A. Oldfield (raoldfi@sandia.gov)
//
// *************************************************************************
//@HEADER
 */
/*
 * This is sample code generated by rpcgen.
 * These are only templates and you can use them
 * as a guideline for developing your own functions.
 */

#ifndef __STDC_CONSTANT_MACROS
#define __STDC_CONSTANT_MACROS
#endif

#include "Trios_config.h"
#ifdef HAVE_TRIOS_PNETCDF

#include <stdint.h>
#include <inttypes.h>
#include <sys/stat.h>

#include <assert.h>

#include <pnetcdf.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <map>
#include <list>
#include <vector>
#include <string>
#include <vector>
#include <algorithm>
using namespace std;

#include <Trios_nssi_client.h>

#include "Trios_nssi_fprint_types.h"
#include "Trios_nnti_fprint_types.h"

#include "netcdf_args.h"
#include "netcdf_debug.h"  /* netcdf_debug_level */
#include "netcdf_config.h"

#include "NcGroupInfo.h"
#include "NcVarInfo.h"

#include <mpi.h>

#include "aggregation.h"

#include "create_subchunks.h"

#include "Trios_timer.h"



/** Map of open groups */
//map<int, NcGroupInfo *> group_map;

//static nssi_service svc;
static int default_svc;
static MPI_Comm ncmpi_collective_op_comm;
static int global_rank=-1;
static int collective_op_rank=-1;
static nssi_service *svcs;
struct netcdf_config nc_cfg;


size_t* NC_coord_zero;
size_t* NC_coord_one;


enum nc_data_mode {
    NC_DEFINE_MODE,
    NC_INDEP_DATA_MODE,
    NC_COLL_DATA_MODE,
    NC_ASYNC_CLOSED,
    NC_CLOSED
};
typedef enum nc_data_mode nc_data_mode;

typedef list<nssi_request *> request_list_t;
typedef list<nssi_request *>::iterator request_list_iterator_t;

struct nc_file_state {
    nc_data_mode mode;
    request_list_t outstanding_sync_requests;
    request_list_t outstanding_close_requests;
};
typedef struct nc_file_state nc_file_state;

/** Association between ncid and netcdf file state */
map<int, nc_file_state *> file_state_map;


#undef BUILD_FOR_QACINA

#if defined(BUILD_FOR_QACINA)
#define MANIPULATE_ARGS(netcdf_id) { netcdf_id--; }
#undef USE_ASYNC_CLOSE
#undef USE_ASYNC_SYNC
#else
#define MANIPULATE_ARGS(netcdf_id) { }
#define USE_ASYNC_CLOSE 1
#define USE_ASYNC_SYNC  1
#endif


/* ************************ Private functions ******************* */


static int get_config_from_env(struct netcdf_config *netcdf_cfg)
{
    char *env_contact_file=getenv("NETCDF_CONTACT_INFO");
    char *env_write_mode=getenv("NSSI_WRITE_MODE");
    char *env_use_subchunking=getenv("NSSI_USE_SUBCHUNKING");

    memset(netcdf_cfg, 0, sizeof(struct netcdf_config));

    if ((env_write_mode==NULL) || (!strcmp(env_write_mode, "WRITE_DIRECT"))) {
        netcdf_cfg->write_type = WRITE_DIRECT;
        log_debug(netcdf_debug_level, "using %s", env_write_mode);
    } else if (!strcmp(env_write_mode, "WRITE_AGGREGATE_INDEPENDENT")) {
        netcdf_cfg->write_type = WRITE_AGGREGATE_INDEPENDENT;
        log_debug(netcdf_debug_level, "using %s", env_write_mode);
    } else if (!strcmp(env_write_mode, "WRITE_AGGREGATE_COLLECTIVE")) {
        netcdf_cfg->write_type = WRITE_AGGREGATE_COLLECTIVE;
        log_debug(netcdf_debug_level, "using %s", env_write_mode);
    } else if (!strcmp(env_write_mode, "WRITE_CACHING_INDEPENDENT")) {
        netcdf_cfg->write_type = WRITE_CACHING_INDEPENDENT;
        log_debug(netcdf_debug_level, "using %s", env_write_mode);
    } else if (!strcmp(env_write_mode, "WRITE_CACHING_COLLECTIVE")) {
        netcdf_cfg->write_type = WRITE_CACHING_COLLECTIVE;
        log_debug(netcdf_debug_level, "using %s", env_write_mode);
    }

    sscanf(env_use_subchunking, "%lu", &netcdf_cfg->use_subchunking);
    log_debug(netcdf_debug_level, "use_subchunking %lu", netcdf_cfg->use_subchunking);

    if (env_contact_file != NULL) {
        int i;
        FILE *f=NULL;
        long  fsize=0, bytes_to_read=0, bytes_read=0;
        char *fbuf;
        int lcount;
        char *start, *end;

        struct stat sbuf;

        while (stat(env_contact_file, &sbuf)) { log_debug(netcdf_debug_level, "%s: %s", env_contact_file, strerror(errno)); usleep(500); }

        f=fopen(env_contact_file, "r");
        if (!f) {
            perror(env_contact_file);
        } else {
            fsize = sbuf.st_size;
            fbuf=(char *)calloc(fsize+1,1);

            bytes_to_read=fsize;

            while(bytes_to_read > 0) {
                bytes_read+=fread(fbuf+bytes_read, 1, bytes_to_read, f);
                if (bytes_read != fsize) {
                    log_error(netcdf_debug_level, "bytes_read(%lld) != fsize(%lld)", (int64_t)bytes_read, (int64_t)fsize);
                }
                bytes_to_read=fsize-bytes_read;

                log_debug(netcdf_debug_level, "fsize(%lld) bytes_read(%lld) bytes_to_read(%lld)", (int64_t)fsize, (int64_t)bytes_read, (int64_t)bytes_to_read);
            }

            fclose(f);

            lcount=0;
            start=end=fbuf;
            do {
                end=strchr(start, '\n');
                if (end!=NULL) {
                    lcount++;
                    start=end+1;
                }
            } while(end!=NULL);

            netcdf_cfg->num_servers=lcount;

            log_debug(netcdf_debug_level, "lcount(%lld)", (int64_t)lcount);

            netcdf_cfg->netcdf_server_urls = (char **)calloc(lcount, sizeof(char *));
            if (!netcdf_cfg->netcdf_server_urls) {
                log_error(netcdf_debug_level, "could not allocate netcdf services");
                return NSSI_ENOMEM;
            }
            for (i=0;i<lcount;i++) {
                netcdf_cfg->netcdf_server_urls[i] = (char *)calloc(NNTI_URL_LEN, sizeof(char));
                if (!netcdf_cfg->netcdf_server_urls[i]) {
                    log_error(netcdf_debug_level, "could not allocate netcdf services");
                    return NSSI_ENOMEM;
                }
            }
            start=end=fbuf;
            for (i=0;i<lcount;i++) {
                end=strchr(start, '\n');
                if (end==NULL) {
                    log_error(netcdf_debug_level, "URL #%d is missing");
                } else {
                    int url_len=end-start;
                    if (url_len==0) {
                        log_error(netcdf_debug_level, "URL #%d is missing");
                    }
                    memcpy(netcdf_cfg->netcdf_server_urls[i], start, url_len);
                    netcdf_cfg->netcdf_server_urls[i][url_len]='\0';
                    log_debug(netcdf_debug_level, "netcdf_server_url[%d]=%s", i, netcdf_cfg->netcdf_server_urls[i]);
                    start=end+1;
                }
            }
        }
    }

    return(NSSI_OK);
}


extern "C"
void netcdf_client_fini(void)
{
    int rc=NSSI_OK;

    log_debug(netcdf_debug_level, "enter");

    if (collective_op_rank == 0) {
        rc=nssi_kill(&svcs[default_svc], 0, 1000);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }

    log_debug(netcdf_debug_level, "exit");

    return;
}


/**
 * Initialize the netcdf client interface.
 *
 * The init function relies on environment variables to know where to
 * find the netcdf service.
 */
extern "C"
int netcdf_client_init(void)
{
    static int initialized = 0;
    int rc;

    nssi_size i;

    char *log_level_str=NULL;
    char *logfile_base=NULL;
    char *log_file_per_client=NULL;

    char logfile[256];
    log_level debug_level;

    int is_mpi_init, np;

    if (initialized) {
        return 0;
    }

    trios_declare_timer(InitTime);
    trios_start_timer(InitTime);

    log_debug(netcdf_debug_level, "enter");

    MPI_Initialized(&is_mpi_init);
    if (is_mpi_init == FALSE) {
        MPI_Init(NULL,NULL);
    }

    MPI_Comm_rank(MPI_COMM_WORLD, &global_rank);
    MPI_Comm_size(MPI_COMM_WORLD, &np);

#if defined(BUILD_FOR_QACINA)
    /* if we exit normally, we should cleanup after ourselves */
    atexit(netcdf_client_fini);
#endif

    log_level_str=getenv("NSSI_LOG_LEVEL");
    logfile_base=getenv("NSSI_LOG_FILE");
    log_file_per_client=getenv("NSSI_LOG_FILE_PER_NODE");

    trios_declare_timer(LogInitTime);
    trios_start_timer(LogInitTime);
    if (logfile_base != NULL) {
        if (log_file_per_client != NULL) {
            sprintf(logfile, "%s.%04d", logfile_base, global_rank);
        } else {
            sprintf(logfile, "%s", logfile_base);
        }
    }
    if (log_level_str != NULL) {
        sscanf(log_level_str, "%d", (int*)&debug_level);
    }
    if (logfile_base != NULL) {
        logger_init(debug_level, logfile);
    } else {
        logger_init(debug_level, NULL);
    }
    netcdf_debug_level=debug_level;
    trios_stop_timer("log init", LogInitTime);


    nssi_rpc_init(NSSI_DEFAULT_TRANSPORT, NSSI_DEFAULT_ENCODE, NULL);

    /* Register the client operations */
    NSSI_REGISTER_CLIENT_STUB(NETCDF_CREATE_OP, nc_create_args, void, nc_create_res);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_OPEN_OP, nc_open_args, void, nc_open_res);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_DEF_DIM_OP, nc_def_dim_args, void, int);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_DEF_VAR_OP, nc_def_var_args, void, int);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_GET_ATT_OP, nc_get_att_args, void, void);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_PUT_ATT_OP, nc_put_att_args, void, void);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_PUT_VARS_OP, nc_put_vars_args, void, void);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_GET_VARS_OP, nc_get_vars_args, void, void);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_REDEF_OP, int, void, void);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_ENDDEF_OP, int, void, void);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_SYNC_OP, int, void, void);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_CLOSE_OP, int, void, void);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_BEGIN_INDEP_OP, int, void, void);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_END_INDEP_OP, int, void, void);
    NSSI_REGISTER_CLIENT_STUB(NETCDF_SET_FILL_OP, nc_set_fill_args, void, nc_set_fill_res);

    trios_declare_timer(ContactFileTime);
    trios_start_timer(ContactFileTime);
    get_config_from_env(&nc_cfg);
    trios_stop_timer("contact file", ContactFileTime);
    /* all ranks connect to all services */

    if (np <= nc_cfg.num_servers) {
        default_svc = global_rank;
    } else {
        uint32_t extra_clients = np%nc_cfg.num_servers;
        if ((extra_clients) > 0) {
            uint32_t num_big_groups   = extra_clients;
            uint32_t num_small_groups = nc_cfg.num_servers - num_big_groups;
            uint32_t small_group_size = np/nc_cfg.num_servers;
            uint32_t big_group_size   = small_group_size+1;
            int32_t  max_bg_rank      = (big_group_size * num_big_groups) - 1;
            log_debug(debug_level, "nbg(%u) nsg(%u) sgs(%u) bgs(%u) mbgr(%d)",
                    num_big_groups, num_small_groups, small_group_size, big_group_size, max_bg_rank);
            if (global_rank <= max_bg_rank) {
                default_svc = global_rank/big_group_size;
            } else {
                default_svc = ((global_rank - max_bg_rank - 1)/small_group_size)+num_big_groups;
            }
        } else {
            default_svc = global_rank/(np/nc_cfg.num_servers);
        }
    }

    log_debug(debug_level, "global_rank(%d) default_service(%d)", global_rank, default_svc);

    /* create a new communicator for just those clients, who share a default service. */
    MPI_Comm_split(MPI_COMM_WORLD, default_svc, global_rank, &ncmpi_collective_op_comm);
    /* find my rank in the new communicator */
    MPI_Comm_rank(ncmpi_collective_op_comm, &collective_op_rank);

    log_debug(debug_level, "global_rank(%d) collective_op_rank(%d) default_service(%d)", global_rank, collective_op_rank, default_svc);


    svcs=(nssi_service *)calloc(nc_cfg.num_servers, sizeof(nssi_service));
    /* !global_rank0 has a preferred server for data transfers.  connect to preferred server.
     * connect to other servers on-demand.
     */
    trios_declare_timer(GetSvcTime);
    trios_start_timer(GetSvcTime);
    rc = nssi_get_service(
            NSSI_DEFAULT_TRANSPORT,
            nc_cfg.netcdf_server_urls[default_svc],
            -1,
            &svcs[default_svc]);
    if (rc != NSSI_OK) {
        log_error(debug_level, "Couldn't connect to netcdf master: %s", nssi_err_str(rc));
        return rc;
    }
    trios_stop_timer("get svc[default_svc]", GetSvcTime);

    trios_stop_timer("netcdf client init", InitTime);

    NC_coord_zero = (size_t*)malloc(sizeof(size_t)*NC_MAX_VAR_DIMS);
    if(NC_coord_zero == NULL) abort();
    NC_coord_one = (size_t*)malloc(sizeof(size_t)*NC_MAX_VAR_DIMS);
    if(NC_coord_one == NULL) abort();
    for(i=0;i<NC_MAX_VAR_DIMS;i++) {
        NC_coord_one[i] = 1;
        NC_coord_zero[i] = 0;
    }

    initialized = 1;

    return 0;
}


/* ************************ NETCDF LIBRARY STUBS ******************* */

/*
 *
 */
extern "C"
const char *nc_strerror(int ncerr)
{
    static char data[10];
    sprintf(data, "%d", ncerr);
    return data;
}

extern "C"
nc_file_state *new_file_state(nc_data_mode new_mode)
{
    nc_file_state *state=new nc_file_state;
    state->mode = new_mode;

    return(state);
}

/*
 * These functions replace the normal netcdf library functions.
 */
extern "C"
int _nc_create(
        const char *path,
        int cmode,
        size_t initialsz,
        size_t *chunksizehintp,
        int *ncidp)
{
    int rc = NSSI_OK;
    nc_create_args args;
    log_level debug_level = netcdf_debug_level;
    nc_create_res res;
    trios_declare_timer(MulticastTime);
    trios_declare_timer(WaitAnyTime);
    vector<nssi_request *>reqs;

    rc = netcdf_client_init();
    if (rc != 0) {
        log_error(debug_level, "failed to initialize netcdf_client");
        goto cleanup;
    }

    log_debug(netcdf_debug_level, "enter");

    log_debug(debug_level, "Calling _nc_create");

    memset(&args, 0, sizeof(args));
    args.path = (char *)path;
    args.cmode = cmode;
    args.initialsz = initialsz;
    args.chunksizehint = *chunksizehintp;
    args.write_type = nc_cfg.write_type;
    args.num_participants = nc_cfg.num_participants;

    log_debug(debug_level, "calling _nc_create(%s, %d)", path, cmode);

    if (collective_op_rank == 0) {
        trios_start_timer(MulticastTime);
        /* call the remote method */
        rc = nssi_call_rpc_sync(&svcs[default_svc],
                NETCDF_CREATE_OP,
                &args,
                NULL,
                0,
                &res);
        trios_stop_timer("create Multicast", MulticastTime);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }


    *ncidp = res.ncid;

    /* share the file descriptor with the rest of our group */
    MPI_Bcast(ncidp, 1, MPI_INT, 0, ncmpi_collective_op_comm);

    log_debug(debug_level, "got result rc=%d, ncid=%d", rc, *ncidp);

    if (rc == NC_NOERR) {
        *chunksizehintp = res.chunksizehint;
        file_state_map[*ncidp]=new_file_state(NC_DEFINE_MODE);
        if (global_rank == 0) {
            /* create the client-side metadata for the dataset */
            if (group_map.find(*ncidp) == group_map.end()) {
                NcGroupInfo *group = new NcGroupInfo(*ncidp,
                        NcFileInfo(path, cmode, initialsz, *chunksizehintp));
                group_map[*ncidp] = group;
            }
            else {
                log_error(debug_level, "dataset already exists for ncid=%d", *ncidp);
                rc = NC_EEXIST;
                goto cleanup;
            }
        }
    }

cleanup:
    xdr_free((xdrproc_t)xdr_nc_create_res, (char *)&res);

    log_debug(debug_level, "Finished creating dataset ncid=%d, rc=%d", *ncidp, rc);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_create(
                const char *path,
                int cmode,
                int *ncidp)
{
    int rc=NC_NOERR;
    size_t initialsz = 0;
    size_t chunksizehint = NC_SIZEHINT_DEFAULT;

    log_debug(netcdf_debug_level, "enter");

    rc = _nc_create(path, cmode, initialsz, &chunksizehint, ncidp);

#if defined(BUILD_FOR_QACINA)
    (*ncidp)++;
#endif

    return(rc);
}

extern "C"
int _nc_open(
        const char *path,
        int mode,
        size_t *chunksizehintp,
        int *ncidp)
{
    int rc = NSSI_OK;
    nc_open_args args;
    log_level debug_level = netcdf_debug_level;
    nc_open_res res;
    trios_declare_timer(MulticastTime);
    trios_declare_timer(WaitAnyTime);
    vector<nssi_request *>reqs;

    rc = netcdf_client_init();
    if (rc != 0) {
        log_error(debug_level, "failed to initialize netcdf_client");
        goto cleanup;
    }

    log_debug(netcdf_debug_level, "enter");

    log_debug(netcdf_debug_level, "Calling _nc_open");

    memset(&res, 0, sizeof(res));
    memset(&args, 0, sizeof(args));
    args.path = (char *)path;
    args.mode = mode;
    args.chunksizehint = *chunksizehintp;
    args.write_type = nc_cfg.write_type;
    args.num_participants = nc_cfg.num_participants;

    log_debug(debug_level, "calling _nc_open(%s, %d)", path, mode);

    if (collective_op_rank == 0) {
        trios_start_timer(MulticastTime);
        /* call the remote method */
        rc = nssi_call_rpc_sync(&svcs[default_svc],
                NETCDF_OPEN_OP,
                &args,
                NULL,
                0,
                &res);
        trios_stop_timer("create Multicast", MulticastTime);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }

    *ncidp = res.root_group.ncid;

    /* share the file descriptor with the rest of our group */
    MPI_Bcast(ncidp, 1, MPI_INT, 0, ncmpi_collective_op_comm);

    log_debug(debug_level, "got result rc=%d, ncid=%d", rc, *ncidp);

    if (rc == NC_NOERR) {
        *chunksizehintp = res.chunksizehint;
        file_state_map[*ncidp]=new_file_state(NC_COLL_DATA_MODE);
        if (global_rank == 0) {
            /* create an entry in list of open datasets */
            if (group_map.find(*ncidp) == group_map.end()) {
                NcGroupInfo *group = new NcGroupInfo(res.root_group, NcFileInfo(path, mode, 0, *chunksizehintp));
                group_map[*ncidp] = group;
            }
            else {
                log_error(debug_level, "dataset already exists for ncid=%d", *ncidp);
                rc = NC_EEXIST;
                goto cleanup;
            }
        }
    }

cleanup:
    xdr_free((xdrproc_t)xdr_nc_open_res, (char *)&res);

    log_debug(debug_level, "Finished opening dataset ncid=%d, rc=%d", *ncidp, rc);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_open(
        const char *path,
        int mode,
        int *ncidp)
{
    int rc=NC_NOERR;

    size_t chunksizehint = NC_SIZEHINT_DEFAULT;

    log_debug(netcdf_debug_level, "enter");

    rc = _nc_open(path, mode, &chunksizehint, ncidp);

#if defined(BUILD_FOR_QACINA)
    (*ncidp)++;
#endif

    return(rc);
}

/**
 * Delete a dataset.
 *
 * This operation isn't mentioned in the netcdf-api document, but it is in the
 * include file.  It is just a wrapper around the MPI_File_delete function.
 */
extern "C"
int nc_delete(const char *filename)
{
    log_debug(netcdf_debug_level, "enter");

    return MPI_File_delete(const_cast<char *>(filename), MPI_INFO_NULL);
}

extern "C"
int _nc_set_fill(int ncid, int fillmode, int *old_modep)
{
    *old_modep = fillmode;
    int rc = NC_NOERR;
    trios_declare_timer(MulticastTime);
    trios_declare_timer(WaitAnyTime);
    vector<nssi_request *>reqs;

    log_debug(netcdf_debug_level, "enter");

    log_debug(netcdf_debug_level, "Calling _nc_set_fill (fillmode=%d)", fillmode);

    nc_set_fill_args args;
    memset(&args, 0, sizeof(args));
    args.ncid = ncid;
    args.new_fill_mode = fillmode;

    nc_set_fill_res res;
    memset(&res, 0, sizeof(res));

    if (collective_op_rank == 0) {
        trios_start_timer(MulticastTime);
        /* call the remote method */
        rc = nssi_call_rpc_sync(&svcs[default_svc],
                NETCDF_SET_FILL_OP,
                &args,
                NULL,
                0,
                &res);
        trios_stop_timer("_nc_set_fill Multicast", MulticastTime);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }

    *old_modep=res.old_fill_mode;

    log_debug(netcdf_debug_level, "got result from _nc_set_fill: rc=%d, fillmode=%d, old_modep=%d",rc, fillmode, *old_modep);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_set_fill(int ncid, int fillmode, int *old_modep)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_set_fill(ncid, fillmode, old_modep);
}

extern "C"
int _nc_begin_indep_data(int ncid)
{
    int rc = NC_NOERR;
    log_level debug_level = netcdf_debug_level;

    log_debug(netcdf_debug_level, "enter");

    log_debug(debug_level, "Calling remote _nc_begin_indep_stub");

    MPI_Barrier(MPI_COMM_WORLD);

    if (collective_op_rank == 0) {
        /* call the remote method */
        rc = nssi_call_rpc_sync(&svcs[default_svc],
                NETCDF_BEGIN_INDEP_OP,
                &ncid,
                NULL,
                0,
                NULL);

        file_state_map[ncid]->mode = NC_INDEP_DATA_MODE;

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }

    MPI_Barrier(MPI_COMM_WORLD);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_begin_indep_data(int ncid)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_begin_indep_data(ncid);
}

extern "C"
int _nc_end_indep_data(int ncid)
{
    int rc = NC_NOERR;
    log_level debug_level = netcdf_debug_level;

    log_debug(netcdf_debug_level, "enter");

    log_debug(debug_level, "Calling remote _nc_end_indep_stub");

    MPI_Barrier(MPI_COMM_WORLD);

    if (collective_op_rank == 0) {
        /* call the remote method */
        rc = nssi_call_rpc_sync(&svcs[default_svc],
                NETCDF_END_INDEP_OP,
                &ncid,
                NULL,
                0,
                NULL);

        file_state_map[ncid]->mode = NC_COLL_DATA_MODE;

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }

    MPI_Barrier(MPI_COMM_WORLD);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_end_indep_data(int ncid)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_end_indep_data(ncid);
}

/**
 * Add a new dimension to an open netCDF dataset.
 */
extern "C"
int _nc_def_dim(
                int ncid,
                const char *name,
                size_t len,
                int *dimidp)
{
    int rc = NC_NOERR;
    int dimid;
    trios_declare_timer(MulticastTime);
    trios_declare_timer(WaitAnyTime);
    vector<nssi_request *>reqs;

    log_debug(netcdf_debug_level, "enter");

    log_debug(netcdf_debug_level, "Calling _nc_def_dim");

    assert(file_state_map[ncid]->mode == NC_DEFINE_MODE);

    nc_def_dim_args args;
    memset(&args, 0, sizeof(args));
    args.ncid = ncid;
    args.name = (char *)name;
    args.len = len;

    if (collective_op_rank == 0) {
        trios_start_timer(MulticastTime);
        /* call the remote method */
        rc = nssi_call_rpc_sync(&svcs[default_svc],
                NETCDF_DEF_DIM_OP,
                &args,
                NULL,
                0,
                &dimid);
        trios_stop_timer("create Multicast", MulticastTime);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }

    *dimidp=dimid;

    /* share the file descriptor with the rest of our group */
    MPI_Bcast(dimidp, 1, MPI_INT, 0, ncmpi_collective_op_comm);

    log_debug(netcdf_debug_level, "got result from _nc_def_dim: rc=%d, dimidp=%d",rc, *dimidp);

    if (global_rank == 0) {
        /* add the dimension to the open dataset */
        if (group_map.find(ncid) != group_map.end()) {
            group_map[ncid]->def_dim(*dimidp, name, len);
        }
        else {
            log_error(netcdf_debug_level, "dataset for ncid=%d not found", ncid);
            rc = NC_EBADID;
            goto cleanup;
        }
    }

cleanup:
    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_def_dim(
                int ncid,
                const char *name,
                size_t len,
                int *dimidp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_def_dim(ncid, name, len, dimidp);
}


/**
 * Define a variable for an open netCDF dataset.
 */
extern "C"
int _nc_def_var(
                int ncid,
                const char *name,
                nc_type xtype,
                int ndims,
                const int dimids[],
                int *varidp)
{
    int rc = NC_NOERR;
    int varid;
    trios_declare_timer(MulticastTime);
    trios_declare_timer(WaitAnyTime);

    vector<nssi_request *>reqs;

    log_debug(netcdf_debug_level, "enter");

    assert(file_state_map[ncid]->mode == NC_DEFINE_MODE);

    nc_def_var_args args;
    memset(&args, 0, sizeof(args));
    args.ncid = ncid;
    args.name = (char *)name;
    args.xtype = (int)xtype;
    args.dimids.dimids_len = ndims;
    args.dimids.dimids_val = (int *)&dimids[0];

    log_debug(netcdf_debug_level, "Calling _nc_def_var(ncid=%d, name=\"%s\", xtype=%d,"
            " ndims=%d, dimids[0]=%d", ncid, name, xtype, ndims, args.dimids.dimids_val[0]);

    if (collective_op_rank == 0) {
        trios_start_timer(MulticastTime);
        /* call the remote method */
        rc = nssi_call_rpc_sync(&svcs[default_svc],
                NETCDF_DEF_VAR_OP,
                &args,
                NULL,
                0,
                &varid);
        trios_stop_timer("create Multicast", MulticastTime);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }

    *varidp=varid;

    /* share the file descriptor with the rest of our group */
    MPI_Bcast(varidp, 1, MPI_INT, 0, ncmpi_collective_op_comm);

    log_debug(netcdf_debug_level, "got result from _nc_def_var: rc=%d, varidp=%d",rc, *varidp);

    if (global_rank == 0) {
        /* add the variable to the open dataset */
        if (group_map.find(ncid) != group_map.end()) {
            group_map[ncid]->def_var(*varidp, name, xtype, ndims, dimids);
        }
        else {
            log_error(netcdf_debug_level, "dataset for ncid=%d not found", ncid);
            rc = NC_EBADID;
            goto cleanup;
        }
    }

cleanup:
    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_def_var(
                int ncid,
                const char *name,
                nc_type xtype,
                int ndims,
                const int dimids[],
                int *varidp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_def_var(ncid, name, xtype, ndims, dimids, varidp);
}

/* ************** INQUIRY FUNCTIONS ***************** */

extern "C"
int _nc_inq(
        int ncid,
        int *ndimsp,
        int *nvarsp,
        int *nattsp,
        int *unlimdimidp)
{
    log_debug(netcdf_debug_level, "enter");

    if (group_map.find(ncid) == group_map.end()) {
        return NC_ENOTNC;
    }
    else {
        return group_map[ncid]->inq(ndimsp, nvarsp, nattsp, unlimdimidp);
    }
}
extern "C"
int nc_inq(
        int ncid,
        int *ndimsp,
        int *nvarsp,
        int *nattsp,
        int *unlimdimidp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq(ncid, ndimsp, nvarsp, nattsp, unlimdimidp);
}

extern "C"
int _nc_inq_attlen(
        int ncid,
        int varid,
        const char *name,
        size_t *lenp)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    if (group_map.find(ncid) == group_map.end()) {
        return NC_ENOTNC;
    }

    /* global attribute */
    rc = group_map[ncid]->inq_attlen(varid, name, lenp);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_attlen(
        int ncid,
        int varid,
        const char *name,
        size_t *lenp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_attlen(ncid, varid, name, lenp);
}

extern "C"
int _nc_inq_atttype(
        int ncid,
        int varid,
        const char *name,
        nc_type *xtypep)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    if (group_map.find(ncid) == group_map.end()) {
        return NC_ENOTNC;
    }

    rc = group_map[ncid]->inq_atttype(varid, name, xtypep);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_atttype(
        int ncid,
        int varid,
        const char *name,
        nc_type *xtypep)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_atttype(ncid, varid, name, xtypep);
}

extern "C"
int _nc_inq_att(
        int ncid,
        int varid,
        const char *name,
        nc_type *xtypep,
        size_t *lenp)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter (name=%s)", name);

    rc = _nc_inq_attlen(ncid, varid, name, lenp);
    if (rc != NC_NOERR) {
        log_error(netcdf_debug_level, "%s", nc_strerror(rc));
        return rc;
    }

    rc = _nc_inq_atttype(ncid, varid, name, xtypep);
    if (rc != NC_NOERR) {
        log_error(netcdf_debug_level, "%s", nc_strerror(rc));
        return rc;
    }

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_att(
        int ncid,
        int varid,
        const char *name,
        nc_type *xtypep,
        size_t *lenp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_att(ncid, varid, name, xtypep, lenp);
}

extern "C"
int _nc_inq_attname(
        int ncid,
        int varid,
        int attnum,
        char *name)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    if (group_map.find(ncid) == group_map.end()) {
        return NC_ENOTNC;
    }

    rc = group_map[ncid]->inq_attname(varid, attnum, name);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_attname(
        int ncid,
        int varid,
        int attnum,
        char *name)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_attname(ncid, varid, attnum, name);
}

extern "C"
int _nc_inq_dim(
        int ncid,
        int dimid,
        char *name,
        size_t *lenp)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    /* find dataset */
    if (group_map.find(ncid) != group_map.end()) {
        rc = group_map[ncid]->inq_dim(dimid, name, lenp);
    }
    else {
        rc = NC_ENOTNC;
    }

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_dim(
        int ncid,
        int dimid,
        char *name,
        size_t *lenp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_dim(ncid, dimid, name, lenp);
}

extern "C"
int _nc_inq_dimlen(
        int ncid,
        int dimid,
        size_t *lenp)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    /* find dataset */
    if (group_map.find(ncid) != group_map.end()) {
        rc = group_map[ncid]->inq_dimlen(dimid, lenp);
    }
    else {
        rc = NC_ENOTNC;
    }

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_dimlen(
        int ncid,
        int dimid,
        size_t *lenp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_dimlen(ncid, dimid, lenp);
}

extern "C"
int _nc_inq_dimid(
        int ncid,
        const char *name,
        int *dimidp)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");
    log_debug(netcdf_debug_level, "looking for dim named '%s'", name);

    /* find dataset */
    if (group_map.find(ncid) != group_map.end()) {
        rc = group_map[ncid]->inq_dimid(name, dimidp);
        log_debug(netcdf_debug_level, "dim named '%s' has id '%d'", name, *dimidp);
    }
    else {
        rc = NC_EBADID;
    }

    log_debug(netcdf_debug_level, "exit (rc=%d)", rc);

    return rc;
}
extern "C"
int nc_inq_dimid(
        int ncid,
        const char *name,
        int *dimidp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_dimid(ncid, name, dimidp);
}

extern "C"
int _nc_inq_unlimdim(
        int ncid,
        int *unlimdimidp)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    /* find dataset */
    if (group_map.find(ncid) != group_map.end()) {
        rc = group_map[ncid]->inq_unlimdimid(unlimdimidp);
    }
    else {
        rc = NC_ENOTNC;
    }

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_unlimdim(
        int ncid,
        int *unlimdimidp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_unlimdim(ncid, unlimdimidp);
}

extern "C"
int _nc_inq_var(
        int ncid,
        int varid,
        char *name,
        nc_type *xtypep,
        int *ndimsp,
        int *dimids,
        int *nattsp)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    log_debug(netcdf_debug_level, "Calling _nc_inq_var");

    /* find dataset */
    if (group_map.find(ncid) == group_map.end()) {
        rc = NC_ENOTNC;
        goto cleanup;
    }
    else {
        /* find variable in dataset */
        rc = group_map[ncid]->inq_var(varid, name, xtypep, ndimsp, dimids, nattsp);
    }

cleanup:
    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_var(
        int ncid,
        int varid,
        char *name,
        nc_type *xtypep,
        int *ndimsp,
        int *dimids,
        int *nattsp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_var(ncid, varid, name, xtypep, ndimsp, dimids, nattsp);
}

/**
 * Return the name and size of an nc_type.
 *
 * TODO: add support for user-defined types.
 */
extern "C"
int _nc_inq_type(
        int ncid,
        nc_type xtype,
        char *name,
        size_t *sizep)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    switch (xtype) {

    case NC_BYTE:
        if (name) strcpy(name, "NC_BYTE");
        *sizep = sizeof(char);
        break;

    case NC_CHAR:
        if (name) strcpy(name, "NC_CHAR");
        *sizep = sizeof(char);
        break;

    case NC_SHORT:
        if (name) strcpy(name, "NC_SHORT");
        *sizep = sizeof(short);
        break;

    case NC_INT:
        if (name) strcpy(name, "NC_INT");
        *sizep = sizeof(int);
        break;

    case NC_FLOAT:
        if (name) strcpy(name, "NC_FLOAT");
        *sizep = sizeof(float);
        break;

    case NC_DOUBLE:
        if (name) strcpy(name, "NC_DOUBLE");
        *sizep = sizeof(double);
        break;

    default:
        log_error(netcdf_debug_level, "Unsupported type=%d", (int)xtype);
        rc = NC_EBADTYPE;
        break;
    }

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_type(
        int ncid,
        nc_type xtype,
        char *name,
        size_t *sizep)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_type(ncid, xtype, name, sizep);
}

extern "C"
int _nc_inq_varid(
        int ncid,
        const char *name,
        int *varidp)
{
    int rc = NC_NOERR;
    log_level debug_level = netcdf_debug_level;

    log_debug(netcdf_debug_level, "enter");

    /* find dataset */
    if (group_map.find(ncid) != group_map.end()) {
        rc = group_map[ncid]->inq_varid(name, varidp);
    }
    else {
        rc = NC_ENOTNC;
    }

    log_debug(debug_level, "found varid=%d for variable \"%s\" in ncid=%d",
            *varidp, name, ncid);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_varid(
        int ncid,
        const char *name,
        int *varidp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_varid(ncid, name, varidp);
}

extern "C"
int _nc_inq_vardimid(
        int ncid,
        int varid,
        int dimids[])
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    /* find dataset */
    if (group_map.find(ncid) != group_map.end()) {
        rc = group_map[ncid]->inq_vardimid(varid, dimids);
    }
    else {
        rc = NC_ENOTNC;
    }

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_vardimid(
        int ncid,
        int varid,
        int dimids[])
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_vardimid(ncid, varid, dimids);
}

extern "C"
int _nc_inq_varndims(
        int ncid,
        int varid,
        int *ndimsp)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    /* find dataset */
    if (group_map.find(ncid) != group_map.end()) {
        rc = group_map[ncid]->inq_varndims(varid, ndimsp);
    }
    else {
        rc = NC_ENOTNC;
    }

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_varndims(
        int ncid,
        int varid,
        int *ndimsp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_varndims(ncid, varid, ndimsp);
}

extern "C"
int _nc_inq_vartype(
        int ncid,
        int varid,
        nc_type *xtypep)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    /* find dataset */
    if (group_map.find(ncid) != group_map.end()) {
        rc = group_map[ncid]->inq_vartype(varid, xtypep);
    }
    else {
        rc = NC_ENOTNC;
    }

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_inq_vartype(
        int ncid,
        int varid,
        nc_type *xtypep)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_inq_vartype(ncid, varid, xtypep);
}



/**
 * Package up an attribute and send a request to the server.
 *
 * WARN: This approach does not work if the client and server
 * use different byte ordering.
 */
extern "C"
int _nc_put_att_type(
        int ncid,
        int varid,
        const char *name,
        nc_type xtype,
        arg_type atype,
        size_t len,
        const void *obj_p)
{
    int rc = NC_NOERR;
    nc_put_att_args args;

    log_debug(netcdf_debug_level, "enter");

    log_debug(netcdf_debug_level, "Calling _nc_put_att");

    memset(&args, 0, sizeof(args));
    args.ncid = ncid;
    args.varid = varid;
    args.name = (char *)name;
    args.xtype = (int) xtype;
    args.atype = atype;
    args.data.data_len = len;
    args.data.data_val = (char *)obj_p;

    if (collective_op_rank == 0) {
        /* call the remote method */
        rc = nssi_call_rpc_sync(&svcs[default_svc],
                NETCDF_PUT_ATT_OP,
                &args,
                NULL,
                0,
                NULL);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }

    if (rc != NSSI_OK) {
        log_error(netcdf_debug_level, "unable to call remote _nc_put_att: %s",
                nssi_err_str(rc));
        goto cleanup;
    }

    if (global_rank == 0) {
        /* add the attribute to the local metadata */
        if (group_map.find(ncid) != group_map.end()) {
            NcGroupInfo *info = group_map[ncid];

            if (varid == NC_GLOBAL) {
                info->def_att(name, xtype, len);
            }
            else {
                if (info->_vars.find(varid) != info->_vars.end()) {
                    info->_vars[varid]->def_att(name, (nc_type)xtype, len);
                }
                else {
                    log_error(netcdf_debug_level, "unable to find varid=%d", varid);
                    rc = NC_ENOTVAR;
                    goto cleanup;
                }
            }
        }
        else {
            log_error(netcdf_debug_level, "unable to find ncid=%d", ncid);
            rc = NC_ENOTNC;
            goto cleanup;
        }
    }

cleanup:
    log_debug(netcdf_debug_level, "exit");

    return rc;
}


/**
 * Add or change a variable attribute of an open netCDF dataset.
 */
extern "C"
int nc_put_att (
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const void *obj_p)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_VOID, len, obj_p);
}

extern "C"
int nc_put_att_text(
                int ncid,
                int varid,
                const char *name,
                size_t len,
                const char *tp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, NC_CHAR, NC_ARG_TEXT, len*sizeof(char), tp);
}

extern "C"
int nc_put_att_uchar(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const unsigned char *up)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_UCHAR, len*sizeof(unsigned char), up);
}

extern "C"
int nc_put_att_schar(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const signed char *cp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_SCHAR, len*sizeof(signed char), cp);
}

extern "C"
int nc_put_att_short (
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const short *sp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_SHORT, len*sizeof(short), sp);
}

extern "C"
int nc_put_att_int(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const int *ip)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_INT, len*sizeof(int), ip);
}

extern "C"
int nc_put_att_long(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const long *lp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_LONG, len*sizeof(long), lp);
}

extern "C"
int nc_put_att_float(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const float *fp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_FLOAT, len*sizeof(float), fp);
}

extern "C"
int nc_put_att_double(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const double *dp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_DOUBLE, len*sizeof(double), dp);
}

extern "C"
int nc_put_att_ubyte(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const unsigned char *op)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_UBYTE, len*sizeof(unsigned char), op);
}

extern "C"
int nc_put_att_ushort(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const unsigned short *op)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_USHORT, len*sizeof(unsigned short), op);
}

extern "C"
int nc_put_att_uint(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const unsigned int *op)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_UINT, len*sizeof(unsigned int), op);
}

extern "C"
int nc_put_att_longlong(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const long long *op)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_LONGLONG, len*sizeof(long long), op);
}

extern "C"
int nc_put_att_ulonglong(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const unsigned long long *op)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_put_att_type(ncid, varid, name, xtype, NC_ARG_ULONGLONG, len*sizeof(unsigned long long), op);
}


/**
 * Package up an attribute and send a request to the server.
 *
 * WARN: This approach does not work if the client and server
 * use different byte ordering.
 */
extern "C"
int _nc_get_att(
        int ncid,
        int varid,
        const char *name,
        void *data)
{
    int rc = NC_NOERR;
    nc_get_att_args args;
    size_t attlen;
    nc_type xtype;
    size_t nbytes;
    size_t typesize;

    log_level debug_level = netcdf_debug_level;

    log_debug(netcdf_debug_level, "enter (name=%s)", name);

    log_debug(debug_level, "Calling _nc_get_att");

    memset(&args, 0, sizeof(args));
    args.ncid = ncid;
    args.varid = varid;
    args.name = const_cast<char *>(name);

    /* Need to get information about the attribute */
    rc = _nc_inq_att(ncid, varid, name, &xtype, &attlen);
    if (rc != NC_NOERR) {
        log_error(debug_level, "Could not inquire about attribute: %s", nc_strerror(rc));
        goto cleanup;
    }

    rc = _nc_inq_type(ncid, xtype, NULL, &typesize);
    if (rc != NC_NOERR) {
        log_error(debug_level, "Could not inquire about type: %s", nc_strerror(rc));
        goto cleanup;
    }

    nbytes = attlen * typesize;


    if (collective_op_rank == 0) {
        /* call the remote method */
        rc = nssi_call_rpc_sync(&svcs[default_svc],
                NETCDF_GET_ATT_OP,
                &args,
                data,
                nbytes,
                NULL);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }
    if (rc != NSSI_OK) {
        log_error(netcdf_debug_level, "unable to call remote _nc_get_att: %s",
                nssi_err_str(rc));
        goto cleanup;
    }

cleanup:
    log_debug(netcdf_debug_level, "exit (rc=%d)", rc);

    return rc;
}

extern "C"
int nc_get_att(
        int ncid,
        int varid,
        const char *name,
        void *data)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_get_att(ncid, varid, name, data);
}

extern "C"
int nc_get_att_text(
                int ncid,
                int varid,
        const char *name,
        char *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_CHAR */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_CHAR) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_uchar(
        int ncid,
        int varid,
        const char *name,
        unsigned char *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_UBYTE */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_UBYTE) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_schar(
        int ncid,
        int varid,
        const char *name,
        signed char *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_BYTE */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_BYTE) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_short(
        int ncid,
        int varid,
        const char *name,
        short *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_SHORT */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_SHORT) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_int(
        int ncid,
        int varid,
        const char *name,
        int *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_INT */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_INT) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_long(
        int ncid,
        int varid,
        const char *name,
        long *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_INT */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_INT) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_float(
        int ncid,
        int varid,
        const char *name,
        float *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_FLOAT */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_FLOAT) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_double(
        int ncid,
        int varid,
        const char *name,
        double *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_DOUBLE */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_DOUBLE) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_ubyte(
        int ncid,
        int varid,
        const char *name,
        unsigned char *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_UBYTE */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_UBYTE) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_ushort(
        int ncid,
        int varid,
        const char *name,
        unsigned short *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_USHORT */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_USHORT) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_uint(
        int ncid,
        int varid,
        const char *name,
        unsigned int *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_UINT */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_UINT) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_longlong(
        int ncid,
        int varid,
        const char *name,
        long long *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_INT64 */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_INT64) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_att_ulonglong(
        int ncid,
        int varid,
        const char *name,
        unsigned long long *data)
{
    int rc;
    nc_type xtype;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* make sure the type is NC_UINT64 */
    rc = _nc_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        return rc;
    }

    if (xtype != NC_UINT64) {
        return NC_EBADTYPE;
    }

    rc = _nc_get_att(ncid, varid, name, data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

/* ********************* OUTPUT FUNCTIONS **********************/

/**
 * The unlimited dimension starts out with length 0.  As records are written,
 * the length increases.  We update the length here.
 */
static int update_unlimdim_dimlen(const int ncid, const size_t dimlen)
{
    int rc = NC_NOERR;

    if (group_map.find(ncid) != group_map.end()) {
        group_map[ncid]->update_unlimdim_dimlen(dimlen);
    }
    else {
        log_error(netcdf_debug_level, "dataset for ncid=%d not found", ncid);
        rc = NC_EBADID;
    }

    return(rc);
}


/**
 * Write a subsampled array of values.
 *
 * This function is sufficient for all other netcdf write functions.
 */
extern "C"
int _nc_put_vars(
                int    ncid,
                int    varid,
                nc_type buftype,
                size_t  buftypesize, /* the size of 'data' based type */
                const size_t *startp,
                const size_t *countp,
                const ptrdiff_t *stridep,
                const void *data)
{
    int rc, i;
    log_level debug_level = netcdf_debug_level;

    /* need copy in case word sizes don't match */
    nc_size_t *start_copy = NULL;
    nc_size_t *count_copy = NULL;
    nc_size_t *stride_copy = NULL;
    int *dimids = NULL;
    size_t *dimlens = NULL;

    int unlimdimid=-1;


    NcGroupInfo *group = group_map[ncid];
    size_t nbytes = 0;

    nc_put_vars_args args;
    int ndims;

    trios_declare_timer(callTime);
    trios_start_timer(callTime);

    log_debug(netcdf_debug_level, "enter");

    /* calculate the size of the buffer. What happens on overruns? Segfault? */
    if (group_map.find(ncid) == group_map.end()) {
        log_error(debug_level, "Could not find group for ncid=%d", ncid);
        rc = NC_EINVAL;
        goto cleanup;
    }

    rc = group->inq_varndims(varid, &ndims);
    if (rc != NC_NOERR) {
        log_error(debug_level, "unable to get ndims");
        goto cleanup;
    }


    dimids = (int *) calloc(ndims, sizeof(int));
    dimlens = (size_t *) calloc(ndims, sizeof(size_t));

    /* get the diminsion IDs for this variable */
    rc = _nc_inq_vardimid(ncid, varid, dimids);
    if (rc != NC_NOERR) {
        log_error(debug_level, "could not get dimids");
        goto cleanup;
    }

    rc = _nc_inq_unlimdim(ncid, &unlimdimid); /* get ID of unlimited dimension */
    if (rc != NC_NOERR) {
        log_error(debug_level, "could not get unlimdimid");
        goto cleanup;
    }
    for (i = 0; i<ndims; i++) {
        rc = _nc_inq_dimlen(ncid, dimids[i], &dimlens[i]);
        if (rc != NC_NOERR) {
            log_error(debug_level, "could not get dimlen");
            goto cleanup;
        }
        if (dimlens[i] == 0) {
            // a dim length of 0 should mean this is the unlimited dim and no
            // records have been written yet.  if it is the unlimited dim, then
            // make the dim length 1.
            if ((unlimdimid != -1) && (unlimdimid == dimids[i])) {
                dimlens[i] = 1;
            }
        }
    }

    /* calculate the number of bytes in the user array */
    if (ndims) {
        size_t count = 1;

        /* copy values */
        if (startp) {
            start_copy = (nc_size_t *)calloc(ndims, sizeof(nc_size_t));
            copy(startp, startp+ndims, start_copy);
        }

        /* set counts */
        if (countp) {
            count_copy = (nc_size_t *)calloc(ndims, sizeof(nc_size_t));
            copy(countp, countp+ndims, count_copy);
        }
        else {
            /* defaults to dimlen[0], dimlen[1], ... dimlen[ndims-1] */
            count_copy = (nc_size_t *)calloc(ndims, sizeof(nc_size_t));

            /* set count[i] to dimlen[i] */
            for (i = 0; i<ndims; i++) {
                count_copy[i] = dimlens[i];
            }
        }

        if (stridep) {
            stride_copy = (nc_size_t *)calloc(ndims, sizeof(nc_size_t));
            copy(stridep, stridep+ndims, stride_copy);
        }

        /* calculate the size (number of elements) of the data array */
        for (i=0; i<ndims; i++) {
            count = count * count_copy[i];
        }
        nbytes = count*buftypesize;

        log_debug(debug_level, "Transferring %lu vals (%lu bytes) to server", count, nbytes);

        if (nc_cfg.use_subchunking) {
//            log_debug(LOG_ALL, "subchunking");
            superchunk_t chunk;
            chunk.ncid=ncid;
            chunk.varid=varid;
            chunk.ndims=ndims;
            chunk.buf=const_cast<void *>(data);
            chunk.len=nbytes;
            chunk.buftype=buftype;
            chunk.datatype_size=buftypesize;
            chunk.start=start_copy;
            chunk.count=count_copy;
            chunk.stride=stride_copy;

            trios_declare_timer(subchunkingTime);
            trios_start_timer(subchunkingTime);
            consolidated_subchunks_map_t *map=netcdf_create_subchunks(&chunk, dimids, dimlens, nc_cfg.bytes_per_server);
            trios_stop_timer("subchunking", subchunkingTime);

            trios_declare_timer(totalPutTime);
            trios_start_timer(totalPutTime);
            consolidated_subchunks_map_iterator_t map_iter = map->begin();
            for (;map_iter != map->end(); map_iter++) {
                consolidated_subchunks_vector_iterator_t vector_iter = (*map_iter).second->begin();
                char outstr[1024];
                for (;vector_iter != (*map_iter).second->end(); vector_iter++) {
                    sprintf(outstr, "service(%02ld) super_start[", (*map_iter).first);
                    char tmpstr[256];
                    for (i=0;i<ndims;i++) {
                        sprintf(tmpstr, "%04ld,", chunk.start[i]);
                        strcat(outstr, tmpstr);
                    }
                    strcat(outstr, "] start[");
                    for (i=0;i<ndims;i++) {
                        sprintf(tmpstr, "%04ld,", (*vector_iter)->start[i]);
                        strcat(outstr, tmpstr);
                    }
                    strcat(outstr, "] count[");
                    for (i=0;i<ndims;i++) {
                        sprintf(tmpstr, "%04ld,", (*vector_iter)->count[i]);
                        strcat(outstr, tmpstr);
                    }
                    sprintf(tmpstr, "] offset_into_superchunk(%ld)", (*vector_iter)->offset_into_superchunk);
                    strcat(outstr, tmpstr);
                    log_debug(debug_level, "%s", outstr);

//                    log_debug(debug_level,
//                            "service(%02ld) super_start[%04ld,%04ld,%04ld,%04ld] start[%04ld,%04ld,%04ld,%04ld] count[%04ld,%04ld,%04ld,%04ld] offset_into_superchunk(%ld)",
//                            (*map_iter).first,
//                            chunk.start[0], chunk.start[1], chunk.start[2], chunk.start[3],
//                            (*vector_iter)->start[0], (*vector_iter)->start[1], (*vector_iter)->start[2], (*vector_iter)->start[3],
//                            (*vector_iter)->count[0], (*vector_iter)->count[1], (*vector_iter)->count[2], (*vector_iter)->count[3],
//                            (*vector_iter)->offset_into_superchunk);


                    /* initialize remote arguments */
                    memset(&args, 0, sizeof(args));
                    args.ncid = ncid;
                    args.varid = varid;
                    args.atype = NC_ARG_VOID;    /* unused */
                    args.start.start_len = (start_copy != NULL)? ndims: 0;
                    args.start.start_val = (*vector_iter)->start;
                    args.count.count_len = (count_copy != NULL)? ndims: 0;
                    args.count.count_val = (*vector_iter)->count;
                    args.stride.stride_len = (stride_copy != NULL)? ndims: 0;
                    args.stride.stride_val = stride_copy;
                    args.buftype       = buftype;
                    args.element_count = count;
                    args.len           = (*vector_iter)->nbytes;

                    log_debug(debug_level, "sending start_len=%d, count_len=%d, stride_len=%d",
                            args.start.start_len, args.count.count_len, args.stride.stride_len);

                    int svc_index=(*map_iter).first;
                    log_debug(debug_level, "svc_index(%d)", svc_index);

                    trios_declare_timer(putTime);
                    trios_start_timer(putTime);
                    if (((nc_cfg.write_type != WRITE_AGGREGATE_COLLECTIVE) &&
                         (nc_cfg.write_type != WRITE_CACHING_COLLECTIVE)) &&
                            (file_state_map[ncid]->mode != NC_INDEP_DATA_MODE)) {
                        _nc_begin_indep_data(ncid);
                        /* Send request to remote service */
                        rc = nssi_call_rpc_sync(&svcs[svc_index], NETCDF_PUT_VARS_OP, &args,
                                ((char *)data)+(*vector_iter)->offset_into_superchunk, (*vector_iter)->nbytes, NULL);
                        if (rc != NSSI_OK) {
                            log_error(debug_level, "unable to call remote _nc_put_vars");
                            goto cleanup;
                        }
                        _nc_end_indep_data(ncid);
                    } else {
                        /* Send request to remote service */
                        rc = nssi_call_rpc_sync(&svcs[svc_index], NETCDF_PUT_VARS_OP, &args,
                                ((char *)data)+(*vector_iter)->offset_into_superchunk, (*vector_iter)->nbytes, NULL);
                        if (rc != NSSI_OK) {
                            log_error(debug_level, "unable to call remote _nc_put_vars");
                            goto cleanup;
                        }
                    }
                    trios_stop_timer("subchunk put", putTime);
                }
            }
            trios_stop_timer("total subchunk put", totalPutTime);
        } else {
//            log_debug(LOG_ALL, "NOT subchunking");
            /* initialize remote arguments */
            memset(&args, 0, sizeof(args));
            args.ncid = ncid;
            args.varid = varid;
            args.atype = NC_ARG_VOID;    /* unused */
            args.start.start_len = (start_copy != NULL)? ndims: 0;
            args.start.start_val = start_copy;
            args.count.count_len = (count_copy != NULL)? ndims: 0;
            args.count.count_val = count_copy;
            args.stride.stride_len = (stride_copy != NULL)? ndims: 0;
            args.stride.stride_val = stride_copy;
            args.buftype       = buftype;
            args.element_count = count;
            args.len           = nbytes;

            log_debug(debug_level, "sending start_len=%d, count_len=%d, stride_len=%d",
                    args.start.start_len, args.count.count_len, args.stride.stride_len);

            int svc_index=default_svc;
            log_debug(debug_level, "svc_index(%d)", svc_index);

//            log_debug(LOG_ALL,
//                    "service(%02ld) ncid(%d) varid(%d) ndims(%d) start[%04ld,%04ld,%04ld,%04ld] count[%04ld,%04ld,%04ld,%04ld]",
//                    svc_index, ncid, varid, ndims,
//                    start_copy[0], start_copy[1], start_copy[2], start_copy[3],
//                    count_copy[0], count_copy[1], count_copy[2], count_copy[3]);


            trios_declare_timer(putTime);
            trios_start_timer(putTime);
            if (((nc_cfg.write_type != WRITE_AGGREGATE_COLLECTIVE) &&
                 (nc_cfg.write_type != WRITE_CACHING_COLLECTIVE)) &&
                    (file_state_map[ncid]->mode != NC_INDEP_DATA_MODE)) {
                _nc_begin_indep_data(ncid);
                /* Send request to remote service */
                rc = nssi_call_rpc_sync(&svcs[svc_index], NETCDF_PUT_VARS_OP, &args, (char *)data, nbytes, NULL);
                if (rc != NSSI_OK) {
                    log_error(debug_level, "unable to call remote _nc_put_vars");
                    goto cleanup;
                }
                _nc_end_indep_data(ncid);
            } else {
                /* Send request to remote service */
                rc = nssi_call_rpc_sync(&svcs[svc_index], NETCDF_PUT_VARS_OP, &args, (char *)data, nbytes, NULL);
                if (rc != NSSI_OK) {
                    log_error(debug_level, "unable to call remote _nc_put_vars");
                    goto cleanup;
                }
            }
            trios_stop_timer("put", putTime);
        }

        if (unlimdimid != -1) {
            for (i=0;i<ndims;i++) {
                if ((unlimdimid == dimids[i]) && start_copy) {
                    rc = update_unlimdim_dimlen(ncid, start_copy[i]+1);
                } else {
                    rc = update_unlimdim_dimlen(ncid, 1);
                }
            }
        }
    }

cleanup:
    if (start_copy) free(start_copy);
    if (count_copy) free(count_copy);
    if (stride_copy) free(stride_copy);
    if (dimids) free(dimids);

    trios_stop_timer("_nc_put_vars", callTime);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_put_vars(
                int ncid, int varid,
                const size_t *startp,
                const size_t *countp,
                const ptrdiff_t *stridep,
                const void *data)
{
    int rc=0;
    nc_type vartype;
    size_t  varsize;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* Get the datatype information so we know the size of the buffer */
    /* This assumes that the 'data' has the same type as the variable varid */
    rc = _nc_inq_vartype(ncid, varid, &vartype);
    if (rc != NC_NOERR) {
        log_error(netcdf_debug_level, "Unable to get variable type");
        return rc;
    }
    rc = _nc_inq_type(ncid, vartype, NULL, &varsize);
    if (rc != NC_NOERR) {
        log_error(netcdf_debug_level, "Unable to get variable type");
        return rc;
    }

    return _nc_put_vars(ncid, varid, vartype, varsize, startp, countp, stridep, data);
}

/**
 * Write a subsampled array of float values.
 */
extern "C"
int nc_put_vars_float(
                int ncid, int varid,
                const size_t *startp,
                const size_t *countp,
                const ptrdiff_t *stridep,
                const float *data)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling nc_put_vars_float");
    return _nc_put_vars(ncid, varid, NC_FLOAT, sizeof(float), startp, countp, stridep, (void*)data);
}

/**
 * Write a subsampled array of float values.
 */
extern "C"
int nc_put_vars_double(
                int ncid, int varid,
                const size_t *startp,
                const size_t *countp,
                const ptrdiff_t *stridep,
                const double *data)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling nc_put_vars_double");
    return _nc_put_vars(ncid, varid, NC_DOUBLE, sizeof(double), startp, countp, stridep, (void*)data);
}

/**
 * Write the entire contents of a text variable.
 */
extern "C"
int nc_put_var_text(int ncid, int varid, const char *tp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling nc_put_var_text");
    return _nc_put_vars(ncid, varid, NC_CHAR, sizeof(char), NULL, NULL, NULL, tp);
}


/**
 * Write the entire contents of an int variable.
 */
extern "C"
int nc_put_var_int(int ncid, int varid, const int *ip)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling nc_put_var_int");
    return _nc_put_vars(ncid, varid, NC_INT, sizeof(int), NULL, NULL, NULL, ip);
}

/**
 * Write the entire contents of a float variable.
 */
extern "C"
int nc_put_var_float(int ncid, int varid, const float *fp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling nc_put_var_float (%f, %f,...)", fp[0], fp[1]);
    return _nc_put_vars(ncid, varid, NC_FLOAT, sizeof(float), NULL, NULL, NULL, fp);
}

/**
 * Write the entire contents of a double variable.
 */
extern "C"
int nc_put_var_double(int ncid, int varid, const double *fp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling nc_put_var_double (%f, %f,...)", fp[0], fp[1]);
    return _nc_put_vars(ncid, varid, NC_DOUBLE, sizeof(double), NULL, NULL, NULL, fp);
}

/**
 * Write an array of values.
 *
 */

extern "C"
int nc_put_vara(
                int ncid,
                int varid,
                const size_t *startp,
                const size_t *countp,
                const void *data)
{
    int rc = NC_NOERR;
    nc_type vartype;
    size_t  varsize;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* Get the datatype information so we know the size of the buffer */
    /* This assumes that the 'data' has the same type as the variable varid */
    rc = _nc_inq_vartype(ncid, varid, &vartype);
    if (rc != NC_NOERR) {
        log_error(netcdf_debug_level, "Unable to get variable type");
        return rc;
    }
    rc = _nc_inq_type(ncid, vartype, NULL, &varsize);
    if (rc != NC_NOERR) {
        log_error(netcdf_debug_level, "Unable to get variable type");
        return rc;
    }

    log_debug(netcdf_debug_level, "Calling nc_put_vara");
    rc = _nc_put_vars(ncid, varid, vartype, varsize, startp, countp, NULL, data);

    return rc;
}

extern "C"
int nc_put_vara_text(
        int ncid,
        int varid,
        const size_t *startp,
        const size_t *countp,
        const char *data)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling nc_put_vara_text");
    rc = _nc_put_vars(ncid, varid, NC_CHAR, sizeof(char), startp, countp, NULL, data);

    return rc;
}

extern "C"
int nc_put_vara_int(
        int ncid,
        int varid,
        const size_t *startp,
        const size_t *countp,
        const int *data)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling nc_put_vara_int");
    rc = _nc_put_vars(ncid, varid, NC_INT, sizeof(int), startp, countp, NULL, data);

    return rc;
}

extern "C"
int nc_put_vara_longlong(
        int ncid,
        int varid,
        const size_t *startp,
        const size_t *countp,
        const long long *data)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling nc_put_vara_longlong");
    rc = _nc_put_vars(ncid, varid, NC_INT64, sizeof(long long), startp, countp, NULL, data);

    return rc;
}

extern "C"
int nc_put_vara_float(
        int ncid,
        int varid,
        const size_t *startp,
        const size_t *countp,
        const float *data)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling nc_put_vara_float");
    rc = _nc_put_vars(ncid, varid, NC_FLOAT, sizeof(float), startp, countp, NULL, data);

    return rc;
}

extern "C"
int nc_put_vara_double(
        int ncid,
        int varid,
        const size_t *startp,
        const size_t *countp,
        const double *data)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling nc_put_vara_double");
    rc = _nc_put_vars(ncid, varid, NC_DOUBLE, sizeof(double), startp, countp, NULL, data);

    return rc;
}


/* ********************* INPUT FUNCTIONS **********************/

/**
 * Write a subsampled array of values.
 *
 * This function is sufficient for all other netcdf write functions.
 */
extern "C"
int _nc_get_vars(
        int ncid, int varid,
        const size_t *startp,
        const size_t *countp,
        const ptrdiff_t *stridep,
        void *data)
{
    int rc, i;
    log_level debug_level = netcdf_debug_level;

    /* need copy in case word sizes don't match */
    nc_size_t *start_copy = NULL;
    nc_size_t *count_copy = NULL;
    nc_size_t *stride_copy = NULL;
    int *dimids = NULL;
    size_t *dimlens = NULL;

    nc_type vartype;
    size_t varsize;

    NcGroupInfo *group = group_map[ncid];
    size_t nbytes = 0;

    nc_get_vars_args args;
    int ndims;

    log_debug(netcdf_debug_level, "enter");


    log_debug(debug_level, "calling _nc_get_vars");


    /* calculate the size of the buffer. What happens on overruns? Segfault? */
    if (group_map.find(ncid) == group_map.end()) {
        log_error(debug_level, "Could not find group for ncid=%d", ncid);
        rc = NC_EINVAL;
        goto cleanup;
    }

    rc = group->inq_varndims(varid, &ndims);
    if (rc != NC_NOERR) {
        log_error(debug_level, "unable to get ndims");
        goto cleanup;
    }


    dimids = (int *) calloc(ndims, sizeof(int));
    dimlens = (size_t *) calloc(ndims, sizeof(size_t));

    /* get the diminsion IDs for this variable */
    rc = _nc_inq_vardimid(ncid, varid, dimids);
    if (rc != NC_NOERR) {
        log_error(debug_level, "could not get dimids");
        goto cleanup;
    }

    /* set count[i] to dimlen[i] */
    for (i = 0; i<ndims; i++) {
        rc = _nc_inq_dimlen(ncid, dimids[i], &dimlens[i]);
        if (rc != NC_NOERR) {
            log_error(debug_level, "could not get dimlen");
            goto cleanup;
        }
    }


    /* calculate the number of bytes in the user array */
    if (ndims) {
        size_t count = 1;

        /* copy values */
        if (startp) {
            start_copy = (nc_size_t *)calloc(ndims, sizeof(nc_size_t));
            copy(startp, startp+ndims, start_copy);
        }

        /* set counts */
        if (countp) {
            count_copy = (nc_size_t *)calloc(ndims, sizeof(nc_size_t));
            copy(countp, countp+ndims, count_copy);
        }
        else {
            /* defaults to dimlen[0], dimlen[1], ... dimlen[ndims-1] */
            count_copy = (nc_size_t *)calloc(ndims, sizeof(nc_size_t));

            /* set count[i] to dimlen[i] */
            for (i = 0; i<ndims; i++) {
                count_copy[i] = dimlens[i];
            }
        }

        if (stridep) {
            stride_copy = (nc_size_t *)calloc(ndims, sizeof(nc_size_t));
            copy(stridep, stridep+ndims, stride_copy);
        }

        /* Get the datatype information so we know the size of the buffer */
        rc = _nc_inq_vartype(ncid, varid, &vartype);
        if (rc != NC_NOERR) {
            log_error(netcdf_debug_level, "Unable to get variable type");
            goto cleanup;
        }

        rc = _nc_inq_type(ncid, vartype, NULL, &varsize);
        if (rc != NC_NOERR) {
            log_error(netcdf_debug_level, "Unable to get variable type");
            goto cleanup;
        }

        /* calculate the size (number of elements) of the data array */
        for (i=0; i<ndims; i++) {
            count = count * count_copy[i];
        }
        nbytes = count*varsize;

        log_debug(debug_level, "Transferring %lu vals (%lu bytes) to client", count, nbytes);

        /* initialize remote arguments */
        memset(&args, 0, sizeof(args));
        args.ncid = ncid;
        args.varid = varid;
        args.atype = NC_ARG_VOID;    /* unused */
        args.start.start_len = (start_copy != NULL)? ndims: 0;
        args.start.start_val = start_copy;
        args.count.count_len = (count_copy != NULL)? ndims: 0;
        args.count.count_val = count_copy;
        args.stride.stride_len = (stride_copy != NULL)? ndims: 0;
        args.stride.stride_val = stride_copy;
        args.len = nbytes;

        log_debug(debug_level, "getting start_len=%d, count_len=%d, stride_len=%d",
                args.start.start_len, args.count.count_len, args.stride.stride_len);

        int svc_index=default_svc;
        log_debug(netcdf_debug_level, "svc_index(%d)", svc_index);

        if (file_state_map[ncid]->mode != NC_INDEP_DATA_MODE) {
            _nc_begin_indep_data(ncid);
            /* Send request to remote service */
            rc = nssi_call_rpc_sync(&svcs[svc_index], NETCDF_GET_VARS_OP, &args, (char *)data, nbytes, NULL);
            if (rc != NSSI_OK) {
                log_error(netcdf_debug_level, "unable to call remote _nc_get_vars");
                goto cleanup;
            }
            _nc_end_indep_data(ncid);
        } else {
            /* Send request to remote service */
            rc = nssi_call_rpc_sync(&svcs[svc_index], NETCDF_GET_VARS_OP, &args, (char *)data, nbytes, NULL);
            if (rc != NSSI_OK) {
                log_error(netcdf_debug_level, "unable to call remote _nc_get_vars");
                goto cleanup;
            }
        }

        log_debug(debug_level, "first two vals = %d, %d, ...",
                        ((int *)data)[0], ((int *)data)[1]);
    }

cleanup:
    if (start_copy) free(start_copy);
    if (count_copy) free(count_copy);
    if (stride_copy) free(stride_copy);
    if (dimids) free(dimids);

    log_debug(debug_level, "Exiting get_vars: rc=%d", rc);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_vars(
        int ncid, int varid,
        const size_t *startp,
        const size_t *countp,
        const ptrdiff_t *stridep,
        void *data)
{
    log_debug(netcdf_debug_level, "enter");

    return _nc_get_vars(ncid, varid, startp, countp, stridep, data);
}

/**
 * Write a subsampled array of float values.
 */
extern "C"
int nc_get_vars_float(
        int ncid, int varid,
        const size_t *startp,
        const size_t *countp,
        const ptrdiff_t *stridep,
        float *data)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_vars_float");
    rc = _nc_get_vars(ncid, varid, startp, countp, stridep, (char *)data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

/**
 * Write a subsampled array of double values.
 */
extern "C"
int nc_get_vars_double(
        int ncid, int varid,
        const size_t *startp,
        const size_t *countp,
        const ptrdiff_t *stridep,
        double *data)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_vars_double");
    rc = _nc_get_vars(ncid, varid, startp, countp, stridep, (char *)data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

extern "C"
int nc_get_var(
        int ncid,
        int varid,
        void *data)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_var");
    rc = _nc_get_vars(ncid, varid, NULL, NULL, NULL, (char *)data);

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

/**
 * Get the entire contents of a text variable.
 */
extern "C"
int nc_get_var_text(int ncid, int varid, char *tp)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_var_text");
    return _nc_get_vars(ncid, varid, NULL, NULL, NULL, tp);
}

/**
 * Get the entire contents of an int variable.
 */
extern "C"
int nc_get_var_int(int ncid, int varid, int *ip)
{
    int rc=0;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_var_int(ncid=%d)", ncid);
    rc = _nc_get_vars(ncid, varid, NULL, NULL, NULL, ip);

    log_debug(netcdf_debug_level, "ip[0]=%d", ip[0]);

    return rc;
}

/**
 * Get the entire contents of a float variable.
 */
extern "C"
int nc_get_var_float(int ncid, int varid, float *fp)
{
    int rc=0;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_var_float(ncid=%d)", ncid);
    rc = _nc_get_vars(ncid, varid, NULL, NULL, NULL, fp);

    log_debug(netcdf_debug_level, "fp[0]=%f", fp[0]);

    return rc;
}

/**
 * Get the entire contents of a double variable.
 */
extern "C"
int nc_get_var_double(int ncid, int varid, double *dp)
{
    int rc=0;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_var_double(ncid=%d)", ncid);
    rc = _nc_get_vars(ncid, varid, NULL, NULL, NULL, dp);

    log_debug(netcdf_debug_level, "dp[0]=%f", dp[0]);

    return rc;
}

/**
 * Read an array of values.
 */
extern "C"
int nc_get_vara(
        int ncid,
        int varid,
        const size_t startp[],
        const size_t countp[],
        void *data)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_vara(ncid=%d)", ncid);
    return _nc_get_vars(ncid, varid, startp, countp, NULL, (char *)data);
}

/**
 * Read an array of values.
 */
extern "C"
int nc_get_vara_text(
        int ncid,
        int varid,
        const size_t startp[],
        const size_t countp[],
        char *data)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_vara_text(ncid=%d)", ncid);
    return _nc_get_vars(ncid, varid, startp, countp, NULL, (char *)data);
}

/**
 * Read an array of values.
 */
extern "C"
int nc_get_vara_int(
        int ncid,
        int varid,
        const size_t startp[],
        const size_t countp[],
        int *data)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_vara_int(ncid=%d)", ncid);
    return _nc_get_vars(ncid, varid, startp, countp, NULL, (char *)data);
}

/**
 * Read an array of values.
 */
extern "C"
int nc_get_vara_long(
        int ncid,
        int varid,
        const size_t startp[],
        const size_t countp[],
        long *data)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_vara_long(ncid=%d)", ncid);
    return _nc_get_vars(ncid, varid, startp, countp, NULL, (char *)data);
}

/**
 * Read an array of values.
 */
extern "C"
int nc_get_vara_longlong(
        int ncid,
        int varid,
        const size_t startp[],
        const size_t countp[],
        long long *data)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_vara_longlong(ncid=%d)", ncid);
    return _nc_get_vars(ncid, varid, startp, countp, NULL, (char *)data);
}

/**
 * Read an array of values.
 */
extern "C"
int nc_get_vara_float(
        int ncid,
        int varid,
        const size_t startp[],
        const size_t countp[],
        float *data)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_vara_float(ncid=%d)", ncid);
    return _nc_get_vars(ncid, varid, startp, countp, NULL, (char *)data);
}

/**
 * Read an array of values.
 */
extern "C"
int nc_get_vara_double(
        int ncid,
        int varid,
        const size_t startp[],
        const size_t countp[],
        double *data)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    log_debug(netcdf_debug_level, "Calling _nc_get_vara_double(ncid=%d)", ncid);
    return _nc_get_vars(ncid, varid, startp, countp, NULL, (char *)data);
}


extern "C"
int nc_put_var1(
        int           ncid,
        int           varid,
        const size_t *coord,
        const void*   value,
        nc_type       memtype)
{
    int rc = NC_NOERR;
    nc_type vartype;
    size_t  varsize;

    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    /* Get the datatype information so we know the size of the buffer */
    /* This assumes that the 'data' has the same type as the variable varid */
    rc = _nc_inq_vartype(ncid, varid, &vartype);
    if (rc != NC_NOERR) {
        log_error(netcdf_debug_level, "Unable to get variable type");
        return rc;
    }
    rc = _nc_inq_type(ncid, vartype, NULL, &varsize);
    if (rc != NC_NOERR) {
        log_error(netcdf_debug_level, "Unable to get variable type");
        return rc;
    }

   return _nc_put_vars(ncid, varid, vartype, varsize, coord, NC_coord_one, NULL, value);
}

extern "C"
int nc_put_var1_text(
        int           ncid,
        int           varid,
        const size_t *coord,
        const char   *value)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

   return _nc_put_vars(ncid, varid, NC_CHAR, sizeof(char), coord, NC_coord_one, NULL, value);
}

extern "C"
int nc_put_var1_int(
        int           ncid,
        int           varid,
        const size_t *coord,
        const int    *value)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

   return _nc_put_vars(ncid, varid, NC_INT, sizeof(int), coord, NC_coord_one, NULL, value);
}

extern "C"
int nc_put_var1_float(
        int           ncid,
        int           varid,
        const size_t *coord,
        const float  *value)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

   return _nc_put_vars(ncid, varid, NC_FLOAT, sizeof(float), coord, NC_coord_one, NULL, value);
}

extern "C"
int nc_put_var1_double(
        int           ncid,
        int           varid,
        const size_t *coord,
        const double *value)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

   return _nc_put_vars(ncid, varid, NC_DOUBLE, sizeof(double), coord, NC_coord_one, NULL, value);
}


extern "C"
int nc_get_var1(
        int           ncid,
        int           varid,
        const size_t *coord,
        void*         value,
        nc_type       memtype)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

   return _nc_get_vars(ncid, varid, coord, NC_coord_one, NULL, value);
}

extern "C"
int nc_get_var1_int(
        int           ncid,
        int           varid,
        const size_t *coord,
        int          *value)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

   return _nc_get_vars(ncid, varid, coord, NC_coord_one, NULL, value);
}


extern "C"
int _nc_redef(int ncid)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    log_debug(netcdf_debug_level, "Calling _nc_redef");

    if (collective_op_rank == 0) {
        /* call the remote method */
        rc = nssi_call_rpc_sync(&svcs[default_svc],
                NETCDF_REDEF_OP,
                &ncid,
                NULL,
                0,
                NULL);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }
    if (rc != NSSI_OK) {
        log_error(netcdf_debug_level, "unable to call remote _nc_redef: %s",
                nssi_err_str(rc));
        goto cleanup;
    }
    file_state_map[ncid]->mode = NC_DEFINE_MODE;

cleanup:
    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_redef(int ncid)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_redef(ncid);
}

extern "C"
int _nc_enddef(int ncid)
{
    int rc = NC_NOERR;
    trios_declare_timer(MulticastTime);
    trios_declare_timer(WaitAnyTime);
    vector<nssi_request *>reqs;

    log_debug(netcdf_debug_level, "enter");

    if (collective_op_rank == 0) {
        trios_start_timer(MulticastTime);
        /* call the remote method */
        rc = nssi_call_rpc_sync(&svcs[default_svc],
                NETCDF_ENDDEF_OP,
                &ncid,
                NULL,
                0,
                NULL);

        trios_stop_timer("enddef Multicast", MulticastTime);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }

    file_state_map[ncid]->mode = NC_COLL_DATA_MODE;

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_enddef(int ncid)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_enddef(ncid);
}


/**
 * Sync an open netCDF dataset.
 */
extern "C"
int _nc_sync_wait(int ncid)
{
    int rc=NSSI_OK;
    nssi_request *req=NULL;
    request_list_iterator_t iter;

    log_debug(netcdf_debug_level, "enter");

    iter = file_state_map[ncid]->outstanding_sync_requests.begin();
    for (;iter != file_state_map[ncid]->outstanding_sync_requests.end();iter++) {
        req = *iter;
        if (req != NULL) {
            nssi_wait(req, &rc);
            free(req);
        }
    }
    file_state_map[ncid]->outstanding_sync_requests.clear();

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_sync_wait(int ncid)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_sync_wait(ncid);
}

/**
 * Sync an open netCDF dataset.
 */
extern "C"
int _nc_sync(int ncid)
{
    int rc=NSSI_OK;
    nssi_request *req=NULL;

    log_debug(netcdf_debug_level, "enter");

    if (collective_op_rank == 0) {
        req=(nssi_request *)calloc(1, sizeof(nssi_request));
        /* call the remote method */
        rc = nssi_call_rpc(&svcs[default_svc],
                NETCDF_SYNC_OP,
                &ncid,
                NULL,
                0,
                NULL,
                req);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }
    if (rc != NSSI_OK) {
        log_error(netcdf_debug_level, "unable to call remote nc_sync: %s",
                nssi_err_str(rc));
        goto cleanup;
    }

    file_state_map[ncid]->outstanding_sync_requests.push_back(req);

#if !defined(USE_ASYNC_SYNC)
    rc=_nc_sync_wait(ncid);
#endif

cleanup:
    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_sync(int ncid)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_sync(ncid);
}

/**
 * Close an open netCDF dataset.
 */
extern "C"
int _nc_close_wait(int ncid)
{
    int rc=NSSI_OK;
    nssi_request *req=NULL;
    request_list_iterator_t iter;

    log_debug(netcdf_debug_level, "enter");

   iter = file_state_map[ncid]->outstanding_close_requests.begin();
    for (;iter != file_state_map[ncid]->outstanding_close_requests.end();iter++) {
        req = *iter;
        if (req != NULL) {
            nssi_wait(req, &rc);
            free(req);
        }
    }
    file_state_map[ncid]->outstanding_close_requests.clear();

    /* remove data structures associated with ncid */
    if (group_map.find(ncid) != group_map.end()) {
        delete group_map[ncid];
        group_map.erase(ncid);
    }

    file_state_map[ncid]->mode = NC_CLOSED;

    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_close_wait(int ncid)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_close_wait(ncid);
}

/**
 * Close an open netCDF dataset.
 */
extern "C"
int _nc_close(int ncid)
{
    int rc=NSSI_OK;
    nssi_request *req=NULL;

    log_debug(netcdf_debug_level, "enter");

    if (collective_op_rank == 0) {
        req=(nssi_request *)calloc(1, sizeof(nssi_request));
        /* call the remote method */
        rc = nssi_call_rpc(&svcs[default_svc],
                NETCDF_CLOSE_OP,
                &ncid,
                NULL,
                0,
                NULL,
                req);

        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    } else {
        MPI_Bcast(&rc, 1, MPI_INT, 0, ncmpi_collective_op_comm);
    }
    if (rc != NSSI_OK) {
        log_error(netcdf_debug_level, "unable to call remote nc_close: %s",
                nssi_err_str(rc));
        goto cleanup;
    }

    file_state_map[ncid]->mode = NC_ASYNC_CLOSED;
    file_state_map[ncid]->outstanding_close_requests.push_back(req);

#if !defined(USE_ASYNC_CLOSE)
    rc=_nc_close_wait(ncid);
#endif

cleanup:
    log_debug(netcdf_debug_level, "exit");

    return rc;
}
extern "C"
int nc_close(int ncid)
{
    log_debug(netcdf_debug_level, "enter");

    MANIPULATE_ARGS(ncid);

    return _nc_close(ncid);
}

extern "C"
int nc_set_file_state(int ncid)
{
    int rc = NC_NOERR;

    log_debug(netcdf_debug_level, "enter");

    if (file_state_map[ncid] == NULL) {
        file_state_map[ncid]=new_file_state(NC_DEFINE_MODE);
    }
    MPI_Bcast(&(file_state_map[ncid]->mode), 1, MPI_INT, 0, MPI_COMM_WORLD);
    log_debug(netcdf_debug_level, "mode is %d", file_state_map[ncid]->mode);

    if (file_state_map[ncid]->mode == NC_CLOSED) {
        log_debug(netcdf_debug_level, "ncid(%d) state is closed.  removing file state.", ncid);
        if (file_state_map.find(ncid) != file_state_map.end()) {
            delete file_state_map[ncid];
            file_state_map.erase(ncid);
        }
    }

    log_debug(netcdf_debug_level, "exit");

    return rc;
}

#if DEBUG
int
main (int argc, char *argv[])
{
    int rc;
    int i,j;
    struct gengetopt_args_info args_info;
    double time;
    nssi_remote_pid server_id;
    nssi_service xfer_svc;
    data_array_t array;
    double start_time;
    FILE *result_fp = stdout;
    log_level debug_level;
    nssi_request *reqs;   /* the array of requests (for async experiments) */
    data_t *results;      /* the array of results (for async experiments) */

    /* unique to each process */
    int num_reqs;
    int myrank=0, np;

    MPI_Init(&argc, &argv);
    MPI_Comm_size(MPI_COMM_WORLD, &np);
    MPI_Comm_rank(MPI_COMM_WORLD, &myrank);

    if (myrank == 0) {
        /* Parse command line options to override defaults */
        if (cmdline_parser(argc, argv, &args_info) != 0) {
            exit(1);
        }
    }

    /* send the args to everyone else */
    MPI_Bcast(&args_info, sizeof(struct gengetopt_args_info), MPI_BYTE,
            0, MPI_COMM_WORLD);

    /* initialize logger */
    logger_init(args_info.verbose_arg, NULL);
    debug_level = args_info.verbose_arg;

    /* open the result file */
    if (myrank == 0) {
        if (args_info.result_file_arg != NULL) {
            result_fp = fopen(args_info.result_file_arg, args_info.result_file_mode_arg);
            if (result_fp == NULL) {
                log_warn(xfer_debug_level,
                        "invalid result file:"
                        "defaults to stdout");
                result_fp = stdout;
            }
        }
    }

    /* initialize NSSI RPC */
    nssi_ptl_init(PTL_IFACE_CLIENT, myrank+args_info.server_pid_arg + 1);
    nssi_rpc_init(NSSI_RPC_PTL, NSSI_RPC_XDR);

    server_id.nid = args_info.server_nid_arg;
    server_id.pid = args_info.server_pid_arg;

    /* If the user did not specify a nid, assume one */
    if (args_info.server_nid_arg == 0)  {
        nssi_remote_pid myid;

        /* get the id of this client process */
        nssi_get_id(&myid);

        /* assume server is on this host */
        server_id.nid = myid.nid;
    }

    print_args(result_fp, &args_info, "%");

    /* register the XDR encoding functions */
    NSSI_REGISTER_CLIENT_STUB(1, xfer_1_args, void, void);
    NSSI_REGISTER_CLIENT_STUB(2, xfer_2_args, void, void);


    log_info(debug_level, "\nTrying to get service at (%u,%u)\n", server_id.nid, server_id.pid);

    /* get the service description from the server */
    if (myrank == 0) {
        rc = nssi_get_service(server_id, -1, &xfer_svc);
        if (rc != NSSI_OK) {
            log_error(xfer_debug_level, "could not get svc description: %s",
                    nssi_err_str(rc));
            return rc;
        }
    }
    /* send the args to everyone else */
    MPI_Bcast(&xfer_svc, sizeof(nssi_service), MPI_BYTE,
            0, MPI_COMM_WORLD);

    log_debug(debug_level, "\nConnected to service\n");

    /* divide the requests among the processors */
    num_reqs = args_info.num_reqs_arg;

    MPI_Barrier(MPI_COMM_WORLD);

    /* allocate requests and results for async reqs */
    reqs = (nssi_request *)calloc(num_reqs, sizeof(nssi_request));
    results = (data_t *)calloc(num_reqs, sizeof(data_t));

    /* initialize the data array */
    array.data_array_t_len = args_info.len_arg;
    array.data_array_t_val = (data_t *)malloc(args_info.len_arg*sizeof(data_t));
    if (array.data_array_t_val == NULL) {
        log_error(xfer_debug_level, "out of space");
        return -1;
    }

    /* initialize the data */
    for (i=0; i<args_info.len_arg; i++) {
        array.data_array_t_val[i].int_val = (int)i;
        array.data_array_t_val[i].float_val = (float)i;
        array.data_array_t_val[i].double_val = (double)i;
    }


    /* loop over the exeriments (count == num_trials, num_reqs == ops_per_trial) */
    for (i=0; i<args_info.count_arg; i++) {

        log_debug(debug_level, "%d: trial=%d, reqs=%d, len=%d\n",
                myrank, i, args_info.num_reqs_arg, args_info.len_arg);

        switch (args_info.type_arg) {

            case XFER_ARGS_BLK:
                start_time = nssi_get_time();

                for (j=0; j<args_info.num_reqs_arg; j++) {
                    rc = xfer_1_clnt_blk(&xfer_svc, &array);
                    if (rc != NSSI_OK) {
                        log_error(xfer_debug_level, "could not transfer data: %s",
                                nssi_err_str(rc));
                        goto cleanup;
                    }
                }

                time = nssi_get_time() - start_time;
                break;

            case XFER_ARGS_NOBLK:
                start_time = nssi_get_time();

                /* submit requests */
                for (j=0; j<args_info.num_reqs_arg; j++) {
                    rc = xfer_1_clnt(&xfer_svc, &array, &reqs[j]);
                    if (rc != NSSI_OK) {
                        log_error(xfer_debug_level, "could not transfer data: %s",
                                nssi_err_str(rc));
                        goto cleanup;
                    }
                }

                /* wait for results */
                rc = nssi_waitall(&reqs, args_info.num_reqs_arg, -1);
                if (rc != NSSI_OK) {
                    log_error(xfer_debug_level, "error transferring data");
                    goto cleanup;
                }

                /* store the average time for an async req */
                time = nssi_get_time() - start_time;
                break;

            case XFER_DATA_BLK:
                start_time = nssi_get_time();

                /* submit requests */
                for (j=0; j<args_info.num_reqs_arg; j++) {
                    rc = xfer_2_clnt_blk(&xfer_svc, &array);
                    if (rc != NSSI_OK) {
                        log_error(xfer_debug_level, "could not transfer data: %s",
                                nssi_err_str(rc));
                        goto cleanup;
                    }
                }

                time = nssi_get_time() - start_time;
                break;

            case XFER_DATA_NOBLK:
                start_time = nssi_get_time();

                /* submit requests */
                for (j=0; j<args_info.num_reqs_arg; j++) {
                    rc = xfer_2_clnt(&xfer_svc, &array, &reqs[j]);
                    if (rc != NSSI_OK) {
                        log_error(xfer_debug_level, "could not transfer data: %s",
                                nssi_err_str(rc));
                        goto cleanup;
                    }
                }

                /* wait for results */
                rc = nssi_waitall(&reqs, args_info.num_reqs_arg, -1);
                if (rc != NSSI_OK) {
                    log_error(xfer_debug_level, "error transferring data");
                    goto cleanup;
                }

                /* store the average time for an async req */
                time = nssi_get_time() - start_time;
                break;

            default:
                log_error(xfer_debug_level, "unrecognized experiment type");
                return -1;
        }

        MPI_Barrier(MPI_COMM_WORLD);
        output_stats(result_fp, &args_info, time);
    }


cleanup:

    free(array.data_array_t_val);
    free(results);
    free(reqs);
    fclose(result_fp);

    nssi_rpc_fini();

    /* sizeof request header */
    fprintf(stderr, "sizeof(nssi_request_header)=%lu\n",
            (unsigned long)sizeof(nssi_request_header));
    fprintf(stderr, "sizeof(nssi_result_header)=%lu\n",
            (unsigned long)sizeof(nssi_result_header));

    MPI_Finalize();
    return 0;
}

#endif // DEBUG

#endif // HAVE_TRIOS_PNETCDF
