[comment {-*- tcl -*- doctools manpage}]
[vset SASL_VERSION 1.3.3]
[manpage_begin SASL n [vset SASL_VERSION]]
[keywords authentication]
[keywords SASL]
[copyright {2005-2006, Pat Thoyts <patthoyts@users.sourceforge.net>}]
[moddesc {Simple Authentication and Security Layer (SASL)}]
[titledesc {Implementation of SASL mechanisms for Tcl}]
[category  Networking]
[require Tcl 8.2]
[require SASL [opt [vset SASL_VERSION]]]
[description]
[para]

The Simple Authentication and Security Layer (SASL) is a framework
for providing authentication and authorization to comunications
protocols. The SASL framework is structured to permit negotiation
among a number of authentication mechanisms. SASL may be used in
SMTP, IMAP and HTTP authentication. It is also in use in XMPP, LDAP
and BEEP. See [sectref MECHANISMS] for the set of available SASL
mechanisms provided with tcllib.

[para]

The SASL framework operates using a simple multi-step challenge
response mechanism. All the mechanisms work the same way although the
number of steps may vary. In this implementation a callback procedure
must be provided from which the SASL framework will obtain users
details. See [sectref "CALLBACK PROCEDURE"] for details of this
procedure.

[section {COMMANDS}]

[list_begin definitions]

[call [cmd "::SASL::new"] [arg "option value [opt ...]"]]

Contruct a new SASL context. See [sectref OPTIONS] for details of the
possible options to this command. A context token is required for most
of the SASL procedures.

[call [cmd "::SASL::configure"] [arg "option value"] [opt [arg "..."]]]

Modify and inspect the SASL context option. See [sectref OPTIONS] for
further details.

[call [cmd "::SASL::step"] [arg "context"] [arg "challenge"] [opt [arg "..."]]]

This is the core procedure for using the SASL framework. The
[cmd step] procedure should be called until it returns 0. Each step takes a
server challenge string and the response is calculated and stored in
the context. Each mechanism may require one or more steps. For some
steps there may be no server challenge required in which case an empty
string should be provided for this parameter. All mechanisms should accept
an initial empty challenge.

[call [cmd "::SASL::response"] [arg "context"]]

Returns the next response string that should be sent to the server.

[call [cmd "::SASL::reset"] [arg "context"]]

Re-initialize the SASL context. Discards any internal state and
permits the token to be reused.

[call [cmd "::SASL::cleanup"] [arg "context"]]

Release all resources associated with the SASL context. The context
token may not be used again after this procedure has been called.

[call [cmd "::SASL::mechanisms"] [opt [arg "type"]] [opt [arg "minimum"]]]

Returns a list of all the available SASL mechanisms. The list is
sorted by the mechanism preference value (see [cmd register]) with the
preferred mechanisms and the head of the list. Any mechanism with a
preference value less than the[arg minimum] (which defaults to 0) is removed
from the returned list. This permits a security threshold to be set. Mechanisms
with a preference less that 25 transmit authentication are particularly
susceptible to eavesdropping and should not be provided unless a secure
channel is in use (eg: tls).
[para]
The [arg type] parameter
may be one of [arg client] or [arg server] and defaults to [arg client].
Only mechanisms that have an implementation matching the [arg type] are
returned (this permits servers to correctly declare support only for
mechanisms that actually provide a server implementation).

[call [cmd "::SASL::register"] [arg "mechanism"] [arg "preference"] \
  [arg "clientproc"] [opt [arg "serverproc"]]]

New mechanisms can be added to the package by registering the
mechanism name and the implementing procedures. The server procedure
is optional. The preference value is an integer that is used to order
the list returned by the [cmd mechanisms] command. Higher values
indicate a preferred mechanism. If the mechanism is already registered
then the recorded values are updated.

[list_end]

[section "OPTIONS"]

[list_begin definitions]

[def [option "-callback"]]

Specify a command to be evaluated when the SASL mechanism requires
information about the user. The command is called with the current
SASL context and a name specifying the information desired. See
[sectref EXAMPLES].

[def [option "-mechanism"]]

Set the SASL mechanism to be used. See [cmd mechanisms] for a list of
supported authentication mechanisms.

[def [option "-service"]]

Set the service type for this context. Some mechanisms may make use of
this parameter (eg DIGEST-MD5, GSSAPI and Kerberos). If not set it
defaults to an empty string. If the [option -type] is set to 'server'
then this option should be set to a valid service identity. Some
examples of valid service names are smtp, ldap, beep and xmpp.

[def [option "-server"]]

This option is used to set the server name used in SASL challenges
when operating as a SASL server.

[def [option "-type"]]

The context type may be one of 'client' or 'server'. The default is to
operate as a client application and respond to server
challenges. Mechanisms may be written to support server-side SASL and
setting this option will cause each [cmd step] to issue the next
challenge. A new context must be created for each incoming client
connection when in server mode.

[list_end]

[section "CALLBACK PROCEDURE"]

When the SASL framework requires any user details it will call the
procedure provided when the context was created with an argument that
specfies the item of information required.
[para]
In all cases a single response string should be returned.

[list_begin definitions]

[def "login"]

The callback procedure should return the users authorization identity.
Return an empty string unless this is to be different to the authentication
identity. Read [lb]1[rb] for a discussion about the specific meaning of
authorization and authentication identities within SASL.

[def "username"]

The callback procedure should return the users authentication identity.
Read [lb]1[rb] for a discussion about the specific meaning of
authorization and authentication identities within SASL.

[def "password"]

The callback procedure should return the password that matches the
authentication identity as used within the current realm.
[para]
For server mechanisms the password callback should always be called with
the authentication identity and the realm as the first two parameters.

[def "realm"]

Some SASL mechanisms use realms to partition authentication identities.
The realm string is protocol dependent and is often the current DNS
domain or in the case of the NTLM mechanism it is the Windows NT domain name.

[def "hostname"]

Returns the client host name - typically [lb]info host[rb].

[list_end]

[section "MECHANISMS"]

[list_begin definitions]

[def "ANONYMOUS"]

As used in FTP this mechanism only passes an email address for
authentication. The ANONYMOUS mechanism is specified in [lb]2[rb].

[def "PLAIN"]

This is the simplest mechanism. The users authentication details are
transmitted in plain text. This mechanism should not be provided
unless an encrypted link is in use - typically after SSL or TLS has
been negotiated.

[def "LOGIN"]

The LOGIN [lb]1[rb] mechanism transmits the users details with base64
encoding. This is no more secure than PLAIN and likewise should not be
used without a secure link.

[def "CRAM-MD5"]

This mechanism avoids sending the users password over the network in
plain text by hashing the password with a server provided random value
(known as a nonce). A disadvantage of this mechanism is that the
server must maintain a database of plaintext passwords for
comparison. CRAM-MD5 was defined in [lb]4[rb].

[def "DIGEST-MD5"]

This mechanism improves upon the CRAM-MD5 mechanism by avoiding the
need for the server to store plaintext passwords. With digest
authentication the server needs to store the MD5 digest of the users
password which helps to make the system more secure. As in CRAM-MD5
the password is hashed with a server nonce and other data before being
transmitted across the network. Specified in [lb]3[rb].

[def "OTP"]

OTP is the One-Time Password system described in RFC 2289 [lb]6[rb].
This mechanism is secure against replay attacks and also avoids storing
password or password equivalents on the server. Only a digest of a seed
and a passphrase is ever transmitted across the network. Requires the
[package otp] package from tcllib and one or more of the cryptographic
digest packages (md5 or sha-1 are the most commonly used).

[def "NTLM"]

This is a proprietary protocol developed by Microsoft [lb]5[rb] and is
in common use for authenticating users in a Windows network
environment. NTLM uses DES encryption and MD4 digests of the users
password to authenticate a connection. Certain weaknesses have been
found in NTLM and thus there are a number of versions of the protocol.
As this mechanism has additional dependencies it is made available as
a separate sub-package. To enable this mechanism your application must
load the [package SASL::NTLM] package.

[def "X-GOOGLE-TOKEN"]

This is a proprietary protocol developed by Google and used for
authenticating users for the Google Talk service. This mechanism makes
a pair of HTTP requests over an SSL channel and so this mechanism
depends upon the availability of the tls and http packages. To enable
this mechanism your application must load the [package SASL::XGoogleToken] package.
In addition you are recommended to make use of the autoproxy package to
handle HTTP proxies reasonably transparently.

[def "SCRAM"]

This is a protocol specified in RFC 5802 [lb]7[rb]. To enable this mechanism
your application must load the [package SASL::SCRAM] package.

[list_end]

[section "EXAMPLES"]

See the examples subdirectory for more complete samples using SASL
with network protocols. The following should give an idea how the SASL
commands are to be used. In reality this should be event
driven. Each time the [cmd step] command is called, the last server
response should be provided as the command argument so that the SASL
mechanism can take appropriate action.

[example {
proc ClientCallback {context command args} {
    switch -exact -- $command {
        login    { return "" }
        username { return $::tcl_platform(user) }
        password { return "SecRet" }
        realm    { return "" }
        hostname { return [info host] }
        default  { return -code error unxpected }
    }
}

proc Demo {{mech PLAIN}} {
    set ctx [SASL::new -mechanism $mech -callback ClientCallback]
    set challenge ""
    while {1} {
        set more_steps [SASL::step $ctx challenge]
        puts "Send '[SASL::response $ctx]'"
        puts "Read server response into challenge var"
        if {!$more_steps} {break}
    }
    SASL::cleanup $ctx
}
}]

[section "REFERENCES"]

[list_begin enumerated]

[enum]
    Myers, J. "Simple Authentication and Security Layer (SASL)",
    RFC 2222, October 1997.
    ([uri http://www.ietf.org/rfc/rfc2222.txt])

[enum]
    Newman, C. "Anonymous SASL Mechanism",
    RFC 2245, November 1997.
    ([uri http://www.ietf.org/rfc/rfc2245.txt])

[enum]
    Leach, P., Newman, C. "Using Digest Authentication as a SASL
    Mechanism", RFC 2831, May 2000,
    ([uri http://www.ietf.org/rfc/rfc2831.txt])

[enum]
    Klensin, J., Catoe, R. and Krumviede, P.,
    "IMAP/POP AUTHorize Extension for Simple Challenge/Response"
    RFC 2195, September 1997.
    ([uri http://www.ietf.org/rfc/rfc2195.txt])

[enum]
    No official specification is available. However,
    [uri http://davenport.sourceforge.net/ntlm.html] provides a good
    description.

[enum]
    Haller, N. et al., "A One-Time Password System",
    RFC 2289, February 1998,
    ([uri http://www.ieft.org/rfc/rfc2289.txt])

[enum]
    Newman, C. et al., "Salted Challenge Response Authentication Mechanism (SCRAM) SASL and GSS-API Mechanisms",
    RFC 5802, July 2010,
    ([uri http://www.ieft.org/rfc/rfc5802.txt])

[list_end]

[section AUTHORS]
Pat Thoyts

[vset CATEGORY sasl]
[include ../doctools2base/include/feedback.inc]
[manpage_end]
