% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fastmap.R
\name{fastmap}
\alias{fastmap}
\title{Create a fastmap object}
\usage{
fastmap(missing_default = NULL)
}
\arguments{
\item{missing_default}{The value to return when \code{get()} is called with a
key that is not in the map. The default is \code{NULL}, but in some cases
it can be useful to return a sentinel value, such as a
\code{\link{key_missing}} object.}
}
\description{
A fastmap object provides a key-value store where the keys are strings and
the values are any R objects.
}
\details{
In R, it is common to use environments as key-value stores, but they can leak
memory: every time a new key is used, R registers it in its global symbol
table, which only grows and is never garbage collected. If many different
keys are used, this can cause a non-trivial amount of memory leakage.

Fastmap objects do not use the symbol table and do not leak memory.

Unlike with environments, the keys in a fastmap are always encoded as UTF-8,
so if you call \code{$set()} with two different strings that have the same
Unicode values but have different encodings, the second call will overwrite
the first value. If you call \code{$keys()}, it will return UTF-8 encoded
strings, and similarly, \code{$as_list()} will return a list with names that
have UTF-8 encoding.

Note that if you call \code{$mset()} with a named argument, where the name is
non-ASCII, R will convert the name to the native encoding before fastmap has
the chance to convert them to UTF-8, and the keys may get mangled in the
process. However, if you use \code{$mset(.list = x)}, then R will not convert
the keys to the native encoding, and the keys will be correctly converted to
UTF-8. With \code{$mget()}, the keys will be converted to UTF-8 before they
are fetched.

Fastmap objects have the following methods:

\describe{
  \item{\code{set(key, value)}}{
    Set a key-value pair. \code{key} must be a string. Returns \code{value}.
  }
  \item{\code{mset(..., .list = NULL)}}{
    Set multiple key-value pairs. The key-value pairs are named arguments,
    and/or a list passed in as \code{.list}. Returns a named list where the
    names are the keys, and the values are the values.
  }
  \item{\code{get(key, missing = missing_default)}}{
    Get a value corresponding to \code{key}. If the key is not in the map,
    return \code{missing}.
  }
  \item{\code{mget(keys, missing = missing_default)}}{
    Get values corresponding to \code{keys}, which is a character vector. The
    values will be returned in a named list where the names are the same as
    the \code{keys} passed in, in the same order. For keys not in the map,
    they will have \code{missing} for their value.
  }
  \item{\code{has(keys)}}{
    Given a vector of keys, returns a logical vector reporting whether each
    key is contained in the map.
  }
  \item{\code{remove(keys)}}{
    Given a vector of keys, remove the key-value pairs from the map. Returns
    a logical vector reporting whether each item existed in (and was removed
    from) the map.
  }
  \item{\code{keys(sort = FALSE)}}{
    Returns a character vector of all the keys. By default, the keys will be
    in arbitrary order. Note that the order can vary across platforms and is
    not guaranteed to be consistent. With \code{sort=TRUE}, the keys will be
    sorted according to their Unicode code point values.
  }
  \item{\code{size()}}{
    Returns the number of items in the map.
  }
  \item{\code{as_list(sort = FALSE)}}{
    Return a named list where the names are the keys from the map, and the
    values are the values. By default, the keys will be in arbitrary order.
    Note that the order can vary across platforms and is not guaranteed to
    be consistent. With \code{sort=TRUE}, the keys will be sorted according
    to their Unicode code point values.
  }
  \item{\code{reset()}}{
    Reset the fastmap object, clearing all items.
  }
}
}
\examples{
# Create the fastmap object
m <- fastmap()

# Set some key-value pairs
m$set("x", 100)
m$set("letters", c("a", "b", "c"))
m$mset(numbers = c(10, 20, 30), nothing = NULL)

# Get values using keys
m$get("x")
m$get("numbers")
m$mget(c("letters", "numbers"))

# Missing keys return NULL by default, but this can be customized
m$get("xyz")

# Check for existence of keys
m$has("x")
m$has("nothing")
m$has("xyz")

# Remove one or more items
m$remove(c("letters", "x"))

# Return number of items
m$size()

# Get all keys
m$keys()

# Return named list that represents all key-value pairs
str(m$as_list())

# Clear the map
m$reset()


# Specify missing value when get() is called
m <- fastmap()
m$get("x", missing = key_missing())
#> <Key Missing>

# Specify the default missing value
m <- fastmap(missing_default = key_missing())
m$get("x")
#> <Key Missing>

}
