/***************************************************************************
 *   Copyright (C) 2000-2008 by Johan Maes                                 *
 *   on4qz@telenet.be                                                      *
 *   http://users.telenet.be/on4qz                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "imageviewer.h"
#include "utils/supportfunctions.h"
#include "utils/logging.h"
#include "configparams.h"
#include "dispatcher.h"
#include "txwidget.h"
#include "utils/qjp2io.h"

#include <QtGui>
#include <QMenu>

#define RATIOSCALE 800.

/**
  \class imageViewer

  The image is stored in it's original format and size.
  All interactions are done on the original image.
  A scaled version is used to display the contents.
*/	


imageViewer::imageViewer(QWidget *parent): QLabel(parent)
{
  addToLog("image creation",LOGIMAG);
  validImage=false;
  setFrameStyle(QFrame::Sunken | QFrame::Panel);
  QBrush b;
  QPalette palette;
  b.setTexture(QPixmap::fromImage(QImage(":/icons/transparency.png")));
  palette.setBrush(QPalette::Active,QPalette::Base,b);
  palette.setBrush(QPalette::Inactive,QPalette::Base,b);
  palette.setBrush(QPalette::Disabled,QPalette::Base,b);
  setPalette(palette);
  setBackgroundRole(QPalette::Base);
  setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Ignored);
  setScaledContents(true);
  setBackgroundRole(QPalette::Dark);

  popup=new QMenu (this);
  newAct = new QAction(tr("&New"),this);
  connect(newAct, SIGNAL(triggered()), this, SLOT(slotNew()));
  loadAct = new QAction(tr("&Load"), this);
  connect(loadAct, SIGNAL(triggered()), this, SLOT(slotLoad()));
  toTXAct = new QAction(tr("&To TX"), this);
  connect(toTXAct, SIGNAL(triggered()), this, SLOT(slotToTX()));
  editAct = new QAction(tr("&Edit"), this);
  connect(editAct, SIGNAL(triggered()), this, SLOT(slotEdit()));
  printAct = new QAction(tr("&Print"), this);
  connect(printAct, SIGNAL(triggered()), this, SLOT(slotPrint()));
  deleteAct = new QAction(tr("&Delete"), this);
  connect(deleteAct, SIGNAL(triggered()), this, SLOT(slotDelete()));
  propertiesAct = new QAction(tr("Propert&ies"), this);
  connect(propertiesAct, SIGNAL(triggered()), this, SLOT(slotProperties()));
  popup->addAction(newAct);
  popup->addAction(loadAct);
  popup->addAction(toTXAct);
  popup->addAction(editAct);
  popup->addAction(printAct);
  popup->addAction(deleteAct);
  popup->addAction(propertiesAct);
  setScaledContents(false);
  setAlignment(Qt::AlignCenter);
  setAutoFillBackground(true);
  QPalette mpalette;
  mpalette.setColor(QPalette::Window,QColor(0,0,128));
  setBackgroundRole(QPalette::Window);
  mpalette.setColor(QPalette::WindowText, Qt::yellow);
  setPalette(mpalette);
  init(RXIMG);
  activeMovie=false;
  //
}

imageViewer::~imageViewer()
{
}

void imageViewer::init(thumbType tp)
{
  setScaledContents(false);
  setAlignment(Qt::AlignCenter);
  setAutoFillBackground(true);
  QPalette mpalette;
  mpalette.setColor(QPalette::Window,QColor(0,0,128));
  setBackgroundRole(QPalette::Window);
  mpalette.setColor(QPalette::WindowText, Qt::yellow);
  setPalette(mpalette);
  addToLog("image creation",LOGIMAG);
  imageFileName="";
  image=QImage();
  setType(tp);
  popupEnabled=true;
  setPixmap(QPixmap());
  clear();
}

bool imageViewer::openImage(QString &filename,QString start,bool ask,bool showMessage,bool emitSignal)
{
  QImage tempImage;
  displayMBoxEvent *stmb=0;
  editorScene ed;
  if(activeMovie)
    {
      activeMovie=false;
      qm.stop();
    }
  if (filename.isEmpty()&&!ask) return false;
  if(ask)
    {
      dirDialog dd((QWidget *)this,"Browse");
      filename=dd.openFileName(start,"*");
    }
  if(filename.isEmpty())
    {
      imageFileName="";
      return false;
    }
  if(!tempImage.load(filename))
    {
      QFile fi(filename);
      // try to load it with the editor, it coud be a template file
      if(!ed.load(fi))
        {
          //try loading jpg2
          tempImage=readJP2Image(filename);
          if(tempImage.isNull())
            {
              if(showMessage)
                {
                  stmb= new displayMBoxEvent("Image Loader",QString("Unable to load image:\n%1").arg(filename));
                  QApplication::postEvent( dispatcherPtr, stmb );  // Qt will delete it when done
                }
              validImage=false;
              imageFileName="";
              return false;
            }
        }
      else
        {
          tempImage=QImage(ed.renderImage(0,0)->copy());
        }
    }
  image=tempImage.convertToFormat(QImage::Format_ARGB32_Premultiplied);
  imageFileName=filename;
  QFileInfo fi(filename);
  if (fi.suffix().toLower()=="gif")
    {
      //we will try a animated gif
      qm.setFileName(filename);
      if(qm.isValid())
        {
          if(qm.frameCount()>1)
            {
              activeMovie=true;
              setMovie(&qm);
              qm.start();
            }
        }
    }
  else
    {
      displayImage(false);
    }
  validImage=true;
  if (emitSignal) emit imageChanged();
  return true;
}

bool imageViewer::openImage(QString &filename,bool showMessage,bool emitSignal)
{
  return openImage(filename,"",false,showMessage,emitSignal);
}

bool imageViewer::openImage(QImage im)
{
  image=im;
  if(!image.isNull())
    {
      displayImage(false);
      validImage=true;
      return true;
    }
  validImage=false;

  return false;
}

bool imageViewer::openImage(QByteArray *ba)
{
  QImage tempImage;
  QBuffer buffer(ba);
  buffer.open(QIODevice::ReadOnly);
  if(tempImage.load(&buffer,NULL))
    {
      imageFileName="";
      validImage=true;
      image=tempImage.convertToFormat(QImage::Format_ARGB32_Premultiplied);
      displayImage(false);
      return true;
    }
  validImage=false;
  return false;
}



bool imageViewer::hasValidImage()
{
  return validImage;
}

QImage *imageViewer::createImage(QSize sz,QColor fill)
{
  image=QImage(sz,QImage::Format_ARGB32_Premultiplied);
  image.fill(fill.rgb());
  displayImage(false);
  emit imageChanged();
  return &image;
}

void imageViewer::copy(imageViewer *src)
{
  imageFileName=src->imageFileName;
  ttype=src->ttype;
  openImage(imageFileName,false,false);
}



QRgb *imageViewer::getScanLineAddress(int line)
{
  return (QRgb *)image.scanLine(line);
}




void imageViewer::displayImage(bool useCompression)
{
  QTemporaryFile itmp(QDir::tempPath()+"/qsstvXXXXXX.jp2");
  if((image.width()==0) || (image.height()==0)) return;

  //  qDebug()<< "displayImage" << image.width() << image.height() << width() << height();
  if(ttype==TXIMG)
    {
      // we want to display the image with the defined quality
      QBuffer buffer;
      if(useCompression)
        {
          if(!itmp.open()) return;
          writeJP2Image(image.convertToFormat(QImage::Format_RGB32), itmp.fileName(),psizeRatio);
          compressedImage=readJP2Image(itmp.fileName());
        }
      else
        {
          compressedImage=image.copy();
        }



      if(compressedImage.isNull()||width()==0 || height()==0)
        {

          return;
        }

      if(hasScaledContents())
        {
          setPixmap(QPixmap::fromImage(compressedImage));
        }
      else
        {
//         qDebug() << "WxH" << width() << height() << compressedImage.width() << compressedImage.height();
          setPixmap(QPixmap::fromImage(compressedImage.scaled(width(),height(),Qt::KeepAspectRatio,Qt::SmoothTransformation)));
        }

    }
  else
    {
      if(hasScaledContents())
        {
          setPixmap(QPixmap::fromImage(image));
        }
      else
        {
//          qDebug() << "WxH" << width() << height() << image.width() << image.height();
          setPixmap(QPixmap::fromImage(image.scaled(width()-2,height()-2,Qt::KeepAspectRatio,Qt::SmoothTransformation)));
        }
    }

}


bool imageViewer::readThumbSettings(QSettings *qs)
{
  qs->beginGroup("Thumbnails");
  addToLog(QString("/THUMB/"+objectName()+"/"),LOGIMAG);
  imageFileName =qs->value(objectName()+"/filename","").toString();
  qs->endGroup();
  return openImage(imageFileName,false,false);
}

void imageViewer::writeThumbSettings(QSettings *qs)
{
  qs->beginGroup("Thumbnails");
  qs->setValue(objectName()+"/filename",imageFileName);
  qs->endGroup();
}

void imageViewer::setType(thumbType tp)
{
  ttype=tp;
  switch(tp)
    {
    case RXIMG:
    case RXTHUMB:
      imageFilePath=rxImagesPath;
    break;
    case TXIMG:
    case TXTHUMB:
      imageFilePath=txImagesPath;
    break;
    case TEMPLATETHUMB:
      imageFilePath=templatesPath;
    break;
    }
  if((tp==RXTHUMB) || (tp==TXTHUMB) || (tp==TEMPLATETHUMB))
    {
      setScaledContents(true);
      setAlignment(Qt::AlignCenter);
    }
}

void imageViewer::mousePressEvent( QMouseEvent *e )
{
  QString temp;
  QString fn;
  if (e->button() == Qt::RightButton)
    {
      if(popupEnabled)
        {
          popup->popup(QCursor::pos());
        }
    }
}

void imageViewer::slotDelete()
{
  if(imageFileName.isEmpty()) return;
  if(QMessageBox::question(this,"Delete file","Do you want to delete the file and\n move it to the trash folder?",QMessageBox::Yes|QMessageBox::No)==QMessageBox::Yes)
    {
      trash(imageFileName,true);
    }
  imageFileName="";
  emit layoutChanged();
}

void imageViewer::slotEdit()
{
  if(imageFileName.isEmpty())
    {
      slotLoad();
      if (imageFileName.isEmpty()) return;
    }
  callEditorEvent *ce = new callEditorEvent( this,imageFileName );
  QApplication::postEvent(dispatcherPtr, ce );  // Qt will delete it when done
}


void imageViewer::slotLoad()
{
  QString fileNameTmp;
  dirDialog dd((QWidget *)this,"Browse");
  fileNameTmp=dd.openFileName(imageFilePath);
  if(openImage(fileNameTmp,true,true))
    {
      imageFileName=fileNameTmp;
      if(ttype==TEMPLATETHUMB)
        {
          templatesChangedEvent *ce = new templatesChangedEvent( );
          QApplication::postEvent(dispatcherPtr, ce );  // Qt will delete it when done
        }
    }
}



void imageViewer::slotNew()
{
  callEditorEvent *ce = new callEditorEvent( this,NULL);
  QApplication::postEvent(dispatcherPtr, ce );  // Qt will delete it when done
}


void imageViewer::slotPrint()
{
}

void imageViewer::slotProperties()
{
  QFileInfo fi(imageFileName);
  if(fi.exists())
    {
      QMessageBox::information(this,"Image Properties",
                               "File: " + imageFileName
                               + "\n File size:     " + QString::number(fi.size())
                               + "\n Image width:   " + QString::number(image.width())
                               + "\n Image heigth:  " + QString::number(image.height())
                               + "\n Last Modified: " + fi.lastModified().toString()
                               ,QMessageBox::Ok);
    }
  else
    {
      QMessageBox::information(this,"Image Properties",
                               "No image loaded",QMessageBox::Ok);
    }

}

void imageViewer::slotToTX()
{
  txWidgetPtr->setImage(imageFileName);
}


void imageViewer::save(QString fileName,QString fmt,bool convertRGB)
{
  QImage im;
  if(image.isNull()) return;
  if(!convertRGB)
    {
      image.save(fileName,fmt.toUpper().toLatin1().data());
    }
  else
    {
      im=image.convertToFormat(QImage::Format_RGB32);
      im.save(fileName,fmt.toUpper().toLatin1().data());
    }
}

void imageViewer::copyToBuffer(QByteArray *ba)
{
  QTemporaryFile itmp(QDir::tempPath()+"/qsstvXXXXXX.jp2");
  if(image.isNull()) return;
  if(!itmp.open()) return;
  writeJP2Image(image.convertToFormat(QImage::Format_RGB32), itmp.fileName(),psizeRatio);
  *ba=itmp.readAll();
}

int imageViewer::calcSize(int &sizeRatio)
{
  if(sizeRatio>100) sizeRatio=100;
  if(sizeRatio<1) sizeRatio=1;
  return (int)(rint(((double)image.byteCount()*sizeRatio))/(RATIOSCALE*1000.));
}


void imageViewer:: setSizeRatio(int sizeRatio)
{
  format="jp2";
  if(sizeRatio>100) sizeRatio=100;
  if(sizeRatio<1) sizeRatio=1;
  psizeRatio=sizeRatio/RATIOSCALE;
}

bool imageViewer::reload()
{
  return openImage(imageFileName ,true,false);
}


void imageViewer::scale( int w, int h)
{
  image=image.scaled(QSize(w,h),Qt::IgnoreAspectRatio,Qt::SmoothTransformation);
}

void imageViewer::applyTemplate(QString templateName, bool useTemplate,int w,int h)
{
  if(!reload()) return ;
  addToLog("apply temlate",LOGIMAG);
  //  sconvert cnv;
  editorScene tscene(0);
  QFile fi(templateName);
  if((!fi.fileName().isEmpty())  && (useTemplate))
    {
      tscene.load(fi);
      tscene.addConversion('c',toCall,true);
      tscene.addConversion('r',rsv);
      tscene.addConversion('o',toOperator);
      tscene.addConversion('t',QDateTime::currentDateTime().toUTC().toString("hh:mm"));
      tscene.addConversion('d',QDateTime::currentDateTime().toUTC().toString("yyyy/MM/dd"));
      tscene.addConversion('m',myCallsign);
      tscene.addConversion('q',myQth);
      tscene.addConversion('l',myLocator);
      tscene.addConversion('n',myLastname);
      tscene.addConversion('f',myFirstname);
      tscene.addConversion('v',qsstvVersion);
      tscene.addConversion('x',comment1);
      tscene.addConversion('y',comment2);
      tscene.addConversion('z',comment3);
      addToLog(QString("applyTemplate size=%1,%2").arg(tscene.width()).arg(tscene.height()),LOGIMAG);
      if(w!=0 && h!=0)
        {
          overlayedImage= QImage(image.scaled(w,h));
          tscene.overlay(&overlayedImage);
        }
      else
        {
          tscene.overlay(&image);
        }
      openImage(*tscene.getImagePtr());
    }
  else
    {
      if(w!=0 && h!=0)
        {
          overlayedImage= QImage(image.scaled(w,h,Qt::IgnoreAspectRatio,Qt::SmoothTransformation)); // same resolution as sstv mode
          openImage(overlayedImage);
        }
      else openImage(image);
    }
  overlayedImage=QImage();
}





