/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower86@gmail.com                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/
#include "formattingtoolbar.h"

#include "colorchooser.h"
#include "qchaticon.h"

#include <QFontDatabase>
#include <QApplication>
#include <QGridLayout>
#include <QSpinBox>
#include <QPushButton>

FormattingToolBar::FormattingToolBar(QWidget* parent)
 : QToolBar(parent),
  m_tableRows(0),
  m_tableCols(0)
{
  QFontDatabase db;

  QGridLayout* grid;


  m_fontFamilyCmbx = new QFontComboBox(this);
  m_fontSizeCmbx   = new QComboBox(this);
  m_textStyleCmbx  = new QComboBox(this);
  m_colorPicker    = new ColorPicker;

  m_tableSizeDlg   = new QDialog(0);
  grid             = new QGridLayout(m_tableSizeDlg);

  m_enterTableSizeLab = new QLabel(m_tableSizeDlg);
  m_rowsLab           = new QLabel(m_tableSizeDlg);
  m_colsLab           = new QLabel(m_tableSizeDlg);

  // creating table size dlg
  // ************************
  QPushButton* btn     = new QPushButton(tr("Ok"), m_tableSizeDlg);
  QSpinBox*    rows    = new QSpinBox(m_tableSizeDlg);
  QSpinBox*    cols    = new QSpinBox(m_tableSizeDlg);

  grid->addWidget(m_enterTableSizeLab, 0, 0, 1, 2);
  grid->addWidget(m_rowsLab          , 1, 0);
  grid->addWidget(m_colsLab          , 1, 1);
  grid->addWidget(rows               , 2, 0);
  grid->addWidget(cols               , 2, 1);
  grid->addWidget(btn                , 3, 0, 1, 2);

  connect(rows, SIGNAL(valueChanged(int)), this, SLOT(setTableRows (int)));
  connect(cols, SIGNAL(valueChanged(int)), this, SLOT(setTableCols(int)));
  connect(btn , SIGNAL(clicked())        , this, SLOT(createTable()));
  connect(btn , SIGNAL(clicked())        , m_tableSizeDlg, SLOT(accept()));
  // ************************

  // FIXME disabling lists while current verion of qt4.3 have a problems with images in lists
//   addWidget(m_textStyleCmbx);
  m_textStyleCmbx->hide();

  addWidget(m_fontFamilyCmbx);
  addWidget(m_fontSizeCmbx);

  (m_setBoldAct      = addAction(tr("Bold")))->setCheckable(true);
  (m_setItalicAct    = addAction(tr("Italic")))->setCheckable(true);
  (m_setUnderlineAct = addAction(tr("Underline")))->setCheckable(true);
  m_insertTableAct   = addAction(tr("Insert Table"));

  addWidget(m_colorPicker);

  m_textStyleCmbx->addItem("Standard");
  m_textStyleCmbx->addItem("Bullet List (Disc)");
  m_textStyleCmbx->addItem("Bullet List (Circle)");
  m_textStyleCmbx->addItem("Bullet List (Square)");
  m_textStyleCmbx->addItem("Ordered List (Decimal)");
  m_textStyleCmbx->addItem("Ordered List (Alpha lower)");
  m_textStyleCmbx->addItem("Ordered List (Alpha upper)");

  connect(m_textStyleCmbx, SIGNAL(activated(int)),
                     this, SIGNAL(wantSetTextStyle(int)));

  m_fontSizeCmbx->setEditable(true);

  foreach(int sz, db.standardSizes())
    m_fontSizeCmbx->addItem(QString::number(sz));

  m_fontSizeCmbx->setCurrentIndex(m_fontSizeCmbx->findText(QString::number(QApplication::font().pointSize())));

  connect(m_setBoldAct     , SIGNAL(triggered(bool)), this, SIGNAL(wantSetBold(bool)));
  connect(m_setItalicAct   , SIGNAL(triggered(bool)), this, SIGNAL(wantSetItalic(bool)));
  connect(m_setUnderlineAct, SIGNAL(triggered(bool)), this, SIGNAL(wantSetUnderline(bool)));
  connect(m_insertTableAct , SIGNAL(triggered(bool)), m_tableSizeDlg, SLOT(exec()));

  connect(m_colorPicker   , SIGNAL(colorChanged(QColor)), this, SIGNAL(wantSetColor(QColor)));
  connect(m_fontFamilyCmbx, SIGNAL(activated(QString))  , this, SIGNAL(wantSetFontFamily(QString)));
  connect(m_fontSizeCmbx  , SIGNAL(activated(QString))  , this, SIGNAL(wantSetFontSize(QString)));

  retranslate();
  setIcons();
}


FormattingToolBar::~FormattingToolBar()
{
  qDebug("[~FormattingToolBar]");
  delete m_tableSizeDlg;
}

void FormattingToolBar::currentCharFormatChanged(const QTextCharFormat & fmt)
{
  m_fontFamilyCmbx->setCurrentIndex(m_fontFamilyCmbx->findText(QFontInfo(fmt.font()).family()));
  m_fontSizeCmbx  ->setCurrentIndex(m_fontSizeCmbx  ->findText(QString::number(fmt.font().pointSize())));

  m_setBoldAct->setChecked(fmt.font().bold());
  m_setItalicAct->setChecked(fmt.font().italic());
  m_setUnderlineAct->setChecked(fmt.font().underline());

  m_colorPicker->setColor(fmt.foreground().color());
}

void FormattingToolBar::retranslate()
{
  m_setBoldAct       ->setText(tr("Bold"));
  m_setItalicAct     ->setText(tr("Italic"));
  m_setUnderlineAct  ->setText(tr("Underline"));
  m_enterTableSizeLab->setText(tr("Enter table size"));
  m_rowsLab          ->setText(tr("Rows count:"));
  m_colsLab          ->setText(tr("Columns count:"));
  m_tableSizeDlg     ->setWindowTitle(tr("Insert Table"));
}

void FormattingToolBar::setIcons()
{
  m_setBoldAct     ->setIcon(QChatIcon::icon("format-text-bold"));
  m_setItalicAct   ->setIcon(QChatIcon::icon("format-text-italic"));
  m_setUnderlineAct->setIcon(QChatIcon::icon("format-text-underline"));
  m_insertTableAct ->setIcon(QChatIcon::icon("insert-table"));
}
