# Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import os
import shutil
import yaml
import zipfile

from murano import context
from murano.db import session


MANIFEST = {'Format': 'MuranoPL/1.0',
            'Type': 'Application',
            'Description': 'MockApp for API tests',
            'Author': 'Mirantis, Inc'}


def dummy_context(user='test_username', tenant_id='test_tenant_id',
                  request_id='dummy-request', **kwargs):

    # NOTE(kzaitsev) we need to pass non-False value to request_id, to
    # prevent it being generated by oslo during tests.
    params = {
        'request_id': request_id,
        'tenant': tenant_id,
        'user': user,
    }
    params.update(kwargs)
    return context.RequestContext.from_dict(params)


def save_models(*models):
    s = session.get_session()
    for m in models:
        m.save(s)


def compose_package(app_name, package_dir,
                    require=None, archive_dir=None, add_class_name=False,
                    manifest_required=True, version=None):
    """Composes a murano package

    Composes package `app_name` manifest and files from `package_dir`.
    Includes `require` section if any in the manifest file.
    Puts the resulting .zip file into `archive_dir` if present or in the
    `package_dir`.
    """
    tmp_package_dir = os.path.join(archive_dir, os.path.basename(package_dir))
    shutil.copytree(package_dir, tmp_package_dir)
    package_dir = tmp_package_dir

    if manifest_required:
        manifest = os.path.join(package_dir, "manifest.yaml")
        with open(manifest, 'w') as f:
            fqn = 'io.murano.apps.' + app_name
            mfest_copy = MANIFEST.copy()
            mfest_copy['FullName'] = fqn
            mfest_copy['Name'] = app_name
            mfest_copy['Classes'] = {fqn: 'mock_muranopl.yaml'}
            if require:
                mfest_copy['Require'] = {str(name): version
                                         for name, version in require}
            if version:
                mfest_copy['Version'] = version
            f.write(yaml.dump(mfest_copy, default_flow_style=False))

    if add_class_name:
        class_file = os.path.join(package_dir, 'Classes', 'mock_muranopl.yaml')
        with open(class_file, 'r') as f:
            contents = f.read()

        index = contents.index('Extends')
        contents = "{0}Name: {1}\n\n{2}".format(contents[:index], app_name,
                                                contents[index:])
        with open(class_file, 'w') as f:
            f.write(contents)

    if require:
        class_file = os.path.join(package_dir, 'Classes', 'mock_muranopl.yaml')
        with open(class_file, 'r') as f:
            content = f.read()

        index_string = 'deploy:\n    Body:\n      '
        index = content.index(index_string) + len(index_string)
        class_names = [req[0][req[0].rfind('.') + 1:] for req in require]
        addition = "".join(["- new({})\n".format(name) + ' ' * 6
                            for name in class_names])
        content = content[:index] + addition + content[index:]
        with open(class_file, 'w') as f:
            f.write(content)

    name = app_name + '.zip'

    if not archive_dir:
        archive_dir = os.path.dirname(os.path.abspath(__file__))
    archive_path = os.path.join(archive_dir, name)

    with zipfile.ZipFile(archive_path, 'w') as zip_file:
        for root, dirs, files in os.walk(package_dir):
            for f in files:
                zip_file.write(
                    os.path.join(root, f),
                    arcname=os.path.join(os.path.relpath(root, package_dir), f)
                )

    return archive_path, name
