/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* file:     submesh_2d.c                                                   */
/*                                                                          */
/*                                                                          */
/* description: Support for master/slave meshes for master->dim == 2        */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.alberta-fem.de                                               */
/*                                                                          */
/*  (c) by D. Koester (2005)                                                */
/*--------------------------------------------------------------------------*/


static void get_slave_elements_rec_2d(MESH *master, MESH *slave,
				      int neigh, EL *m_el, EL *s_el)
{
  /* FUNCNAME("get_slave_elements_rec_2d"); */
  EL_INFO el_info = {};


  if(m_el->child[0]) {
    if(neigh == 0)
      get_slave_elements_rec_2d(master, slave, 2,
				m_el->child[1], s_el);
    else if(neigh == 1)
      get_slave_elements_rec_2d(master, slave, 2,
				m_el->child[0], s_el);
    else if(neigh == 2) {
      /* We MUST copy m_el->new_coord here!!! */
      if (m_el->new_coord) {
	s_el->new_coord = get_real_d(slave);
	COPY_DOW(m_el->new_coord, s_el->new_coord);
      }
      el_info.mesh = slave;
      el_info.el   = s_el;
      s_el->mark   = 1;

      AI_refine_fct_1d(&el_info, nil);

      get_slave_elements_rec_2d(master, slave, 0,
				m_el->child[0], s_el->child[0]);
      get_slave_elements_rec_2d(master, slave, 1,
				m_el->child[1], s_el->child[1]);
    }
  }

  return;
}


static void get_slave_elements_2d(MESH *master, MESH *slave,
				  int (*binding_method)
				  (MESH *master, MACRO_EL *el, int face,
				   void *data), void *data)
{
  FUNCNAME("get_slave_elements_2d");
  MACRO_EL *m_mel, *s_mel;
  int       n, i;

  s_mel = slave->macro_els;
  for(n = 0; n < master->n_macro_el; n++) {
    m_mel = master->macro_els + n;

    for(i = 0; i < N_NEIGH_2D; i++)
      if(binding_method(master, m_mel, i, data)) {
	DEBUG_TEST_EXIT(s_mel,
		    "Ran out of slave macro elements... Wrong meshes?\n");

	get_slave_elements_rec_2d(master, slave,
				  i, m_mel->el, s_mel->el);

	s_mel++;
      }
  }

  return;
}


/****************************************************************************/
/* master interpolation/restriction routines.                               */
/* These call the corresponding refinement/coarsening routines for the slave*/
/* mesh patchwise.                                                          */
/****************************************************************************/

static void master_interpol_2d(DOF_PTR_VEC *m_dpv, RC_LIST_EL *rclist, int mn)
{
  FUNCNAME("master_interpol_2d");
  MESH_MEM_INFO   *m_mem_info = (MESH_MEM_INFO *)
    m_dpv->fe_space->admin->mesh->mem_info;
  int              m_n0 = m_dpv->fe_space->admin->n0_dof[EDGE];
  int              m_n = m_dpv->fe_space->admin->mesh->node[EDGE];
  EL              *m_el, *s_el;
  EL              *s_child[2], *m_child[2];
  int              s_n0, s_n, i, j, n_slaves = m_mem_info->n_slaves;
  MESH            *slave = nil;
  DOF_PTR_VEC     *s_dpv;
  NODE_PROJECTION *s_proj;

/****************************************************************************/
/* Retrieve the slave mesh. Rather unelegant, sorry...                      */
/****************************************************************************/
  for (j = 0; j < n_slaves; j++) {
    slave = m_mem_info->slaves[j];
    if(((MESH_MEM_INFO *)slave->mem_info)->slave_binding == m_dpv) break;
  }
  DEBUG_TEST_EXIT(j < n_slaves, "Slave mesh not found!\n");
  
  s_dpv   = ((MESH_MEM_INFO *)slave->mem_info)->master_binding; 
  s_n0    = s_dpv->fe_space->admin->n0_dof[CENTER];
  s_n     = slave->node[CENTER];

/****************************************************************************/
/* Run over all patch elements. Check if any edges belong to slave elements.*/
/* If so, determine if they should be refined together with the master      */
/* elements. All necessary DOFs are set. New DOFs are set to nil, if they   */
/* should not point to anything. THIS IS NOT JUST A SECURITY MEASURE - IT   */
/* IS ABSOLUTELY NECESSARY! At the moment, new entries in a DOF vector can  */
/* not be guaranteed to be clean.                                           */
/****************************************************************************/

  for(i = 0; i < mn; i++) {
    m_el = rclist[i].el_info.el;

    m_child[0] = m_el->child[0];
    m_child[1] = m_el->child[1];

/****************************************************************************/
/* DOF pointers of the master child elements along the bisection line are   */
/* set to nil.                                                              */
/****************************************************************************/
    m_dpv->vec[m_child[0]->dof[m_n+1][m_n0]] = nil;
    m_dpv->vec[m_child[1]->dof[m_n+0][m_n0]] = nil;

    for(j = 0; j < N_EDGES_2D; j++) {
      s_el = (EL *)m_dpv->vec[m_el->dof[m_n+j][m_n0]];
    
      if(s_el && m_el==(EL *)s_dpv->vec[s_el->dof[s_n][s_n0]]) {
	EL_INFO             el_info = {};

	if(j == 2) {
/****************************************************************************/
/* Refine this slave element. Take precautions to preserve the coarse DOFs  */
/* on master and slave elements, since they will be needed for coarsening.  */
/****************************************************************************/
	  s_el->mark   = MAX(s_el->mark, 1);
	  el_info.el   = s_el;
	  el_info.mesh = slave; 

/* Fill active projection on slave rclist for parametric meshes.            */
	  s_proj = rclist[i].el_info.projections[1+j];
	  if(!s_proj) s_proj = rclist[i].el_info.projections[0];
	  el_info.active_projection = el_info.projections[0] = s_proj;

	  AI_refine_fct_1d(&el_info, nil);

/****************************************************************************/
/* Now retrieve the new child elements and set DOF pointers.                */
/****************************************************************************/
	  s_child[0] = s_el->child[0];
	  s_child[1] = s_el->child[1];
	  
	  m_dpv->vec[m_child[0]->dof[m_n+0][m_n0]] = s_child[0];
	  m_dpv->vec[m_child[1]->dof[m_n+1][m_n0]] = s_child[1];

	  s_dpv->vec[s_child[0]->dof[s_n][s_n0]] = m_child[0];
	  s_dpv->vec[s_child[1]->dof[s_n][s_n0]] = m_child[1];
	}
	else if (s_el) {
/****************************************************************************/
/* If we are not on a refinement edge, set the correct master child         */
/* pointer to point to the slave element. Edge 2 of child[1-j] should point */
/* to the current slave element, and vice versa.                            */
/****************************************************************************/
	  m_dpv->vec[m_child[1-j]->dof[m_n+2][m_n0]] = s_el;

	  s_dpv->vec[s_el->dof[s_n][s_n0]] = m_child[1-j];
	}
      }
/****************************************************************************/
/* If there is no slave element on this refinement edge, zero the master    */
/* child pointers.                                                          */
/****************************************************************************/
      else if (!s_el) {
	if(j == 2) {
	  m_dpv->vec[m_child[0]->dof[m_n+0][m_n0]] = nil;
	  m_dpv->vec[m_child[1]->dof[m_n+1][m_n0]] = nil;
	}
	else
	  m_dpv->vec[m_child[1-j]->dof[m_n+2][m_n0]] = nil;
      }
    }
  }

  return;
}

static void master_restrict_2d(DOF_PTR_VEC *m_dpv, RC_LIST_EL *rclist, int mn)
{
  FUNCNAME("master_restrict_2d");
  MESH_MEM_INFO   *m_mem_info = (MESH_MEM_INFO *)
    m_dpv->fe_space->admin->mesh->mem_info;
  int              m_n0 = m_dpv->fe_space->admin->n0_dof[EDGE];
  int              m_n = m_dpv->fe_space->admin->mesh->node[EDGE];
  EL              *m_el, *s_el, *cm_el;
  EL              *s_child[2], *m_child[2];
  int              s_n0, s_n, i, j, n_slaves = m_mem_info->n_slaves;
  MESH            *slave = nil;
  DOF_PTR_VEC     *s_dpv;

/****************************************************************************/
/* Retrieve the slave mesh. Rather unelegant, sorry...                      */
/****************************************************************************/
  for (j = 0; j < n_slaves; j++) {
    slave = m_mem_info->slaves[j];
    if(((MESH_MEM_INFO *)slave->mem_info)->slave_binding == m_dpv) break;
  }
  DEBUG_TEST_EXIT(j < n_slaves, "Slave mesh not found!\n");
  
  s_dpv   = ((MESH_MEM_INFO *)slave->mem_info)->master_binding; 
  s_n0    = s_dpv->fe_space->admin->n0_dof[CENTER];
  s_n     = slave->node[CENTER];

/****************************************************************************/
/* Run over all patch elements. Check if any edges belong to slave elements.*/
/* If so, determine if they should be coarsened together with the master    */
/* elements.                                                                */
/****************************************************************************/

  for(i = 0; i < mn; i++) {
    m_el = rclist[i].el_info.el;

    m_child[0] = m_el->child[0];
    m_child[1] = m_el->child[1];

    for(j = 0; j < N_EDGES_2D; j++) {
      s_el  = (EL *)m_dpv->vec[m_el->dof[m_n+j][m_n0]];

      if(s_el) {
	cm_el = (EL *)s_dpv->vec[s_el->dof[s_n][s_n0]];

	if (cm_el==m_child[0] || cm_el==m_child[1] || cm_el==m_el) {
/****************************************************************************/
/* Set the slave pointer to point to the parent master element.             */
/****************************************************************************/
	  s_dpv->vec[s_el->dof[s_n][s_n0]] = m_el;

	  if (j==2) {
/****************************************************************************/
/* Coarsen this slave element. Old parent DOFs still exist due to the       */
/* preserve_coarse_dofs-setting during refinement.                          */
/****************************************************************************/
	    s_child[0] = s_el->child[0];
	    s_child[1] = s_el->child[1];

	    s_child[0]->mark = -1;
	    s_child[1]->mark = -1;

	    AI_coarse_recursive_1d(slave, s_el);
	  }	  
	}
      }
    }
  }

  return;
}


static void join_elements_recursive_2d(const MESH *master,
				       const MESH *slave,
				       const DOF_ADMIN *m_admin,
				       const DOF_ADMIN *s_admin,
				       const DOF_PTR_VEC *m_dpv,
				       const DOF_PTR_VEC *s_dpv,
				       const int subsimplex,
				       const EL *m_el,
				       const EL *s_el)
{
  FUNCNAME("join_elements_recursive_2d");

  s_dpv->vec[s_el->dof[slave->node[CENTER]]
	     [s_admin->n0_dof[CENTER]]] = (void *)m_el;

  m_dpv->vec[m_el->dof[master->node[EDGE] + subsimplex]
	     [m_admin->n0_dof[EDGE]]] = (void *)s_el;

  if(m_el->child[0]) {
    if (subsimplex == 2) {   /* this is the refinement edge */
      DEBUG_TEST_EXIT(s_el->child[0],
		  "Could not find slave children!\n");
      
      join_elements_recursive_2d(master, slave, m_admin, s_admin, m_dpv, s_dpv,
				 0, m_el->child[0], s_el->child[0]);
      join_elements_recursive_2d(master, slave, m_admin, s_admin, m_dpv, s_dpv,
				 1, m_el->child[1], s_el->child[1]);
    }
    else
      join_elements_recursive_2d(master, slave, m_admin, s_admin, m_dpv, s_dpv,
				 2, m_el->child[1 - subsimplex], s_el);
  }

  return;
}


/****************************************************************************/
/* get_submesh_2d(master, name, binding_method): returns a 1D submesh of    */
/* master based on the information given by the user routine                */
/* binding_method().                                                        */
/****************************************************************************/

static MESH *get_submesh_2d(MESH *master, const char *name,
			    int (*binding_method)
			    (MESH *master, MACRO_EL *el, int face, void *data),
			    void *data)
{
  FUNCNAME("get_submesh_2d");
  MACRO_DATA       s_data = {};
  MESH_MEM_INFO   *m_info, *s_info;
  MESH            *slave = nil;
  const FE_SPACE  *slave_space, *master_space;
  DOF_PTR_VEC     *slave_to_master_binding;
  DOF_PTR_VEC     *master_to_slave_binding;
  int              s_n_dof[N_NODE_TYPES] = {}, m_n_dof[N_NODE_TYPES] = {};
  MACRO_EL        *m_mel, *s_mel;
  const DOF_ADMIN *m_admin, *s_admin;
  int              i, j, k, n, ne = 0, nv = 0, *vert_ind = nil, index;
  S_CHAR           bound;
  char             new_name[1024];

  m_info = ((MESH_MEM_INFO *)master->mem_info);

/****************************************************************************/
/* Count all needed vertices and elements.                                  */
/****************************************************************************/

  s_data.dim = 1;
  s_data.coords = MEM_ALLOC(master->n_vertices, REAL_D);  /* resized later! */

  vert_ind = MEM_ALLOC(master->n_vertices, int);
  for(i = 0; i < master->n_vertices; i++)
    vert_ind[i] = -1;
  
  for(n = 0; n < master->n_macro_el; n++) {
    m_mel = master->macro_els + n;

    for(i = 0; i < N_EDGES_2D; i++)
      if(binding_method(master, m_mel, i, data)) {
	ne++;
	
	for(j = 0; j < N_VERTICES_2D; j++)
	  if(j != i) {
/* Make use of the slave->mem_info->coords vector to get vertex indices.    */
	    index = (m_mel->coord[j] - m_info->coords[0]) / DIM_OF_WORLD;
	    
	    if (vert_ind[index] < 0) {
	      vert_ind[index] = nv;    
	      for(k = 0; k < DIM_OF_WORLD; k++)
		s_data.coords[nv][k] = m_info->coords[index][k];
	      
	      nv++;
	    }
	  }
      }
  }

/****************************************************************************/
/* Allocate the needed amount of macro elements and coordinates.            */
/* Fill element and coordinate information.                                 */
/****************************************************************************/

  TEST_EXIT(nv,"Bad mesh: no vertices counted!\n");
  TEST_EXIT(ne,"Bad mesh: no elements counted!\n");

  s_data.n_total_vertices = nv;
  s_data.n_macro_elements = ne;
  s_data.coords = MEM_REALLOC(s_data.coords, master->n_vertices,
			      nv, REAL_D);
  s_data.mel_vertices = MEM_ALLOC(ne * N_NEIGH_1D, int);
  ne = 0;

  for(n = 0; n < master->n_macro_el; n++) {
    m_mel = master->macro_els + n;

    for(i = 0; i < N_EDGES_2D; i++)
      if(binding_method(master, m_mel, i, data)) {
	for(j = 0; j < N_VERTICES_2D; j++)
	  if(j != i) {
	    index = (m_mel->coord[j] - m_info->coords[0]) / DIM_OF_WORLD;
	    nv = vert_ind[index];
	    
	    s_data.mel_vertices[N_VERTICES_1D * ne + (j + 2 - i) % 3] = nv;  
	  }
	
	ne++;
      }
  }

  compute_neigh_fast(&s_data);  

  /* First assign a default DIRICHLET boundary. */
  dirichlet_boundary(&s_data);

  /* Now correct the boundary values if the master vertex has nonzero type. */

  ne = 0;
  for(ne = n = 0; n < master->n_macro_el; n++) {
    m_mel = master->macro_els + n;

    for(i = 0; i < N_EDGES_2D; i++)
      if(binding_method(master, m_mel, i, data)) {
	for(j = 0; j < N_NEIGH_1D; j++)
	  if((s_data.boundary[N_NEIGH_1D * ne + j] == DIRICHLET)
	     && (bound = m_mel->vertex_bound[(i+2-j)%3]))
	    s_data.boundary[N_NEIGH_1D * ne + j] = bound;

	ne++;
      }
  }

/****************************************************************************/
/* Allocate a submesh.                                                      */
/****************************************************************************/

  if(!name) {
    static int count_2d = 1;

    sprintf(new_name, "Submesh %d of %s", count_2d, master->name);
    name = new_name;

    count_2d++;
  }

  slave = GET_MESH(1, name, &s_data, nil);

/****************************************************************************/
/* Clean up.                                                                */
/****************************************************************************/

  nv = s_data.n_total_vertices;
  ne = s_data.n_macro_elements;

  MEM_FREE(s_data.coords, nv, REAL_D);
  MEM_FREE(s_data.mel_vertices, ne*N_VERTICES_1D, int);
  MEM_FREE(s_data.neigh, ne*N_NEIGH_1D, int);
  MEM_FREE(s_data.boundary, ne*N_NEIGH_1D, S_CHAR); 
  MEM_FREE(vert_ind, master->n_vertices, int);

/****************************************************************************/
/* Fill more slave elements, if the master mesh was already refined.        */
/****************************************************************************/

  get_slave_elements_2d(master, slave, binding_method, data);

/****************************************************************************/
/*  Allocate special FE spaces for the slave.                               */
/****************************************************************************/

  s_n_dof[CENTER] = 1;

  slave_space = get_fe_space(slave, "Center dof fe_space", s_n_dof, nil, 1);

  slave_to_master_binding = get_dof_ptr_vec("Slave - master pointers",
					    slave_space);

/****************************************************************************/
/*  Allocate special FE spaces for master.                                  */
/****************************************************************************/
  
  m_n_dof[EDGE] = 1;
  master_space = get_fe_space(master, "Edge dof fe_space", m_n_dof, nil, 1);

#if ALBERTA_DEBUG == 1
  check_mesh(slave);
#endif

/****************************************************************************/
/* Allocate special DOF_PTR_VECs for both master and slave. These serve to  */
/* help find the corresponding subsimplex to each boundary master simplex   */
/* during refinement and vice versa.                                        */
/****************************************************************************/

  master_to_slave_binding = get_dof_ptr_vec("Master - slave pointers",
					    master_space);

  master_to_slave_binding->refine_interpol = master_interpol_2d;
  master_to_slave_binding->coarse_restrict = master_restrict_2d;

/****************************************************************************/
/* Set the special pointers in the MESH_MEM_INFO components of both master  */
/* and slave grids.                                                         */
/****************************************************************************/

  s_info                  = (MESH_MEM_INFO *)slave->mem_info;
  s_info->master          = master;
  s_info->binding_method  = binding_method;
  s_info->slave_binding   = master_to_slave_binding;
  s_info->master_binding  = slave_to_master_binding;

  m_info->slaves = MEM_REALLOC(m_info->slaves,
			       m_info->n_slaves,
			       m_info->n_slaves + 1,
			       MESH *);
  m_info->slaves[m_info->n_slaves] = slave;

  m_info->n_slaves++;

/****************************************************************************/
/* Set the element pointer vec entries to the correct values.               */
/* This assumes that slave macro elements were allocated in the order given */
/* by the loop below.                                                       */
/****************************************************************************/

  m_admin = master_to_slave_binding->fe_space->admin;
  s_admin = slave_to_master_binding->fe_space->admin;

  FOR_ALL_DOFS(s_admin, slave_to_master_binding->vec[dof] = nil);
  FOR_ALL_DOFS(m_admin, master_to_slave_binding->vec[dof] = nil);

  s_mel = slave->macro_els;
  for(n = 0; n < master->n_macro_el; n++) {
    m_mel = master->macro_els + n;

    for(i = 0; i < N_NEIGH_2D; i++)
      if(binding_method(master, m_mel, i, data)) {
	DEBUG_TEST_EXIT(s_mel,
		    "Ran out of slave macro elements... Wrong meshes?\n");

	/* Here we take care of node projection function transfer. */
	if(m_mel->projection[i+1])
	  s_mel->projection[0] = m_mel->projection[i+1];
	else
	  s_mel->projection[0] = m_mel->projection[0];
	    
	join_elements_recursive_2d(master, slave, m_admin, s_admin,
				   master_to_slave_binding, 
				   slave_to_master_binding,
				   i, m_mel->el, s_mel->el);
	s_mel++;
      }
  }

/****************************************************************************/
/* Now project all new vertices if the mesh was already refined.            */
/****************************************************************************/

  if(slave->n_elements < slave->n_hier_elements)
    AI_post_refine_1d(slave);

  return slave;
}

