/*
 * (C) Copyright 2015-2021, by Alexey Kudinkin and Contributors.
 *
 * JGraphT : a free Java graph-theory library
 *
 * See the CONTRIBUTORS.md file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0, or the
 * GNU Lesser General Public License v2.1 or later
 * which is available at
 * http://www.gnu.org/licenses/old-licenses/lgpl-2.1-standalone.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR LGPL-2.1-or-later
 */
package org.jgrapht.alg.flow;

import org.jgrapht.*;
import org.jgrapht.generate.*;
import org.jgrapht.graph.*;
import org.jgrapht.util.*;

import java.util.*;

import static org.junit.Assert.assertTrue;

public abstract class MaximumFlowMinimumCutAlgorithmTestBase
{

    protected Graph<Integer, DefaultWeightedEdge> constructDirectedGraph(
        int[] tails, int[] heads, double[] capacities, int[] sources, int[] sinks)
    {
        assertTrue(tails.length == heads.length);
        assertTrue(tails.length == capacities.length);

        DirectedWeightedMultigraph<Integer, DefaultWeightedEdge> network =
            new DirectedWeightedMultigraph<>(DefaultWeightedEdge.class);
        int m = tails.length;
        for (int i = 0; i < m; i++)
            Graphs.addEdgeWithVertices(network, tails[i], heads[i], capacities[i]);
        for (int i = 0; i < sources.length; i++) {
            network.addVertex(sources[i]);
            network.addVertex(sinks[i]);
        }
        return network;
    }

    protected Graph<Integer, DefaultWeightedEdge> constructUndirectedGraph(int[][] edges)
    {
        // Construct undirected graph
        SimpleWeightedGraph<Integer, DefaultWeightedEdge> graph =
            new SimpleWeightedGraph<>(DefaultWeightedEdge.class);
        for (int[] edge : edges) // Format: {u,v,weight}
            Graphs.addEdgeWithVertices(graph, edge[0], edge[1], edge[2]);
        return graph;
    }

    /*************** TEST CASES FOR DIRECTED GRAPHS ***************/

    public Graph<Integer, DefaultWeightedEdge> getDirectedN0()
    {
        return constructDirectedGraph(
            new int[] { 1, 2, 3 }, new int[] { 2, 3, 4 }, new double[] { 10, 10, 5 },
            new int[] { 1 }, new int[] { 4 });
    }

    public Graph<Integer, DefaultWeightedEdge> getDirectedN1()
    {
        return constructDirectedGraph(
            new int[] {}, new int[] {}, new double[] {}, new int[] { 1 }, new int[] { 4057218 });
    }

    public Graph<Integer, DefaultWeightedEdge> getDirectedN2()
    {
        return constructDirectedGraph(
            new int[] { 3, 1, 4, 3, 2, 8, 2, 5, 7 }, new int[] { 1, 4, 8, 2, 8, 6, 5, 7, 6 },
            new double[] { 1, 1, 1, 1, 1, 1, 1, 1, 1 }, new int[] { 3 }, new int[] { 6 });
    }

    public Graph<Integer, DefaultWeightedEdge> getDirectedN3()
    {
        return constructDirectedGraph(
            new int[] { 5, 5, 5, 1, 1, 4, 2, 7, 8, 3 }, new int[] { 1, 4, 2, 7, 8, 3, 8, 6, 6, 6 },
            new double[] { 7, 8, 573146, 31337, 1, 1, 1, 1, 2391717, 170239 }, new int[] { 5 },
            new int[] { 6 });
    }

    public Graph<Integer, DefaultWeightedEdge> getDirectedN4()
    {
        return constructDirectedGraph(
            new int[] { 1, 1, 2, 2, 3 }, new int[] { 2, 3, 3, 4, 4 },
            new double[] { 1000000000.0, 1000000000.0, 1.0, 1000000000.0, 1000000000.0 },
            new int[] { 1 }, new int[] { 4 });
    }

    public Graph<Integer, DefaultWeightedEdge> getDirectedN6()
    {
        return constructDirectedGraph(
            new int[] { 46, 27, 44, 49, 11, 22, 17, 37, 12, 7, 12, 17, 26, 35, 4, 14, 16, 27, 2, 8,
                33, 38, 2, 32, 43, 22, 32, 14, 43, 44, 43, 43, 24, 49, 16, 44, 10, 43, 10, 34, 34,
                21, 24, 42, 11, 9, 23, 49, 49, 34, 34, 45, 17, 10, 43, 22, 38, 49, 3, 33, 13, 15, 2,
                48, 20, 18, 29, 36, 13, 37, 5, 8, 33, 47, 47, 18, 24, 22, 42, 48, 17, 34, 42, 8, 38,
                10, 39, 47, 31, 19, 6, 25, 17, 38, 33, 21, 39, 2, 32, 33, 46, 22, 37, 39, 15, 38, 4,
                26, 44, 37, 17, 1, 48, 31, 31, 19, 34, 31, 40, 42, 30, 6, 14, 50, 30, 33, 44, 7, 50,
                32, 43, 26, 43, 4, 28, 41, 7, 25, 1, 49, 27, 15, 38, 24, 9, 46, 45, 49, 6, 17, 18,
                12, 36, 5, 31, 9, 50, 20, 30, 48, 40, 35, 15, 21, 9, 40, 47, 5, 36, 31, 10, 10, 1,
                6, 39, 21, 42, 35, 41, 37, 33, 23, 42, 46, 37, 24, 12, 37, 27, 24, 36, 36, 18, 37,
                50, 5, 50, 17, 7, 24, 4, 30, 41, 47, 7, 4, 44, 9, 29, 13, 48, 35, 19, 2, 46, 46, 39,
                9, 13, 41, 7, 18, 48, 3, 31, 42, 32, 8, 46, 5, 4, 48, 43, 7, 49, 9, 45, 29, 6, 26,
                13, 16, 25, 43, 41, 38, 45, 15, 44, 22, 21, 29, 10, 49, 35, 43, 9, 28, 5, 41, 16,
                34, 27, 23, 1, 22, 29, 42, 50, 24, 23, 43, 16, 20, 33, 35, 33, 30, 9, 39, 4, 7, 20,
                37, 6, 39, 46, 12, 21, 33, 34, 47, 20, 30, 22, 31, 50, 16, 13, 23, 32, 36, 41, 45,
                16, 27, 4, 42, 3, 18, 38, 29, 44, 49, 32, 14, 31, 18, 6, 44, 20, 45, 48, 50, 35, 48,
                48, 15, 26, 2, 36, 36, 42, 1, 5, 27, 48, 44, 41, 12, 16, 16, 43, 33, 44, 42, 35, 37,
                47, 35, 50, 9, 37, 14, 29, 23, 4, 9, 28, 15, 1, 2, 35, 27, 2, 39, 46, 11, 47, 18,
                25, 35, 4, 34, 37, 11, 48, 34, 26, 12, 22, 5, 1, 7, 7, 43, 32, 25, 17, 48, 48, 24,
                32, 41, 28, 7, 16, 18, 16, 16, 21, 30, 15, 39, 2, 26, 31, 27, 38, 1, 3, 14, 14, 19,
                42, 12, 15, 39, 14, 47, 21, 26, 8, 33, 23, 38, 34, 17, 1, 37, 4, 34, 27, 26, 40, 30,
                15, 43, 27, 6, 28, 4, 13, 39, 11, 3, 11, 44, 39, 35, 47, 49, 20, 25, 4, 38, 32, 48,
                13, 32, 31, 10, 8, 46, 39, 19, 35, 23, 46, 33, 22, 12, 21, 22, 36, 3, 38, 32, 14,
                27, 26, 7, 49, 40, 33, 49, 36, 40, 11, 45, 38, 25, 37, 19, 19, 13, 5, 32, 13, 41,
                43, 18, 50, 17, 18, 36, 38, 25, 2, 32, 2, 30, 22, 15, 15, 43, 29, 17, 38, 28, 21, 1,
                12, 27, 2, 30, 8, 33, 13, 26, 27, 33, 12, 18, 14, 14, 50, 17, 9, 12, 40, 19, 46, 29,
                45, 37, 24, 39, 3, 4, 46, 29, 1, 15, 25, 13, 28, 29, 1, 10, 46, 24, 25, 4, 6, 33,
                35, 21, 5, 46, 23, 35, 32, 48, 23, 6, 6, 12, 35, 21, 32, 41, 44, 22, 26, 22, 19, 31,
                11, 1, 16, 21, 16, 40, 6, 42, 32, 44, 24, 30, 13, 50, 6, 44, 16, 46, 47, 3, 17, 6,
                22, 18, 23, 29, 39, 44, 42, 50, 8, 35, 6, 19, 32, 31, 36, 30, 45, 14, 22, 39, 44,
                37, 4, 41, 13, 24, 28, 7, 13, 39, 36, 39, 47, 8, 50, 3, 20, 3, 1, 4, 24, 4, 33, 32,
                34, 8, 50, 36, 28, 41, 14, 50, 42, 1, 29, 16, 38, 12, 21, 46, 8, 29, 34, 27, 48, 32,
                14, 24, 47, 31, 11, 16, 3, 29, 20, 2, 30, 17, 45, 9, 28, 42, 43, 26, 42, 32, 45, 9,
                20, 50, 11, 1, 25, 13, 42, 50, 23, 28, 5, 28, 17, 43, 44, 7, 38, 36, 29, 33, 14, 34,
                38, 22, 32, 29, 43, 14, 32, 29, 9, 5, 36, 11, 8, 13, 40, 28, 43, 49, 37, 37, 7, 9,
                16, 28, 8, 36, 14, 13, 26, 38, 3, 6, 35, 2, 13, 8, 14, 34, 34, 2, 43, 46, 29, 3, 7,
                35, 42, 39, 48, 15, 31, 38, 10, 40, 13, 24, 19, 38, 4, 25, 37, 44, 22, 3, 30, 30,
                46, 12, 22, 47, 50, 26, 34, 6, 26, 37, 50, 27, 10, 10, 36, 25, 14, 21, 15, 47, 25,
                4, 9, 12, 40, 18, 49, 20, 44, 18, 48, 28, 46, 49, 34, 5, 19, 37, 20, 25, 50, 46, 47,
                47, 7, 19, 50, 36, 26, 23, 8, 10, 7, 10, 11, 28, 23, 9, 33, 4, 25, 10, 49, 26, 15,
                40, 39, 14, 5, 50, 35, 37, 42, 15, 25, 45, 40, 31, 24, 21, 14, 19, 37, 26, 44, 4,
                30, 4, 30, 1, 30, 48, 16, 8, 10, 50, 47, 18, 16, 31, 27, 4, 49, 7, 49, 7, 46, 43,
                28, 15, 34, 6, 33, 1, 10, 42, 38, 40, 28, 29, 13, 48, 33, 14, 16, 4, 47, 4, 4, 37,
                5, 21, 18, 31, 32, 46, 11, 5, 18, 30, 34, 16, 15, 6, 38, 2, 11, 38, 4, 16, 10, 39,
                2, 15, 37, 3, 21, 33, 42, 28, 17, 31, 29, 43, 36, 42, 39, 4, 24, 14, 3, 47, 29, 25,
                14, 50, 2, 25, 23, 1, 3, 23, 42, 37, 41, 12, 45, 45, 38, 1, 41, 21, 36, 18, 12, 44,
                5, 9, 8, 44, 40, 30, 12, 17, 45, 21, 16, 12, 18, 35, 13, 17, 45, 49 },
            new int[] { 29, 5, 25, 16, 38, 2, 49, 19, 48, 47, 46, 37, 19, 44, 2, 10, 27, 37, 1, 40,
                36, 36, 9, 3, 19, 29, 45, 44, 11, 18, 45, 10, 11, 28, 25, 36, 49, 30, 45, 35, 39,
                20, 47, 40, 6, 12, 18, 21, 35, 10, 40, 6, 28, 50, 44, 27, 14, 9, 9, 4, 5, 38, 48,
                27, 47, 11, 33, 16, 26, 7, 48, 30, 16, 11, 2, 34, 38, 16, 14, 41, 1, 44, 33, 19, 24,
                26, 48, 9, 44, 42, 30, 18, 16, 33, 41, 48, 23, 4, 11, 30, 48, 30, 8, 37, 50, 28, 35,
                9, 17, 45, 22, 11, 29, 20, 11, 31, 24, 8, 35, 12, 37, 39, 11, 22, 6, 9, 13, 38, 36,
                1, 20, 27, 17, 44, 41, 42, 4, 28, 36, 8, 45, 14, 4, 35, 15, 21, 37, 32, 8, 27, 36,
                8, 19, 2, 14, 27, 11, 4, 49, 50, 24, 24, 49, 2, 13, 38, 32, 26, 23, 48, 3, 33, 30,
                49, 34, 7, 43, 15, 14, 12, 48, 25, 13, 38, 9, 27, 40, 33, 11, 13, 37, 39, 50, 2, 10,
                49, 37, 43, 17, 3, 6, 41, 31, 8, 46, 39, 40, 39, 40, 42, 33, 50, 17, 14, 13, 28, 11,
                6, 16, 11, 23, 25, 4, 2, 35, 48, 33, 22, 45, 1, 13, 44, 7, 19, 38, 43, 11, 35, 27,
                15, 39, 33, 35, 33, 5, 8, 16, 11, 22, 36, 5, 21, 34, 18, 46, 46, 29, 31, 24, 2, 10,
                1, 16, 19, 9, 39, 34, 27, 18, 17, 33, 2, 47, 39, 14, 1, 13, 9, 35, 8, 49, 50, 14,
                34, 13, 15, 10, 14, 3, 25, 7, 21, 5, 32, 45, 24, 38, 23, 3, 7, 19, 28, 33, 17, 4,
                41, 32, 22, 36, 16, 50, 12, 26, 27, 9, 45, 40, 9, 22, 45, 33, 3, 24, 30, 49, 22, 23,
                8, 21, 34, 1, 2, 37, 21, 32, 15, 30, 4, 19, 13, 11, 3, 22, 5, 35, 23, 31, 31, 20,
                41, 46, 16, 49, 50, 5, 34, 45, 30, 23, 4, 5, 29, 30, 43, 26, 29, 4, 5, 42, 20, 7, 8,
                21, 30, 14, 34, 20, 11, 42, 15, 31, 28, 7, 18, 28, 41, 23, 48, 7, 14, 8, 39, 13, 40,
                39, 25, 29, 8, 31, 28, 31, 39, 27, 20, 50, 36, 30, 33, 10, 33, 4, 29, 33, 29, 35,
                29, 21, 49, 15, 7, 19, 22, 43, 31, 42, 6, 38, 24, 42, 11, 18, 16, 40, 46, 43, 3, 31,
                48, 13, 10, 11, 22, 47, 31, 43, 17, 50, 43, 40, 2, 33, 29, 36, 11, 17, 40, 16, 39,
                21, 40, 16, 30, 47, 9, 45, 36, 19, 1, 26, 3, 18, 30, 38, 7, 27, 12, 35, 35, 36, 48,
                16, 26, 48, 50, 15, 14, 6, 31, 10, 49, 44, 10, 48, 34, 41, 4, 8, 4, 20, 25, 35, 47,
                15, 48, 33, 48, 22, 36, 44, 47, 40, 40, 43, 9, 24, 27, 38, 18, 7, 36, 4, 49, 17, 26,
                23, 47, 10, 26, 24, 10, 49, 39, 22, 29, 28, 31, 44, 26, 41, 34, 5, 24, 8, 14, 44,
                18, 25, 38, 39, 16, 43, 42, 10, 29, 26, 44, 18, 45, 20, 31, 3, 29, 20, 28, 2, 12, 9,
                49, 21, 7, 8, 28, 44, 21, 6, 23, 37, 20, 7, 9, 42, 18, 31, 42, 3, 44, 28, 32, 29,
                47, 19, 33, 24, 36, 25, 26, 49, 5, 4, 18, 17, 43, 12, 42, 17, 49, 24, 35, 10, 48,
                23, 13, 12, 17, 21, 47, 47, 4, 15, 13, 42, 32, 5, 50, 4, 26, 50, 47, 21, 46, 32, 28,
                11, 10, 31, 6, 3, 10, 33, 28, 38, 16, 35, 39, 28, 19, 35, 30, 12, 5, 9, 26, 2, 36,
                31, 34, 8, 5, 47, 43, 43, 16, 3, 50, 1, 49, 9, 11, 27, 32, 31, 30, 28, 6, 32, 29,
                42, 26, 30, 47, 13, 44, 26, 15, 49, 18, 15, 35, 40, 34, 13, 30, 28, 40, 31, 12, 46,
                49, 34, 25, 17, 46, 16, 46, 36, 39, 5, 37, 4, 25, 47, 9, 34, 21, 3, 12, 22, 1, 1, 6,
                31, 13, 7, 49, 31, 7, 27, 47, 8, 37, 24, 45, 11, 30, 41, 45, 40, 41, 43, 26, 1, 5,
                12, 33, 46, 21, 32, 1, 37, 25, 22, 45, 18, 10, 38, 12, 27, 5, 26, 32, 38, 12, 25,
                41, 3, 42, 2, 33, 31, 6, 4, 29, 25, 26, 29, 48, 15, 37, 33, 41, 16, 29, 47, 1, 13,
                5, 44, 28, 3, 14, 17, 23, 43, 26, 30, 35, 1, 2, 24, 39, 49, 40, 5, 50, 41, 12, 42,
                37, 33, 19, 33, 49, 44, 28, 30, 37, 3, 8, 35, 6, 23, 11, 50, 20, 50, 21, 40, 36, 17,
                38, 41, 5, 45, 40, 33, 27, 7, 8, 44, 35, 37, 35, 50, 4, 6, 47, 12, 44, 39, 8, 47,
                25, 4, 23, 31, 48, 16, 39, 38, 43, 5, 23, 44, 41, 1, 37, 16, 1, 47, 35, 16, 9, 4, 3,
                46, 6, 27, 47, 45, 42, 7, 12, 45, 22, 20, 10, 38, 19, 15, 37, 34, 15, 31, 8, 44, 1,
                18, 31, 34, 45, 20, 17, 49, 27, 50, 16, 22, 18, 11, 6, 38, 20, 37, 42, 36, 25, 9,
                39, 15, 34, 28, 11, 12, 45, 32, 29, 13, 8, 19, 21, 12, 16, 31, 5, 14, 37, 4, 27, 5,
                5, 32, 44, 11, 29, 9, 1, 23, 18, 31, 36, 47, 41, 47, 43, 30, 41, 33, 5, 2, 32, 3, 9,
                8, 39, 17, 44, 29, 13, 32, 50, 13, 35, 10, 14, 32, 35, 8, 14, 17, 40, 2, 33, 29, 18,
                36, 46, 13, 50, 10, 11, 22, 47, 21, 1, 21, 48, 12, 50, 39, 37 },
            new double[] { 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0 },
            new int[] { 1 }, new int[] { 50 });
    }

    public Graph<Integer, DefaultWeightedEdge> getDirectedN7()
    {
        return constructDirectedGraph(
            new int[] { 3, 42, 11, 19, 16, 45, 18, 35, 38, 7, 15, 12, 33, 21, 47, 12, 40, 25, 29,
                41, 9, 40, 46, 19, 10, 6, 41, 3, 10, 48, 23, 38, 9, 50, 11, 48, 7, 2, 36, 48, 40, 7,
                10, 13, 8, 50, 34, 12, 24, 15, 16, 31, 15, 29, 10, 27, 42, 46, 47, 39, 30, 22, 27,
                28, 32, 44, 15, 15, 23, 6, 11, 34, 14, 24, 5, 9, 29, 34, 42, 10, 41, 50, 40, 42, 34,
                36, 25, 32, 40, 29, 49, 41, 34, 30, 33, 26, 18, 28, 40, 39, 14, 7, 20, 14, 12, 44,
                13, 11, 40, 14, 43, 47, 28, 1, 47, 5, 36, 35, 15, 1, 48, 5, 10, 6, 22, 41, 35, 20,
                40, 41, 22, 22, 37, 37, 50, 42, 26, 6, 11, 44, 27, 16, 43, 50, 4, 13, 25, 43, 18,
                10, 45, 41, 43, 49, 2, 38, 50, 39, 42, 29, 36, 3, 22, 37, 8, 16, 50, 26, 45, 5, 5,
                38, 2, 6, 23, 44, 24, 43, 2, 26, 30, 2, 46, 46, 25, 12, 47, 46, 38, 35, 34, 5, 2,
                13, 18, 42, 5, 25, 29, 29, 27, 5, 18, 36, 13, 13, 41, 19, 28, 20, 9, 49, 12, 41, 14,
                45, 22, 2, 15, 24, 44, 50, 7, 4, 2, 23, 49, 47, 40, 16, 21, 38, 18, 19, 46, 26, 47,
                42, 30, 29, 2, 37, 47, 34, 39, 12, 50, 27, 20, 36, 24, 11, 32, 35, 48, 27, 29, 44,
                12, 46, 7, 20, 48, 39, 39, 9, 20, 42, 35, 33, 41, 41, 13, 45, 12, 14, 21, 36, 25,
                20, 8, 31, 27, 40, 34, 23, 1, 10, 14, 19, 47, 5, 40, 46, 36, 10, 32, 37, 9, 10, 8,
                26, 20, 16, 15, 11, 43, 29, 16, 1, 23, 13, 5, 9, 26, 8, 26, 26, 37, 33, 20, 17, 22,
                12, 26, 23, 39, 37, 50, 2, 41, 37, 12, 35, 45, 27, 9, 14, 3, 21, 23, 35, 39, 35, 6,
                24, 48, 3, 1, 46, 28, 6, 50, 12, 27, 35, 21, 9, 15, 17, 15, 44, 30, 28, 21, 17, 19,
                25, 4, 49, 11, 49, 22, 4, 31, 15, 31, 29, 18, 26, 46, 13, 24, 32, 5, 9, 1, 5, 13,
                24, 31, 50, 35, 11, 36, 3, 41, 38, 41, 8, 37, 25, 8, 16, 45, 23, 16, 8, 9, 40, 7,
                30, 50, 26, 39, 9, 33, 1, 48, 49, 37, 49, 7, 29, 17, 27, 5, 28, 31, 40, 40, 18, 39,
                3, 34, 21, 20, 34, 38, 36, 42, 24, 44, 29, 28, 38, 40, 10, 46, 29, 9, 11, 35, 10,
                38, 11, 2, 43, 4, 26, 6, 32, 31, 45, 16, 46, 21, 14, 18, 43, 2, 28, 16, 30, 5, 19,
                7, 41, 45, 35, 3, 48, 9, 43, 17, 37, 4, 33, 27, 45, 32, 34, 32, 12, 17, 16, 33, 4,
                36, 13, 42, 36, 9, 1, 35, 10, 2, 48, 29, 21, 27, 16, 48, 38, 18, 32, 46, 15, 2, 12,
                36, 31, 2, 20, 37, 50, 28, 32, 7, 9, 33, 8, 47, 22, 21, 17, 20, 29, 22, 16, 38, 22,
                39, 42, 32, 36, 13, 46, 45, 47, 9, 10, 30, 24, 24, 26, 34, 41, 5, 12, 47, 15, 36,
                16, 19, 9, 42, 50, 23, 26, 42, 1, 39, 16, 49, 39, 28, 36, 16, 6, 37, 7, 4, 22, 50,
                41, 34, 12, 45, 24, 19, 41, 10, 11, 43, 19, 31, 45, 21, 32, 19, 40, 38, 4, 45, 30,
                13, 14, 1, 14, 7, 5, 7, 17, 15, 15, 10, 12, 25, 31, 20, 10, 28, 10, 24, 50, 21, 18,
                41, 24, 28, 3, 36, 17, 29, 17, 32, 43, 8, 12, 38, 26, 18, 22, 50, 16, 3, 33, 14, 49,
                30, 50, 27, 47, 34, 34, 37, 30, 42, 2, 47, 18, 31, 12, 16, 26, 44, 33, 47, 14, 42,
                29, 8, 20, 13, 45, 12, 1, 29, 24, 3, 29, 44, 7, 49, 40, 33, 4, 4, 37, 1, 48, 26, 2,
                39, 10, 45, 23, 15, 28, 21, 39, 3, 36, 1, 21, 23, 26, 12, 24, 3, 47, 4, 46, 11, 29,
                10, 2, 3, 22, 26, 30, 35, 44, 1, 34, 9, 12, 7, 44, 14, 1, 25, 36, 32, 24, 11, 16,
                31, 46, 2, 33, 44, 38, 19, 18, 48, 35, 29, 47, 39, 31, 21, 12, 30, 15, 37, 18, 43,
                6, 19, 49, 7, 24, 46, 38, 6, 23, 3, 30, 39, 28, 14, 41, 48, 23, 18, 36, 18, 3, 11,
                34, 48, 5, 15, 49, 12, 37, 47, 24, 38, 27, 5, 27, 48, 17, 43, 39, 2, 23, 17, 20, 11,
                1, 25, 35, 44, 10, 28, 47, 40, 13, 6, 16, 5, 46, 47, 10, 39, 42, 3, 42, 26, 6, 1, 2,
                48, 16, 14, 6, 6, 33, 27, 42, 32, 43, 13, 32, 10, 30, 32, 25, 4, 24, 15, 43, 33, 2,
                14, 43, 2, 46, 4, 48, 33, 8, 47, 44, 43, 46, 24, 23, 17, 32, 38, 20, 9, 15, 47, 42,
                43, 50, 34, 4, 2, 43, 17, 40, 27, 3, 43, 33, 38, 2, 22, 33, 27, 9, 37, 45, 9, 13,
                22, 10, 16, 25, 9, 19, 35, 43, 47, 10, 4, 15, 13, 30, 34, 43, 28, 34, 48, 28, 11,
                33, 28, 19, 31, 5, 2, 32, 48, 49, 36, 42, 50, 41, 6, 39, 18, 12, 21, 9, 50, 15, 38,
                24, 33, 22, 20, 12, 30, 20, 13, 16, 29, 3, 25, 21, 49, 22, 30, 44, 41, 21, 39, 17,
                39, 35, 43, 1, 27, 20, 30, 29, 24, 15, 41, 42, 37, 22, 15, 18, 39, 49, 18, 45, 43,
                14, 25, 30, 25, 33, 47, 7, 46, 24, 40, 9, 33, 7, 45, 33, 23, 21, 44, 21, 14, 4, 22,
                43, 49, 28, 20, 17, 6, 47, 46, 34, 4, 3, 46, 28, 36, 5, 34, 38, 5, 23, 10, 36, 30,
                19, 32, 13, 35, 5, 19, 13, 43, 8, 28, 17, 11, 45, 13, 1, 3, 47, 20, 12, 46, 30, 15,
                17, 2, 42, 48, 37, 33, 7, 35, 13, 32, 36, 18, 32, 32, 21, 49, 38, 20, 21, 43, 25,
                23, 15, 48, 14, 3, 8, 45, 12, 24, 32, 34, 26, 49, 46, 36, 11, 8, 44, 50, 18, 30, 19,
                26, 42, 14, 22, 16, 39, 43, 2, 40, 12, 34, 4, 22, 9, 44, 23, 46, 37, 31, 9, 6, 6,
                24, 27, 45, 9, 44, 18, 32, 35, 7, 12, 45, 6, 18, 7, 29, 22, 23, 1, 34, 26, 38, 15,
                25, 12, 36, 46, 21, 49, 30, 19, 24, 34, 41, 20, 12, 19, 13, 25, 18, 19, 1, 18, 9, 4,
                39, 16, 41, 36, 33, 32, 12, 20, 47, 26, 17, 30, 8, 50, 43, 31, 16, 25, 12, 34, 29,
                37, 35, 33, 45, 47, 7, 40, 19, 15, 16, 34, 45, 28, 33, 22, 10, 49, 31, 30, 29, 21,
                29, 26, 44, 17, 42, 28, 20, 41, 29, 26, 28, 9, 15, 18, 1, 40, 27, 15, 1, 50, 1, 4,
                11, 28, 42, 6, 3, 39, 29, 44, 4, 44, 10, 35, 23, 49, 9, 17, 34, 19, 33, 23, 5, 48,
                24, 10, 8, 41, 47, 11, 7, 35, 28, 23, 37, 25, 46, 8, 38, 32, 35, 42, 28, 18, 1, 13,
                21, 26, 46, 25, 37, 36, 19, 8, 2, 32, 43, 19, 50, 34, 50, 30, 19, 24, 5, 44, 39, 9,
                10, 39, 47, 7, 4, 40, 40, 13, 38, 11, 33, 24, 23, 20, 40, 34, 25, 22, 24, 31, 43,
                14, 10, 18, 36, 27, 29, 20, 48, 44, 3, 46, 22, 27, 15, 19, 26, 18, 50, 14, 37, 26,
                14, 22, 4, 37, 21, 23, 38, 20, 9, 46, 32, 48, 37, 12, 47, 30, 39, 28, 9, 28, 6, 39,
                32, 5, 50, 27, 50, 42, 32, 36, 35, 30, 45, 24, 16, 11, 11, 36, 18, 21, 4, 23, 29,
                19, 1, 21, 31, 27, 44, 40, 17, 26, 13, 13, 34, 26, 27, 7, 20, 26, 12, 14, 18, 45,
                29, 43, 44, 14, 12, 1, 7, 40, 36, 32, 13, 43, 33, 25, 7, 24, 39, 35, 7, 47, 25, 14,
                27, 41, 50, 45, 25, 21, 18, 43, 3, 11, 11, 46, 13, 37, 49, 1, 37, 38, 8, 7, 40, 14,
                39, 8, 1, 40, 14, 5, 43, 24, 37, 45, 47, 34, 37, 16, 37, 21, 38, 21, 20, 13, 22, 50,
                21, 29, 19, 11, 27, 18, 28, 35, 1, 17, 48, 33, 40, 38, 43, 49, 10, 18, 50, 37, 15,
                1, 11, 40, 7, 13, 7, 4, 26, 28, 4, 35, 27, 40, 30, 21, 3, 8, 38, 45, 5, 35, 41, 31,
                37, 15 },
            new int[] { 30, 47, 2, 22, 17, 34, 33, 23, 11, 14, 34, 13, 9, 48, 23, 10, 1, 22, 40, 28,
                42, 45, 48, 9, 18, 27, 22, 6, 22, 1, 11, 17, 49, 22, 42, 43, 32, 50, 30, 15, 32, 18,
                19, 26, 48, 36, 6, 24, 32, 40, 47, 44, 43, 21, 13, 16, 46, 11, 35, 15, 20, 13, 36,
                26, 4, 13, 49, 47, 6, 21, 5, 19, 9, 3, 4, 33, 17, 12, 41, 48, 15, 8, 10, 19, 8, 9,
                11, 9, 11, 45, 9, 29, 3, 11, 35, 15, 42, 32, 23, 7, 46, 23, 26, 1, 8, 19, 35, 40,
                16, 13, 6, 18, 30, 25, 20, 8, 42, 46, 3, 49, 26, 15, 1, 12, 23, 33, 11, 25, 4, 49,
                2, 10, 38, 36, 3, 28, 38, 8, 49, 45, 49, 14, 12, 11, 5, 19, 32, 17, 5, 14, 24, 18,
                47, 28, 10, 34, 40, 2, 34, 13, 20, 27, 26, 40, 46, 31, 48, 36, 46, 32, 13, 4, 24, 7,
                12, 43, 22, 7, 40, 41, 33, 3, 27, 10, 10, 26, 40, 35, 7, 48, 45, 44, 29, 20, 48, 25,
                7, 4, 37, 47, 32, 1, 21, 15, 28, 44, 19, 2, 35, 4, 2, 21, 11, 35, 3, 8, 25, 36, 35,
                19, 24, 38, 11, 45, 49, 50, 16, 34, 22, 20, 5, 15, 22, 4, 3, 30, 22, 21, 35, 14, 16,
                32, 24, 23, 5, 4, 33, 40, 44, 33, 34, 15, 1, 18, 32, 5, 35, 7, 38, 4, 24, 16, 9, 48,
                1, 27, 47, 3, 17, 40, 27, 1, 24, 48, 43, 32, 20, 8, 43, 23, 50, 29, 20, 19, 30, 30,
                18, 11, 47, 36, 5, 6, 30, 29, 22, 35, 50, 27, 11, 38, 49, 48, 43, 48, 2, 12, 48, 2,
                50, 12, 17, 47, 27, 7, 6, 40, 19, 27, 29, 32, 14, 3, 6, 2, 49, 4, 10, 41, 18, 48,
                14, 4, 50, 36, 9, 19, 37, 45, 14, 43, 21, 12, 6, 29, 25, 45, 20, 44, 38, 41, 29, 37,
                21, 1, 47, 28, 28, 17, 9, 49, 45, 47, 4, 1, 41, 19, 28, 20, 44, 20, 8, 32, 20, 28,
                5, 22, 22, 12, 31, 11, 45, 9, 39, 24, 10, 12, 4, 46, 30, 31, 15, 14, 7, 7, 21, 50,
                32, 6, 13, 42, 22, 8, 7, 19, 30, 39, 22, 17, 41, 20, 5, 15, 30, 17, 41, 50, 48, 28,
                21, 3, 2, 48, 10, 50, 14, 28, 50, 16, 25, 36, 44, 49, 49, 32, 35, 36, 37, 25, 22,
                29, 14, 49, 29, 26, 25, 3, 38, 47, 32, 8, 30, 37, 20, 27, 16, 24, 43, 26, 29, 13,
                34, 41, 24, 22, 29, 23, 3, 16, 38, 30, 30, 33, 34, 13, 24, 42, 8, 8, 28, 39, 15, 13,
                4, 33, 23, 10, 6, 45, 24, 36, 16, 38, 3, 46, 5, 9, 1, 27, 3, 34, 31, 18, 44, 24, 30,
                25, 15, 40, 27, 24, 45, 6, 17, 35, 30, 23, 37, 24, 7, 7, 39, 10, 9, 7, 24, 27, 21,
                43, 26, 39, 46, 33, 37, 45, 17, 34, 24, 32, 11, 26, 2, 47, 11, 17, 19, 47, 11, 6,
                19, 10, 38, 34, 34, 37, 28, 9, 46, 42, 34, 36, 46, 1, 12, 3, 34, 29, 16, 42, 32, 20,
                15, 15, 27, 25, 13, 42, 43, 24, 30, 31, 42, 20, 14, 21, 19, 45, 27, 48, 17, 14, 33,
                25, 39, 34, 46, 39, 3, 4, 29, 2, 11, 41, 24, 16, 50, 48, 16, 8, 29, 12, 48, 45, 12,
                46, 17, 13, 29, 27, 16, 19, 38, 37, 39, 26, 11, 13, 39, 17, 11, 36, 6, 34, 46, 9,
                10, 50, 35, 16, 44, 18, 9, 22, 45, 39, 49, 8, 8, 43, 29, 6, 28, 44, 26, 31, 2, 47,
                18, 16, 4, 19, 38, 32, 11, 21, 33, 2, 36, 21, 48, 42, 10, 49, 43, 1, 50, 49, 41, 32,
                48, 10, 48, 28, 46, 29, 38, 21, 5, 50, 13, 7, 45, 7, 39, 26, 9, 12, 7, 25, 16, 11,
                23, 33, 26, 27, 3, 7, 31, 28, 32, 46, 20, 32, 2, 36, 39, 34, 50, 1, 9, 37, 7, 31, 1,
                16, 37, 44, 36, 49, 49, 26, 26, 7, 47, 25, 27, 27, 47, 24, 41, 47, 35, 12, 2, 30,
                23, 11, 28, 44, 5, 25, 22, 26, 11, 44, 12, 8, 20, 43, 4, 28, 16, 50, 36, 1, 44, 48,
                46, 13, 43, 8, 17, 4, 36, 3, 42, 14, 29, 50, 20, 40, 1, 17, 14, 48, 6, 29, 20, 26,
                42, 7, 41, 50, 20, 30, 8, 29, 9, 18, 21, 10, 42, 31, 29, 6, 46, 6, 34, 9, 40, 12,
                42, 14, 15, 1, 3, 17, 41, 4, 46, 5, 11, 23, 50, 29, 50, 46, 33, 42, 22, 50, 23, 38,
                33, 35, 5, 11, 15, 18, 40, 49, 20, 34, 11, 17, 7, 13, 14, 31, 41, 1, 5, 9, 15, 16,
                2, 6, 2, 40, 3, 1, 33, 37, 12, 34, 39, 45, 40, 50, 30, 12, 10, 6, 24, 1, 47, 20, 27,
                25, 3, 28, 15, 4, 31, 11, 41, 29, 17, 21, 15, 29, 16, 21, 33, 16, 5, 12, 26, 43, 14,
                40, 29, 24, 40, 26, 10, 3, 42, 41, 50, 4, 29, 2, 18, 15, 37, 32, 33, 36, 34, 30, 31,
                42, 17, 1, 31, 31, 26, 14, 9, 14, 37, 13, 41, 37, 42, 2, 3, 32, 13, 28, 50, 30, 39,
                39, 20, 36, 6, 44, 24, 3, 46, 39, 10, 32, 50, 24, 16, 4, 21, 26, 44, 32, 5, 41, 12,
                46, 24, 37, 50, 17, 18, 46, 19, 30, 35, 6, 38, 25, 26, 23, 1, 39, 39, 19, 33, 23, 3,
                6, 41, 50, 20, 11, 42, 4, 8, 15, 15, 12, 34, 29, 41, 45, 35, 10, 48, 43, 28, 5, 43,
                1, 30, 5, 28, 29, 36, 10, 25, 16, 15, 38, 36, 29, 14, 23, 42, 11, 37, 46, 37, 10,
                24, 33, 27, 32, 31, 43, 24, 11, 48, 17, 24, 14, 48, 38, 17, 40, 11, 12, 24, 46, 19,
                25, 21, 43, 10, 15, 37, 6, 17, 23, 18, 26, 19, 28, 24, 12, 6, 10, 46, 29, 39, 42, 4,
                11, 43, 31, 43, 5, 1, 6, 3, 18, 37, 3, 6, 26, 40, 8, 39, 49, 49, 38, 42, 42, 27, 9,
                11, 18, 15, 20, 5, 15, 50, 19, 21, 28, 47, 4, 30, 22, 16, 46, 30, 12, 12, 40, 44,
                16, 1, 3, 50, 9, 4, 14, 27, 31, 31, 37, 47, 28, 30, 23, 44, 20, 46, 39, 17, 45, 40,
                13, 47, 39, 25, 14, 1, 38, 2, 25, 35, 37, 14, 33, 40, 13, 17, 48, 13, 25, 37, 17,
                19, 10, 43, 10, 13, 10, 22, 50, 15, 23, 5, 41, 11, 46, 26, 47, 13, 26, 13, 2, 8, 14,
                41, 48, 5, 20, 7, 9, 3, 31, 45, 25, 38, 47, 6, 33, 10, 26, 36, 13, 33, 4, 3, 24, 23,
                5, 29, 6, 46, 17, 2, 39, 17, 15, 13, 42, 26, 18, 6, 9, 42, 21, 36, 3, 38, 31, 5, 2,
                7, 27, 22, 48, 21, 46, 8, 34, 37, 19, 41, 43, 16, 20, 16, 13, 15, 25, 12, 22, 39,
                19, 49, 5, 5, 47, 8, 32, 44, 28, 44, 29, 33, 42, 12, 13, 8, 23, 21, 9, 38, 45, 14,
                38, 5, 32, 17, 18, 17, 20, 40, 42, 28, 19, 6, 12, 3, 35, 39, 39, 15, 49, 1, 9, 22,
                38, 36, 42, 18, 29, 24, 38, 43, 12, 36, 43, 42, 47, 2, 35, 16, 11, 18, 35, 5, 7, 36,
                4, 42, 40, 12, 23, 39, 4, 34, 34, 6, 40, 48, 13, 30, 35, 44, 22, 48, 2, 31, 34, 14,
                47, 9, 22, 44, 49, 24, 3, 38, 3, 39, 22, 20, 41, 1, 50, 29, 26, 41, 7, 26, 42, 28,
                22, 3, 35, 19, 1, 32, 29, 7, 27, 34, 35, 1, 13, 11, 49, 18, 23, 36, 48, 49, 43, 9,
                47, 21, 5, 6, 21, 15, 38, 39, 44, 17, 28, 30, 36, 30, 8, 41, 46, 5, 50, 49, 27, 45,
                48, 32, 34, 34, 45, 37, 49, 37, 16, 24, 44, 9, 8, 27, 24, 38, 28, 14, 25, 18, 6, 37,
                33, 3, 12, 36, 16, 24, 17, 4, 41, 48, 37, 11, 38, 45, 26, 35, 10, 36, 33, 5, 37, 25,
                13, 32, 50, 19, 22, 5, 34, 18, 44, 6, 38, 12, 26, 34, 13, 11, 47, 38, 31, 31, 33,
                31, 2, 31, 44, 9, 48, 4, 14, 32, 12, 40, 8, 27, 20, 25, 36, 10, 37, 11, 26, 24, 40,
                4, 32, 25, 23, 49, 43, 38, 39, 9, 5 },
            new double[] { 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0 },
            new int[] { 1 }, new int[] { 50 });
    }

    public Graph<Integer, DefaultWeightedEdge> getDirectedN8()
    {
        return constructDirectedGraph(
            new int[] { 0, 0, 1, 1, 2, 2, 3, 3, 4, 4 }, new int[] { 1, 2, 2, 3, 1, 4, 2, 5, 3, 5 },
            new double[] { 16, 13, 10, 12, 4, 14, 9, 20, 7, 4 }, new int[] { 0 }, new int[] { 5 });
    }

    public Graph<Integer, DefaultWeightedEdge> getDirectedN9()
    {
        return constructDirectedGraph(
            new int[] { 0, 0, 0, 1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 5, 5, 6, 6, 7, 7 },
            new int[] { 1, 2, 3, 5, 6, 2, 6, 4, 3, 2, 4, 7, 7, 6, 8, 4, 8, 6, 8 },
            new double[] { 12, 8, 11, 7, 6, 5, 9, 4, 2, 13, 6, 12, 7, 4, 3, 15, 9, 8, 10 },
            new int[] { 0 }, new int[] { 8 });
    }

    public Graph<Integer, DefaultWeightedEdge> getDirectedN10()
    {
        return constructDirectedGraph(
            new int[] { 82, 75, 26, 4, 43, 89, 54, 42, 48, 26, 42, 66, 76, 80, 93, 88, 13, 17, 37,
                83, 90, 54, 81, 29, 8, 29, 4, 90, 6, 66, 32, 49, 29, 49, 10, 74, 46, 35, 6, 43, 20,
                63, 77, 10, 29, 22, 15, 39, 91, 76, 70, 38, 72, 45, 81, 6, 66, 11, 97, 27, 72, 17,
                70, 76, 31, 94, 1, 41, 17, 59, 8, 26, 97, 76, 92, 5, 67, 83, 82, 64, 92, 15, 42, 72,
                100, 74, 31, 26, 95, 58, 72, 73, 20, 3, 82, 4, 71, 70, 42, 35, 12, 4, 95, 76, 33,
                100, 3, 37, 57, 42, 43, 45, 94, 11, 4, 32, 19, 69, 1, 34, 25, 92, 24, 81, 82, 94,
                12, 44, 35, 30, 76, 37, 34, 7, 91, 90, 5, 63, 95, 71, 88, 76, 33, 29, 34, 35, 43,
                15, 5, 33, 92, 14, 93, 50, 47, 82, 4, 35, 25, 54, 3, 81, 38, 60, 72, 70, 99, 71, 49,
                83, 85, 94, 15, 68, 40, 1, 59, 20, 34, 100, 6, 40, 25, 6, 88, 21, 19, 88, 6, 66, 18,
                91, 56, 55, 63, 44, 75, 5, 6, 77, 50, 39, 74, 24, 82, 92, 38, 58, 66, 18, 48, 85,
                75, 75, 84, 22, 53, 91, 46, 94, 43, 48, 43, 29, 99, 52, 96, 26, 47, 30, 75, 36, 57,
                25, 83, 63, 27, 99, 3, 31, 95, 83, 31, 23, 7, 37, 43, 81, 74, 57, 67, 72, 93, 98,
                69, 81, 10, 93, 70, 75, 60, 60, 4, 30, 85, 73, 91, 39, 14, 98, 2, 72, 51, 12, 54,
                90, 87, 27, 16, 28, 29, 93, 36, 91, 73, 27, 98, 91, 100, 16, 86, 47, 51, 70, 80, 28,
                71, 78, 82, 31, 37, 94, 76, 63, 90, 39, 24, 11, 27, 95, 66, 35, 25, 69, 63, 70, 61,
                72, 16, 47, 28, 63, 71, 95, 49, 64, 6, 13, 70, 3, 10, 26, 96, 83, 85, 34, 38, 35,
                36, 56, 89, 54, 53, 61, 79, 11, 61, 73, 78, 73, 86, 44, 58, 36, 66, 94, 72, 69, 62,
                63, 2, 69, 40, 75, 93, 58, 20, 58, 24, 86, 5, 72, 30, 26, 66, 37, 90, 86, 20, 36,
                87, 27, 60, 28, 10, 3, 44, 67, 87, 12, 84, 15, 45, 83, 39, 87, 31, 38, 46, 82, 91,
                98, 65, 39, 47, 17, 27, 59, 66, 32, 49, 30, 54, 85, 83, 34, 25, 62, 14, 58, 11, 48,
                24, 23, 85, 85, 82, 25, 53, 13, 80, 3, 83, 48, 7, 10, 65, 66, 30, 87, 53, 46, 28, 7,
                98, 77, 16, 9, 87, 13, 64, 46, 63, 19, 99, 4, 52, 28, 40, 71, 86, 51, 12, 64, 37,
                21, 1, 13, 52, 53, 100, 6, 59, 28, 93, 67, 43, 38, 47, 45, 16, 65, 38, 13, 12, 41,
                2, 94, 40, 91, 35, 96, 14, 99, 18, 93, 80, 62, 70, 51, 24, 8, 72, 57, 11, 94, 59,
                45, 45, 64, 36, 35, 100, 13, 78, 56, 97, 46, 42, 32, 53, 25, 55, 100, 33, 63, 71,
                16, 62, 10, 87, 53, 52, 67, 82, 23, 95, 53, 55, 61, 48, 63, 96, 65, 82, 69, 6, 38,
                55, 33, 22, 37, 93, 18, 40, 84, 71, 51, 72, 18, 41, 24, 6, 47, 78, 46, 63, 45, 75,
                74, 21, 52, 58, 30, 16, 93, 93, 43, 63, 37, 98, 6, 31, 96, 87, 64, 71, 14, 32, 100,
                77, 29, 93, 52, 62, 99, 21, 55, 86, 92, 49, 23, 83, 75, 42, 88, 23, 46, 31, 33, 1,
                54, 71, 45, 61, 89, 10, 52, 21, 59, 36, 90, 41, 13, 86, 51, 70, 78, 35, 48, 22, 43,
                7, 3, 99, 13, 86, 36, 100, 34, 75, 83, 23, 65, 30, 5, 57, 53, 98, 23, 97, 58, 49,
                56, 91, 48, 42, 76, 33, 47, 78, 94, 55, 74, 64, 31, 18, 14, 49, 94, 47, 27, 38, 87,
                69, 39, 27, 86, 90, 12, 36, 68, 13, 2, 72, 27, 23, 79, 71, 46, 69, 28, 73, 65, 35,
                19, 51, 30, 91, 40, 38, 36, 2, 2, 28, 17, 17, 79, 18, 67, 2, 22, 71, 20, 82, 59, 69,
                55, 66, 97, 9, 37, 82, 75, 33, 74, 10, 4, 60, 71, 14, 41, 32, 30, 60, 36, 12, 64,
                37, 68, 27, 22, 80, 78, 83, 20, 93, 42, 12, 94, 94, 16, 55, 93, 32, 86, 66, 92, 41,
                92, 95, 51, 48, 16, 91, 68, 81, 9, 96, 7, 70, 45, 35, 61, 6, 16, 98, 74, 43, 4, 14,
                92, 24, 75, 15, 29, 81, 94, 97, 84, 19, 38, 29, 59, 87, 16, 56, 26, 76, 71, 30, 28,
                71, 82, 3, 46, 81, 59, 82, 39, 51, 94, 6, 5, 77, 81, 18, 10, 20, 83, 90, 74, 31, 58,
                88, 75, 98, 91, 80, 28, 77, 19, 12, 94, 24, 61, 78, 13, 91, 27, 98, 92, 5, 97, 63,
                26, 12, 9, 19, 79, 61, 39, 43, 62, 62, 53, 83, 19, 71, 100, 18, 44, 37, 42, 34, 88,
                39, 83, 11, 22, 32, 94, 81, 19, 75, 75, 66, 33, 76, 20, 54, 54, 27, 27, 94, 67, 44,
                72, 51, 19, 54, 91, 76, 29, 64, 63, 60, 84, 77, 87, 22, 96, 9, 43, 36, 93, 15, 2,
                90, 91, 95, 18, 39, 77, 60, 7, 90, 68, 71, 100, 69, 16, 35, 88, 4, 76, 79, 26, 45,
                30, 26, 36, 30, 47, 44, 15, 29, 82, 41, 38, 70, 36, 70, 13, 87, 17, 15, 53, 24, 49,
                89, 61, 23, 91, 46, 66, 46, 10, 37, 11, 79, 82, 87, 9, 25, 12, 91, 8, 34, 8, 72, 74,
                52, 92, 71, 76, 33, 88, 68, 95, 83, 94, 28, 57, 18, 21, 68, 79, 20, 55, 48, 14, 19,
                38, 94, 27, 8, 80, 2, 50, 99 },
            new int[] { 18, 94, 91, 37, 92, 18, 98, 61, 84, 15, 72, 22, 100, 11, 37, 69, 2, 43, 92,
                15, 65, 40, 79, 57, 79, 7, 100, 7, 33, 73, 64, 85, 11, 77, 74, 77, 10, 80, 36, 9,
                78, 74, 64, 53, 77, 89, 60, 95, 56, 37, 86, 78, 82, 25, 12, 56, 50, 65, 52, 64, 3,
                90, 74, 55, 35, 96, 35, 92, 6, 69, 42, 50, 2, 16, 10, 6, 58, 34, 41, 51, 81, 84, 32,
                95, 21, 4, 53, 85, 2, 51, 7, 27, 94, 35, 87, 72, 92, 30, 23, 19, 87, 23, 66, 6, 67,
                7, 75, 87, 53, 64, 70, 38, 97, 73, 84, 57, 53, 14, 29, 82, 24, 50, 10, 32, 25, 62,
                82, 93, 32, 6, 81, 83, 38, 82, 35, 100, 3, 43, 41, 1, 86, 73, 40, 56, 16, 4, 41, 48,
                36, 3, 77, 98, 100, 14, 67, 7, 93, 92, 16, 100, 40, 68, 61, 20, 78, 44, 27, 33, 23,
                8, 27, 79, 88, 10, 64, 50, 76, 9, 90, 13, 5, 5, 98, 94, 79, 28, 59, 11, 62, 89, 20,
                94, 15, 72, 69, 90, 66, 79, 37, 83, 79, 75, 18, 95, 13, 53, 10, 100, 95, 63, 6, 12,
                35, 18, 72, 71, 73, 51, 93, 98, 58, 52, 85, 3, 82, 9, 32, 89, 16, 21, 5, 10, 75, 92,
                92, 85, 73, 5, 1, 11, 49, 79, 16, 6, 93, 93, 60, 55, 7, 51, 23, 42, 96, 58, 64, 95,
                60, 10, 100, 87, 21, 59, 17, 56, 61, 38, 93, 78, 39, 34, 32, 4, 74, 30, 93, 1, 56,
                97, 26, 57, 39, 57, 22, 38, 89, 57, 36, 44, 63, 14, 18, 76, 80, 80, 48, 3, 30, 64,
                91, 92, 35, 71, 53, 76, 43, 70, 17, 5, 32, 45, 78, 76, 93, 98, 49, 61, 86, 2, 76,
                96, 29, 99, 14, 98, 26, 66, 100, 56, 40, 46, 50, 55, 60, 56, 74, 86, 54, 6, 49, 53,
                1, 18, 15, 10, 66, 62, 13, 15, 17, 4, 29, 49, 1, 21, 48, 21, 6, 86, 81, 94, 38, 88,
                56, 55, 30, 29, 95, 23, 70, 69, 22, 76, 90, 23, 68, 4, 25, 44, 18, 34, 40, 71, 98,
                71, 89, 61, 40, 83, 93, 1, 13, 85, 28, 89, 28, 10, 49, 52, 49, 34, 27, 28, 63, 64,
                80, 51, 19, 61, 71, 100, 66, 77, 59, 72, 65, 7, 8, 46, 19, 78, 23, 53, 40, 42, 10,
                86, 22, 55, 71, 89, 100, 31, 88, 31, 89, 90, 13, 6, 79, 39, 2, 82, 53, 97, 12, 27,
                88, 10, 63, 51, 99, 76, 24, 65, 16, 19, 85, 54, 20, 45, 55, 41, 67, 89, 11, 73, 26,
                83, 42, 38, 1, 3, 95, 33, 24, 95, 27, 94, 75, 68, 77, 86, 47, 25, 29, 96, 39, 16,
                36, 37, 47, 61, 45, 80, 79, 27, 67, 28, 32, 11, 31, 39, 48, 19, 16, 56, 87, 19, 17,
                87, 82, 73, 12, 26, 2, 7, 55, 87, 35, 81, 78, 40, 67, 38, 93, 57, 42, 61, 36, 67,
                86, 50, 23, 52, 26, 88, 10, 79, 20, 55, 43, 75, 92, 72, 32, 19, 96, 56, 13, 100, 85,
                29, 72, 3, 54, 100, 41, 28, 98, 78, 14, 92, 63, 86, 82, 69, 73, 23, 43, 98, 23, 58,
                70, 36, 92, 70, 35, 18, 34, 46, 78, 25, 8, 12, 88, 10, 79, 3, 11, 42, 35, 93, 13,
                83, 26, 62, 8, 26, 47, 85, 90, 55, 50, 92, 94, 18, 21, 9, 96, 50, 40, 34, 31, 43,
                80, 43, 80, 59, 35, 35, 93, 35, 62, 93, 80, 35, 94, 41, 61, 32, 21, 55, 15, 52, 39,
                49, 75, 20, 72, 68, 84, 13, 46, 25, 75, 49, 85, 51, 68, 45, 26, 97, 1, 90, 85, 51,
                1, 63, 53, 74, 89, 81, 35, 22, 37, 64, 55, 61, 86, 84, 83, 28, 96, 77, 96, 10, 7, 6,
                21, 49, 88, 15, 59, 12, 61, 31, 37, 42, 78, 50, 66, 27, 62, 11, 96, 10, 12, 50, 20,
                82, 57, 92, 14, 98, 99, 37, 42, 83, 22, 32, 44, 80, 98, 21, 17, 13, 38, 79, 63, 86,
                63, 31, 23, 100, 85, 49, 63, 42, 38, 8, 41, 79, 54, 84, 65, 53, 65, 2, 48, 79, 42,
                60, 37, 72, 11, 43, 99, 69, 42, 32, 23, 55, 34, 31, 94, 87, 70, 6, 23, 30, 81, 29,
                50, 89, 12, 48, 80, 8, 77, 61, 59, 37, 14, 72, 54, 18, 37, 37, 16, 16, 76, 28, 11,
                98, 52, 69, 67, 55, 12, 77, 91, 30, 43, 15, 30, 36, 63, 38, 92, 74, 11, 2, 52, 83,
                77, 92, 30, 57, 86, 38, 21, 67, 93, 64, 2, 7, 6, 7, 4, 6, 51, 68, 65, 81, 76, 50,
                80, 60, 93, 43, 68, 39, 14, 14, 48, 25, 56, 69, 99, 55, 30, 84, 74, 56, 61, 83, 83,
                49, 57, 3, 50, 81, 12, 76, 99, 42, 28, 2, 53, 36, 76, 48, 66, 13, 43, 80, 84, 5, 30,
                30, 26, 49, 77, 33, 69, 24, 83, 34, 33, 75, 11, 6, 33, 70, 74, 38, 59, 60, 2, 93,
                38, 88, 88, 32, 63, 100, 68, 62, 23, 72, 17, 93, 32, 29, 47, 8, 79, 9, 40, 31, 61,
                43, 42, 72, 99, 6, 47, 22, 5, 15, 29, 40, 25, 33, 14, 36, 47, 93, 62, 49, 75, 44,
                81, 18, 63, 6, 36, 55, 87, 82, 96, 62, 18, 77, 95, 96, 31, 64, 61, 47, 34, 95, 53,
                52, 79, 12, 89, 85, 38, 26, 73, 58, 17, 26, 74, 8, 58, 36, 76, 43, 36, 3, 79, 83,
                82, 50, 17, 68, 31, 62, 69, 66, 20, 63, 38, 64, 8, 75, 26, 16, 27, 52, 7, 13, 10,
                26, 67, 43, 70, 77, 1 },
            new double[] { 20, 58, 39, 57, 25, 13, 51, 21, 29, 14, 37, 37, 33, 22, 55, 54, 44, 35,
                51, 30, 53, 48, 34, 57, 12, 20, 13, 42, 52, 33, 55, 55, 47, 47, 12, 22, 35, 13, 48,
                45, 20, 35, 15, 40, 58, 51, 34, 46, 22, 22, 29, 37, 26, 56, 17, 26, 18, 47, 53, 57,
                47, 48, 15, 18, 18, 27, 11, 28, 18, 28, 48, 32, 17, 11, 36, 40, 57, 22, 45, 30, 27,
                33, 27, 29, 15, 12, 49, 53, 11, 51, 33, 40, 28, 24, 26, 33, 17, 32, 15, 31, 46, 25,
                29, 13, 27, 27, 41, 57, 48, 23, 40, 35, 58, 31, 57, 17, 46, 51, 50, 11, 31, 44, 40,
                25, 46, 14, 44, 22, 52, 10, 38, 13, 59, 55, 51, 20, 35, 22, 54, 19, 27, 21, 31, 33,
                37, 38, 26, 31, 45, 51, 46, 53, 23, 17, 18, 38, 26, 18, 50, 35, 15, 55, 18, 39, 25,
                35, 43, 50, 51, 16, 58, 42, 38, 51, 27, 39, 49, 51, 12, 41, 37, 41, 38, 11, 44, 23,
                49, 22, 20, 33, 32, 25, 23, 51, 13, 12, 40, 44, 44, 38, 25, 42, 29, 58, 48, 50, 20,
                26, 33, 22, 14, 54, 22, 38, 49, 32, 19, 34, 35, 50, 53, 20, 27, 10, 42, 32, 30, 53,
                24, 12, 10, 36, 12, 55, 15, 23, 53, 58, 37, 30, 56, 12, 45, 36, 29, 53, 32, 17, 10,
                45, 54, 26, 54, 58, 29, 48, 38, 45, 56, 36, 57, 34, 34, 49, 53, 13, 33, 52, 46, 34,
                15, 54, 55, 56, 31, 17, 44, 47, 37, 44, 39, 42, 10, 18, 15, 12, 21, 40, 46, 48, 29,
                11, 36, 52, 45, 24, 27, 26, 33, 22, 25, 22, 18, 51, 11, 46, 22, 13, 13, 28, 30, 48,
                59, 28, 21, 20, 39, 17, 46, 25, 27, 35, 55, 46, 46, 34, 31, 45, 20, 11, 50, 34, 37,
                30, 23, 37, 27, 36, 12, 44, 28, 39, 33, 28, 31, 51, 40, 46, 14, 53, 32, 16, 58, 27,
                17, 44, 59, 22, 22, 37, 53, 48, 22, 19, 30, 36, 40, 53, 14, 29, 58, 57, 42, 36, 58,
                59, 16, 36, 24, 26, 30, 33, 56, 38, 31, 15, 21, 22, 16, 58, 21, 25, 46, 22, 39, 47,
                48, 21, 50, 10, 13, 32, 57, 49, 37, 15, 27, 37, 59, 38, 28, 59, 59, 50, 32, 31, 37,
                21, 11, 41, 11, 43, 21, 37, 32, 51, 14, 18, 13, 49, 38, 16, 17, 35, 33, 45, 12, 16,
                13, 30, 42, 41, 39, 20, 49, 26, 14, 25, 59, 34, 40, 27, 52, 23, 22, 32, 49, 39, 32,
                56, 17, 20, 38, 44, 55, 39, 30, 27, 50, 41, 34, 56, 17, 17, 12, 43, 24, 17, 17, 35,
                18, 46, 23, 30, 24, 11, 16, 27, 36, 40, 17, 11, 40, 35, 14, 39, 54, 32, 12, 15, 49,
                29, 13, 30, 31, 35, 29, 40, 24, 24, 48, 56, 43, 47, 52, 16, 29, 20, 30, 17, 16, 35,
                33, 37, 38, 49, 27, 11, 15, 49, 54, 53, 56, 18, 43, 14, 17, 55, 41, 47, 41, 21, 14,
                41, 39, 50, 35, 20, 24, 46, 55, 14, 28, 50, 57, 32, 53, 14, 54, 55, 34, 24, 25, 48,
                48, 55, 16, 54, 22, 32, 18, 19, 15, 11, 27, 47, 20, 27, 43, 10, 43, 26, 44, 43, 41,
                20, 29, 13, 35, 16, 25, 28, 22, 40, 37, 53, 13, 29, 51, 25, 18, 24, 18, 38, 38, 31,
                53, 16, 20, 37, 43, 16, 10, 56, 23, 35, 22, 46, 14, 46, 46, 11, 49, 43, 13, 30, 25,
                19, 41, 27, 46, 46, 28, 29, 56, 26, 18, 23, 47, 59, 23, 45, 39, 33, 43, 12, 42, 28,
                32, 43, 16, 25, 30, 10, 45, 10, 42, 44, 29, 28, 12, 40, 33, 16, 52, 41, 48, 28, 45,
                26, 18, 55, 11, 39, 21, 27, 44, 41, 30, 27, 36, 41, 41, 25, 49, 26, 39, 10, 55, 34,
                31, 54, 26, 58, 38, 13, 12, 55, 57, 30, 27, 42, 17, 18, 19, 17, 14, 16, 35, 26, 36,
                21, 25, 25, 46, 39, 46, 19, 43, 18, 13, 37, 44, 47, 55, 14, 55, 10, 19, 40, 18, 16,
                17, 23, 24, 49, 46, 26, 20, 56, 19, 37, 21, 50, 14, 23, 16, 16, 21, 31, 16, 36, 23,
                53, 16, 54, 35, 47, 36, 26, 45, 56, 48, 54, 13, 38, 17, 53, 37, 36, 31, 58, 50, 11,
                13, 23, 58, 31, 15, 42, 43, 41, 37, 47, 26, 30, 28, 51, 57, 43, 24, 19, 42, 41, 56,
                21, 56, 56, 43, 40, 16, 55, 22, 42, 21, 57, 28, 17, 17, 49, 53, 32, 12, 23, 32, 27,
                27, 52, 24, 26, 56, 31, 16, 12, 17, 51, 27, 11, 50, 51, 19, 41, 38, 21, 59, 48, 14,
                38, 19, 42, 16, 41, 39, 29, 55, 21, 14, 59, 35, 19, 56, 18, 32, 43, 15, 46, 31, 50,
                52, 12, 16, 36, 29, 55, 45, 52, 18, 13, 17, 45, 50, 49, 51, 54, 59, 59, 40, 33, 40,
                32, 33, 41, 33, 39, 26, 10, 59, 28, 59, 13, 40, 49, 32, 14, 11, 16, 46, 10, 35, 57,
                47, 35, 32, 21, 29, 30, 20, 15, 53, 48, 24, 27, 22, 12, 32, 39, 32, 48, 48, 18, 18,
                44, 26, 18, 56, 56, 42, 39, 41, 32, 18, 57, 48, 50, 44, 31, 48, 58, 32, 12, 12, 58,
                10, 31, 33, 17, 53, 23, 56, 35, 40, 19, 32, 35, 59, 52, 24, 47, 45, 37, 15, 26, 25,
                39, 49, 37, 10, 43, 37, 23, 45, 51, 34, 12, 25, 56, 27, 54, 52, 15, 11, 34, 46, 53,
                55, 52, 31, 21, 50, 14, 12, 14, 46, 14, 21, 53, 37, 57, 23, 36, 2147483647 },
            new int[] { 1 }, new int[] { 99 });
    }

    public Graph<Integer, DefaultWeightedEdge> getDirectedN11()
    {
        return constructDirectedGraph(
            new int[] { 99, 38, 98, 90, 54, 21, 86, 38, 65, 95, 83, 61, 54, 26, 55, 78, 3, 18, 65,
                24, 86, 31, 16, 48, 57, 35, 20, 48, 62, 57, 80, 64, 63, 16, 24, 91, 60, 31, 16, 79,
                6, 66, 98, 85, 24, 33, 63, 84, 81, 79, 33, 16, 11, 39, 42, 42, 48, 95, 36, 58, 100,
                71, 6, 60, 32, 59, 1, 75, 51, 72, 67, 68, 25, 15, 41, 27, 95, 9, 30, 99, 20, 35, 33,
                90, 62, 23, 27, 51, 41, 80, 26, 14, 10, 26, 6, 87, 32, 95, 7, 78, 19, 39, 81, 36,
                36, 80, 86, 92, 57, 66, 55, 30, 38, 8, 63, 35, 42, 88, 29, 42, 33, 18, 69, 17, 45,
                64, 9, 9, 61, 45, 43, 62, 69, 31, 82, 87, 25, 8, 56, 68, 6, 84, 73, 94, 63, 63, 98,
                86, 22, 77, 95, 66, 36, 42, 48, 14, 5, 23, 2, 79, 47, 62, 31, 47, 59, 1, 78, 43, 47,
                100, 43, 67, 68, 45, 36, 37, 81, 2, 22, 22, 45, 27, 60, 14, 93, 23, 87, 19, 11, 11,
                36, 20, 63, 55, 85, 43, 70, 21, 11, 38, 26, 50, 31, 3, 1, 68, 5, 29, 1, 58, 82, 62,
                48, 53, 59, 99, 15, 42, 90, 52, 31, 77, 7, 27, 75, 91, 96, 81, 15, 36, 61, 72, 4,
                53, 45, 49, 88, 91, 34, 67, 98, 40, 86, 24, 14, 52, 11, 69, 86, 27, 98, 2, 13, 4,
                35, 84, 73, 14, 68, 8, 24, 72, 64, 51, 75, 95, 32, 35, 39, 42, 47, 87, 100, 41, 15,
                13, 15, 76, 52, 84, 80, 4, 56, 11, 98, 43, 5, 91, 85, 34, 72, 32, 75, 23, 28, 66,
                100, 10, 82, 30, 58, 54, 85, 46, 90, 80, 78, 7, 1, 39, 61, 26, 64, 81, 97, 20, 48,
                40, 51, 87, 70, 14, 99, 43, 47, 79, 41, 42, 33, 75, 71, 32, 91, 51, 79, 23, 91, 21,
                44, 9, 75, 33, 33, 46, 68, 60, 83, 24, 63, 73, 16, 17, 7, 53, 65, 61, 26, 34, 73,
                63, 71, 58, 40, 15, 13, 27, 72, 31, 39, 1, 47, 84, 93, 24, 62, 93, 89, 9, 47, 8, 71,
                84, 20, 15, 83, 76, 57, 43, 6, 85, 6, 16, 56, 29, 53, 43, 7, 55, 72, 33, 58, 75, 36,
                87, 83, 76, 6, 94, 94, 91, 99, 1, 82, 51, 82, 20, 88, 9, 94, 45, 77, 56, 85, 5, 41,
                26, 93, 17, 65, 55, 46, 61, 52, 40, 82, 4, 56, 69, 15, 13, 11, 50, 51, 47, 2, 38,
                64, 71, 30, 18, 70, 63, 22, 39, 12, 68, 4, 43, 21, 60, 58, 93, 69, 25, 49, 72, 98,
                84, 90, 74, 23, 83, 65, 94, 16, 7, 63, 43, 17, 29, 94, 91, 17, 40, 9, 26, 83, 66,
                73, 99, 18, 85, 21, 86, 60, 75, 94, 87, 72, 12, 44, 46, 22, 31, 51, 92, 8, 94, 19,
                16, 30, 3, 89, 69, 34, 25, 13, 59, 48, 95, 60, 56, 21, 60, 23, 85, 86, 61, 83, 83,
                88, 47, 65, 72, 59, 52, 79, 9, 52, 73, 94, 40, 62, 26, 83, 88, 39, 37, 76, 70, 15,
                31, 75, 82, 44, 49, 30, 39, 88, 45, 29, 90, 1, 37, 78, 84, 56, 78, 80, 58, 80, 6,
                39, 92, 3, 12, 1, 58, 94, 60, 96, 52, 88, 84, 37, 47, 66, 75, 62, 65, 65, 29, 28,
                12, 26, 57, 74, 95, 32, 21, 11, 83, 9, 89, 43, 37, 87, 52, 5, 96, 14, 97, 16, 41,
                89, 35, 65, 99, 97, 33, 55, 63, 66, 85, 19, 2, 23, 41, 21, 69, 60, 19, 82, 53, 9,
                99, 96, 77, 57, 100, 41, 95, 28, 20, 92, 83, 8, 38, 57, 97, 74, 86, 74, 48, 75, 6,
                2, 98, 58, 60, 50, 15, 10, 77, 14, 9, 1, 62, 40, 27, 48, 49, 39, 10, 91, 44, 29, 52,
                73, 87, 19, 34, 76, 29, 20, 39, 87, 96, 84, 1, 72, 51, 94, 48, 21, 27, 22, 50, 77,
                7, 76, 79, 11, 43, 80, 50, 17, 81, 55, 84, 99, 2, 59, 12, 41, 22, 59, 11, 17, 37,
                71, 94, 41, 58, 21, 26, 16, 86, 96, 92, 90, 27, 25, 46, 33, 86, 97, 12, 9, 85, 1,
                52, 7, 98, 76, 40, 17, 40, 15, 100, 29, 47, 45, 13, 75, 2, 33, 14, 80, 39, 18, 45,
                20, 75, 30, 62, 37, 33, 47, 54, 7, 1, 5, 57, 94, 66, 46, 74, 19, 23, 45, 96, 24, 10,
                99, 9, 3, 75, 54, 82, 18, 66, 68, 89, 93, 14, 39, 57, 91, 16, 32, 16, 29, 88, 59, 8,
                12, 26, 92, 73, 6, 49, 48, 2, 69, 41, 59, 2, 73, 66, 100, 52, 26, 10, 53, 21, 9, 59,
                72, 13, 96, 85, 22, 97, 36, 89, 54, 71, 17, 96, 29, 25, 89, 29, 26, 85, 68, 34, 5,
                82, 40, 9, 85, 29, 23, 75, 67, 2, 75, 59, 9, 27, 95, 19, 76, 56, 57, 21, 18, 84, 66,
                61, 63, 66, 63, 28, 13, 79, 55, 33, 99, 73, 54, 64, 45, 74, 88, 5, 78, 26, 94, 57,
                99, 44, 83, 97, 95, 77, 51, 20, 5, 7, 37, 86, 27, 26, 97, 5, 22, 50, 14, 90, 38, 93,
                60, 35, 54, 85, 59, 35, 38, 22, 35, 94, 44, 13, 35, 96, 14, 31, 1, 23, 83, 14, 48,
                23, 80, 5, 22, 90, 56, 4, 66, 84, 2, 88, 90, 53, 64, 77, 95, 19, 40, 74, 26, 81, 37,
                66, 57, 76, 66, 14, 84, 11, 94, 62, 30, 10, 46, 99, 25, 98, 84, 21, 68, 48, 75, 85,
                98, 61, 64, 31, 40, 21, 39, 76, 100, 46, 87, 6, 17, 42, 70, 60, 51, 88, 38, 56, 96,
                100, 62, 99, 92, 65, 71, 58, 39, 47, 14, 11, 22, 91, 5, 66, 19, 94, 95, 27, 5, 30,
                71, 59, 12, 28, 92, 91, 100, 64, 81, 42, 81, 38, 20, 48, 57, 20, 11, 82, 50, 21, 93,
                89, 61, 54, 71, 50, 67, 52, 7, 54, 82, 18, 64, 99, 8, 96, 77, 35, 39, 56, 80, 85,
                64, 2, 57, 18, 31, 36, 39, 32, 64, 38, 87, 34, 52, 15, 38, 10, 64, 74, 79, 56, 52,
                77, 14, 41, 69, 65, 82, 27, 7, 47, 16, 7, 60, 8, 62, 38, 78, 38, 99 },
            new int[] { 84, 13, 58, 15, 4, 41, 68, 47, 16, 42, 8, 39, 39, 39, 28, 34, 84, 19, 85,
                17, 25, 13, 69, 81, 74, 17, 79, 1, 64, 99, 35, 44, 93, 39, 65, 67, 26, 22, 74, 33,
                75, 34, 41, 57, 58, 11, 46, 28, 39, 100, 41, 99, 53, 83, 48, 43, 39, 33, 73, 96, 65,
                65, 98, 44, 6, 28, 35, 55, 21, 63, 49, 95, 90, 86, 66, 11, 96, 92, 21, 61, 29, 62,
                52, 1, 75, 3, 22, 41, 7, 67, 28, 24, 5, 22, 30, 73, 65, 71, 93, 20, 70, 43, 47, 53,
                54, 8, 43, 89, 90, 73, 48, 18, 21, 10, 40, 36, 50, 54, 96, 79, 60, 3, 81, 63, 5, 36,
                27, 45, 28, 1, 100, 29, 60, 48, 24, 38, 44, 92, 7, 15, 7, 44, 2, 5, 66, 91, 27, 75,
                57, 27, 55, 77, 94, 27, 78, 67, 23, 5, 44, 61, 62, 18, 23, 58, 38, 3, 56, 65, 49,
                56, 45, 25, 81, 75, 89, 55, 13, 15, 5, 55, 78, 94, 87, 58, 24, 13, 27, 72, 73, 28,
                51, 18, 73, 89, 21, 91, 72, 17, 96, 37, 57, 61, 29, 31, 26, 2, 91, 27, 60, 77, 94,
                43, 59, 49, 66, 6, 16, 25, 63, 2, 4, 41, 89, 14, 28, 3, 27, 63, 52, 8, 33, 89, 14,
                98, 88, 68, 58, 12, 81, 40, 75, 4, 99, 12, 25, 28, 71, 2, 53, 24, 16, 20, 53, 98,
                68, 80, 59, 22, 16, 77, 96, 24, 45, 97, 83, 92, 63, 46, 23, 34, 59, 15, 46, 54, 89,
                87, 10, 73, 25, 81, 44, 41, 43, 10, 55, 31, 64, 54, 67, 83, 82, 85, 25, 38, 96, 86,
                43, 91, 57, 23, 70, 55, 40, 13, 61, 66, 48, 3, 21, 97, 27, 43, 77, 64, 40, 58, 9,
                91, 30, 78, 16, 54, 33, 75, 43, 54, 74, 93, 4, 26, 47, 70, 59, 45, 74, 53, 88, 55,
                9, 77, 19, 9, 66, 86, 91, 81, 70, 1, 67, 28, 80, 36, 79, 33, 78, 76, 5, 39, 71, 16,
                46, 63, 33, 30, 76, 92, 99, 99, 86, 56, 93, 79, 32, 43, 76, 84, 90, 17, 8, 13, 18,
                2, 26, 85, 61, 40, 86, 59, 68, 99, 3, 36, 20, 86, 68, 95, 59, 68, 94, 93, 91, 23,
                92, 31, 12, 70, 67, 58, 39, 46, 83, 75, 11, 33, 4, 67, 17, 63, 40, 81, 79, 69, 91,
                15, 6, 87, 17, 98, 34, 9, 64, 32, 91, 35, 84, 60, 27, 70, 29, 35, 89, 75, 49, 57,
                59, 40, 100, 99, 81, 66, 2, 75, 48, 46, 27, 29, 80, 11, 32, 68, 40, 1, 66, 23, 19,
                52, 9, 49, 34, 6, 93, 57, 9, 11, 28, 30, 8, 93, 94, 80, 61, 75, 69, 69, 19, 96, 10,
                79, 89, 85, 12, 4, 96, 55, 28, 32, 10, 63, 87, 71, 42, 17, 6, 24, 23, 50, 64, 69,
                22, 44, 15, 4, 20, 18, 70, 94, 55, 92, 26, 88, 21, 66, 29, 92, 68, 23, 6, 92, 64,
                14, 90, 16, 50, 31, 96, 99, 82, 95, 80, 85, 25, 19, 57, 97, 54, 1, 63, 89, 83, 47,
                92, 58, 16, 25, 28, 38, 32, 33, 51, 47, 9, 58, 46, 78, 49, 83, 71, 30, 51, 62, 76,
                89, 10, 100, 97, 99, 15, 25, 52, 8, 77, 6, 50, 93, 53, 60, 9, 3, 95, 39, 93, 72, 57,
                22, 1, 35, 61, 19, 20, 16, 16, 66, 2, 39, 64, 10, 52, 37, 75, 6, 34, 84, 98, 30, 86,
                96, 83, 42, 86, 78, 45, 56, 32, 7, 17, 58, 27, 93, 7, 61, 41, 46, 97, 24, 74, 13,
                72, 70, 7, 36, 55, 2, 48, 38, 86, 40, 56, 89, 66, 67, 15, 88, 16, 42, 56, 54, 46,
                83, 69, 61, 58, 6, 36, 100, 60, 34, 82, 89, 60, 19, 30, 99, 60, 62, 21, 30, 49, 73,
                97, 14, 10, 32, 15, 73, 88, 69, 68, 39, 76, 69, 66, 100, 98, 91, 40, 83, 35, 31, 57,
                50, 22, 46, 94, 89, 97, 9, 87, 3, 36, 57, 26, 92, 48, 63, 64, 61, 95, 52, 30, 35,
                30, 26, 94, 80, 26, 46, 48, 22, 33, 9, 86, 87, 58, 41, 7, 39, 5, 81, 75, 92, 53, 47,
                52, 96, 91, 58, 95, 53, 41, 12, 4, 18, 8, 25, 52, 4, 29, 36, 20, 79, 71, 73, 16, 61,
                73, 37, 50, 24, 24, 73, 46, 59, 79, 56, 18, 14, 44, 100, 98, 54, 12, 37, 36, 87, 57,
                78, 34, 79, 21, 63, 40, 54, 39, 84, 3, 33, 19, 81, 85, 65, 51, 49, 25, 59, 32, 57,
                65, 83, 8, 80, 31, 23, 89, 63, 26, 40, 62, 89, 75, 37, 25, 33, 79, 88, 70, 62, 54,
                89, 7, 21, 56, 43, 51, 34, 12, 95, 69, 26, 57, 84, 20, 11, 76, 21, 41, 52, 18, 16,
                54, 21, 49, 41, 88, 16, 56, 95, 27, 53, 73, 56, 14, 40, 16, 59, 6, 17, 15, 30, 6, 5,
                67, 17, 98, 42, 11, 31, 58, 12, 41, 40, 89, 12, 94, 82, 20, 9, 39, 49, 41, 53, 12,
                26, 55, 45, 50, 81, 38, 22, 86, 72, 45, 81, 80, 46, 12, 67, 69, 73, 71, 18, 42, 3,
                82, 53, 34, 70, 99, 56, 87, 40, 42, 44, 60, 33, 99, 54, 55, 55, 32, 21, 29, 73, 73,
                74, 33, 84, 77, 34, 99, 55, 54, 33, 56, 3, 28, 6, 40, 82, 47, 49, 7, 55, 47, 30, 75,
                32, 42, 43, 79, 72, 45, 52, 34, 29, 41, 26, 57, 18, 100, 3, 35, 30, 27, 6, 53, 84,
                22, 60, 99, 35, 9, 72, 48, 99, 22, 39, 86, 8, 26, 83, 49, 8, 30, 5, 81, 64, 51, 99,
                29, 77, 78, 80, 19, 27, 25, 78, 3, 74, 83, 61, 19, 60, 18, 72, 1, 3, 62, 65, 85, 11,
                83, 55, 41, 1, 62, 29, 71, 68, 50, 85, 85, 95, 71, 29, 83, 91, 64, 2, 16, 35, 44,
                88, 50, 24, 8, 33, 90, 46, 1, 11, 7, 47, 16, 98, 61, 31, 86, 15, 88, 73, 59, 19, 43,
                98, 10, 10, 82, 48, 49, 9, 51, 44, 51, 17, 87, 75, 1, 75, 19, 42, 16, 80, 56, 55,
                15, 12, 33, 61, 67, 6, 64, 20, 58, 13, 11, 95, 25, 55, 67, 7, 1 },
            new double[] { 20, 58, 39, 57, 25, 13, 51, 21, 29, 14, 37, 37, 33, 22, 55, 54, 44, 35,
                51, 30, 53, 48, 34, 57, 12, 20, 13, 42, 52, 33, 55, 55, 47, 47, 12, 22, 35, 13, 48,
                45, 20, 35, 15, 40, 58, 51, 34, 46, 22, 22, 29, 37, 26, 56, 17, 26, 18, 47, 53, 57,
                47, 48, 15, 18, 18, 27, 11, 28, 18, 28, 48, 32, 17, 11, 36, 40, 57, 22, 45, 30, 27,
                33, 27, 29, 15, 12, 49, 53, 11, 51, 33, 40, 28, 24, 26, 33, 17, 32, 15, 31, 46, 25,
                29, 13, 27, 27, 41, 57, 48, 23, 40, 35, 58, 31, 57, 17, 46, 51, 50, 11, 31, 44, 40,
                25, 46, 14, 44, 22, 52, 10, 38, 13, 59, 55, 51, 20, 35, 22, 54, 19, 27, 21, 31, 33,
                37, 38, 26, 31, 45, 51, 46, 53, 23, 17, 18, 38, 26, 18, 50, 35, 15, 55, 18, 39, 25,
                35, 43, 50, 51, 16, 58, 42, 38, 51, 27, 39, 49, 51, 12, 41, 37, 41, 38, 11, 44, 23,
                49, 22, 20, 33, 32, 25, 23, 51, 13, 12, 40, 44, 44, 38, 25, 42, 29, 58, 48, 50, 20,
                26, 33, 22, 14, 54, 22, 38, 49, 32, 19, 34, 35, 50, 53, 20, 27, 10, 42, 32, 30, 53,
                24, 12, 10, 36, 12, 55, 15, 23, 53, 58, 37, 30, 56, 12, 45, 36, 29, 53, 32, 17, 10,
                45, 54, 26, 54, 58, 29, 48, 38, 45, 56, 36, 57, 34, 34, 49, 53, 13, 33, 52, 46, 34,
                15, 54, 55, 56, 31, 17, 44, 47, 37, 44, 39, 42, 10, 18, 15, 12, 21, 40, 46, 48, 29,
                11, 36, 52, 45, 24, 27, 26, 33, 22, 25, 22, 18, 51, 11, 46, 22, 13, 13, 28, 30, 48,
                59, 28, 21, 20, 39, 17, 46, 25, 27, 35, 55, 46, 46, 34, 31, 45, 20, 11, 50, 34, 37,
                30, 23, 37, 27, 36, 12, 44, 28, 39, 33, 28, 31, 51, 40, 46, 14, 53, 32, 16, 58, 27,
                17, 44, 59, 22, 22, 37, 53, 48, 22, 19, 30, 36, 40, 53, 14, 29, 58, 57, 42, 36, 58,
                59, 16, 36, 24, 26, 30, 33, 56, 38, 31, 15, 21, 22, 16, 58, 21, 25, 46, 22, 39, 47,
                48, 21, 50, 10, 13, 32, 57, 49, 37, 15, 27, 37, 59, 38, 28, 59, 59, 50, 32, 31, 37,
                21, 11, 41, 11, 43, 21, 37, 32, 51, 14, 18, 13, 49, 38, 16, 17, 35, 33, 45, 12, 16,
                13, 30, 42, 41, 39, 20, 49, 26, 14, 25, 59, 34, 40, 27, 52, 23, 22, 32, 49, 39, 32,
                56, 17, 20, 38, 44, 55, 39, 30, 27, 50, 41, 34, 56, 17, 17, 12, 43, 24, 17, 17, 35,
                18, 46, 23, 30, 24, 11, 16, 27, 36, 40, 17, 11, 40, 35, 14, 39, 54, 32, 12, 15, 49,
                29, 13, 30, 31, 35, 29, 40, 24, 24, 48, 56, 43, 47, 52, 16, 29, 20, 30, 17, 16, 35,
                33, 37, 38, 49, 27, 11, 15, 49, 54, 53, 56, 18, 43, 14, 17, 55, 41, 47, 41, 21, 14,
                41, 39, 50, 35, 20, 24, 46, 55, 14, 28, 50, 57, 32, 53, 14, 54, 55, 34, 24, 25, 48,
                48, 55, 16, 54, 22, 32, 18, 19, 15, 11, 27, 47, 20, 27, 43, 10, 43, 26, 44, 43, 41,
                20, 29, 13, 35, 16, 25, 28, 22, 40, 37, 53, 13, 29, 51, 25, 18, 24, 18, 38, 38, 31,
                53, 16, 20, 37, 43, 16, 10, 56, 23, 35, 22, 46, 14, 46, 46, 11, 49, 43, 13, 30, 25,
                19, 41, 27, 46, 46, 28, 29, 56, 26, 18, 23, 47, 59, 23, 45, 39, 33, 43, 12, 42, 28,
                32, 43, 16, 25, 30, 10, 45, 10, 42, 44, 29, 28, 12, 40, 33, 16, 52, 41, 48, 28, 45,
                26, 18, 55, 11, 39, 21, 27, 44, 41, 30, 27, 36, 41, 41, 25, 49, 26, 39, 10, 55, 34,
                31, 54, 26, 58, 38, 13, 12, 55, 57, 30, 27, 42, 17, 18, 19, 17, 14, 16, 35, 26, 36,
                21, 25, 25, 46, 39, 46, 19, 43, 18, 13, 37, 44, 47, 55, 14, 55, 10, 19, 40, 18, 16,
                17, 23, 24, 49, 46, 26, 20, 56, 19, 37, 21, 50, 14, 23, 16, 16, 21, 31, 16, 36, 23,
                53, 16, 54, 35, 47, 36, 26, 45, 56, 48, 54, 13, 38, 17, 53, 37, 36, 31, 58, 50, 11,
                13, 23, 58, 31, 15, 42, 43, 41, 37, 47, 26, 30, 28, 51, 57, 43, 24, 19, 42, 41, 56,
                21, 56, 56, 43, 40, 16, 55, 22, 42, 21, 57, 28, 17, 17, 49, 53, 32, 12, 23, 32, 27,
                27, 52, 24, 26, 56, 31, 16, 12, 17, 51, 27, 11, 50, 51, 19, 41, 38, 21, 59, 48, 14,
                38, 19, 42, 16, 41, 39, 29, 55, 21, 14, 59, 35, 19, 56, 18, 32, 43, 15, 46, 31, 50,
                52, 12, 16, 36, 29, 55, 45, 52, 18, 13, 17, 45, 50, 49, 51, 54, 59, 59, 40, 33, 40,
                32, 33, 41, 33, 39, 26, 10, 59, 28, 59, 13, 40, 49, 32, 14, 11, 16, 46, 10, 35, 57,
                47, 35, 32, 21, 29, 30, 20, 15, 53, 48, 24, 27, 22, 12, 32, 39, 32, 48, 48, 18, 18,
                44, 26, 18, 56, 56, 42, 39, 41, 32, 18, 57, 48, 50, 44, 31, 48, 58, 32, 12, 12, 58,
                10, 31, 33, 17, 53, 23, 56, 35, 40, 19, 32, 35, 59, 52, 24, 47, 45, 37, 15, 26, 25,
                39, 49, 37, 10, 43, 37, 23, 45, 51, 34, 12, 25, 56, 27, 54, 52, 15, 11, 34, 46, 53,
                55, 52, 31, 21, 50, 14, 12, 14, 46, 14, 21, 53, 37, 57, 23, 36, 19, 52, 42, 20, 36,
                49, 58, 21, 39, 33, 50, 43, 31, 53, 41, 31, 20, 47, 40, 11, 36, 11, 40, 29, 24, 13,
                44, 40, 34, 13, 49, 20, 21, 51, 29, 15, 40, 24, 35, 49, 39, 22, 37, 52, 25, 49, 38,
                16, 57, 19, 53, 50, 32, 49, 38, 41, 18, 14, 46, 21, 59, 26, 13, 10, 44, 40, 53, 11,
                22, 11, 40, 24, 19, 23, 36, 42, 41, 16, 12, 34, 52, 48, 32, 50, 18, 27, 46, 22, 21,
                21, 50, 20, 37, 20, 43, 15, 43, 32, 27, 44, 2147483647 },
            new int[] { 1 }, new int[] { 99 });
    }

    public Graph<Integer, DefaultWeightedEdge> getDirectedN12()
    {
        return constructDirectedGraph(
            new int[] { 11, 68, 97, 73, 17, 65, 76, 72, 25, 50, 26, 9, 54, 29, 93, 73, 4, 62, 84,
                40, 78, 77, 59, 75, 82, 30, 15, 51, 9, 92, 20, 51, 95, 59, 24, 50, 74, 65, 50, 30,
                27, 76, 88, 82, 39, 45, 87, 40, 71, 90, 50, 63, 16, 72, 79, 85, 76, 13, 64, 73, 82,
                79, 99, 75, 97, 91, 2, 54, 53, 63, 15, 24, 92, 87, 89, 98, 65, 75, 36, 95, 28, 7,
                64, 93, 32, 85, 34, 36, 50, 14, 77, 28, 37, 29, 86, 82, 40, 77, 64, 3, 9, 47, 44, 7,
                69, 57, 1, 31, 12, 44, 12, 55, 81, 78, 13, 68, 76, 12, 76, 98, 78, 33, 25, 43, 72,
                92, 64, 29, 4, 98, 16, 96, 70, 41, 19, 37, 15, 66, 52, 72, 4, 82, 4, 58, 60, 56, 5,
                59, 49, 77, 21, 79, 38, 83, 38, 29, 78, 92, 9, 25, 65, 48, 1, 28, 33, 42, 8, 82, 8,
                10, 45, 70, 47, 50, 82, 80, 92, 76, 70, 92, 29, 69, 59, 25, 74, 25, 41, 59, 6, 88,
                21, 91, 98, 62, 10, 99, 67, 24, 83, 23, 92, 100, 94, 34, 8, 88, 82, 34, 45, 4, 34,
                97, 78, 48, 67, 26, 32, 78, 92, 16, 97, 75, 13, 52, 55, 47, 52, 13, 10, 73, 11, 60,
                75, 48, 11, 4, 46, 58, 54, 36, 96, 65, 89, 39, 13, 80, 3, 42, 91, 2, 32, 74, 33, 55,
                82, 25, 84, 64, 67, 53, 37, 80, 15, 72, 4, 65, 14, 36, 50, 14, 36, 68, 35, 87, 95,
                13, 96, 86, 16, 98, 10, 91, 2, 23, 5, 89, 37, 6, 11, 99, 48, 84, 99, 66, 11, 94, 6,
                51, 91, 54, 86, 20, 48, 67, 87, 82, 64, 64, 32, 76, 70, 54, 74, 4, 21, 96, 14, 96,
                87, 39, 82, 90, 43, 61, 87, 41, 15, 9, 27, 81, 58, 19, 43, 18, 2, 46, 85, 37, 15,
                28, 34, 89, 73, 61, 41, 71, 2, 20, 70, 78, 77, 43, 51, 100, 31, 35, 26, 34, 5, 13,
                20, 82, 3, 58, 80, 78, 51, 5, 4, 62, 65, 38, 42, 53, 34, 1, 98, 70, 21, 55, 52, 23,
                13, 58, 76, 58, 66, 47, 54, 15, 3, 28, 91, 19, 25, 2, 65, 94, 26, 79, 91, 66, 43,
                78, 73, 80, 68, 32, 3, 39, 19, 64, 24, 16, 3, 72, 54, 78, 62, 41, 45, 78, 42, 84,
                95, 89, 58, 29, 47, 16, 14, 54, 65, 17, 46, 77, 81, 22, 37, 100, 26, 61, 71, 55, 10,
                76, 48, 76, 71, 64, 42, 45, 37, 64, 5, 46, 98, 94, 88, 70, 34, 63, 67, 65, 46, 8,
                89, 28, 55, 5, 83, 89, 66, 92, 61, 38, 75, 99, 33, 89, 53, 76, 26, 55, 22, 19, 22,
                20, 18, 8, 82, 43, 76, 60, 97, 80, 33, 93, 33, 32, 35, 65, 67, 96, 100, 94, 86, 79,
                29, 26, 75, 23, 100, 3, 83, 83, 26, 32, 79, 71, 1, 22, 18, 57, 46, 20, 58, 86, 12,
                27, 11, 3, 31, 6, 97, 56, 70, 90, 18, 79, 88, 34, 2, 85, 26, 79, 36, 83, 81, 71, 94,
                100, 17, 83, 28, 30, 37, 1, 8, 16, 87, 5, 30, 4, 49, 1, 41, 2, 70, 46, 17, 8, 81,
                10, 42, 49, 22, 38, 93, 100, 100, 13, 13, 38, 93, 46, 62, 67, 14, 41, 17, 81, 79,
                22, 37, 63, 42, 83, 23, 13, 86, 27, 43, 42, 89, 5, 39, 52, 4, 48, 15, 54, 29, 95,
                75, 71, 92, 56, 35, 30, 93, 3, 45, 100, 76, 81, 61, 95, 9, 91, 29, 100, 40, 26, 81,
                81, 94, 48, 66, 55, 54, 53, 54, 57, 32, 95, 98, 50, 100, 17, 98, 94, 56, 40, 57, 81,
                89, 88, 29, 2, 90, 80, 86, 77, 61, 40, 88, 22, 2, 92, 81, 15, 78, 27, 54, 34, 52,
                95, 88, 82, 76, 86, 53, 52, 14, 35, 23, 82, 6, 61, 73, 7, 64, 72, 56, 61, 16, 11,
                33, 59, 99 },
            new int[] { 10, 81, 41, 77, 26, 75, 75, 49, 74, 41, 85, 49, 59, 19, 98, 56, 41, 11, 4,
                77, 66, 94, 91, 23, 28, 28, 93, 54, 37, 67, 53, 9, 91, 88, 89, 79, 83, 3, 90, 92,
                47, 21, 92, 3, 82, 42, 29, 24, 40, 66, 52, 20, 29, 77, 20, 53, 60, 80, 52, 53, 50,
                67, 2, 39, 42, 27, 50, 15, 7, 39, 53, 65, 27, 77, 16, 3, 8, 92, 8, 93, 19, 61, 97,
                9, 62, 11, 13, 61, 23, 28, 96, 89, 89, 75, 11, 47, 27, 78, 51, 1, 18, 83, 4, 40, 65,
                89, 69, 16, 84, 25, 59, 56, 67, 27, 23, 64, 38, 56, 3, 22, 29, 55, 77, 7, 6, 83, 45,
                57, 11, 44, 53, 53, 95, 11, 89, 20, 66, 68, 58, 31, 43, 75, 84, 88, 11, 61, 57, 36,
                1, 66, 37, 9, 74, 58, 83, 2, 14, 12, 88, 2, 46, 100, 14, 47, 90, 58, 49, 17, 11, 23,
                80, 73, 52, 93, 74, 21, 99, 88, 90, 66, 59, 41, 29, 80, 69, 40, 21, 65, 89, 35, 53,
                8, 35, 27, 57, 85, 80, 55, 40, 88, 89, 86, 82, 4, 51, 50, 64, 64, 22, 22, 91, 53,
                91, 25, 99, 91, 28, 67, 63, 27, 51, 96, 16, 83, 77, 59, 91, 5, 21, 32, 23, 9, 90,
                92, 97, 33, 43, 93, 65, 40, 3, 4, 91, 77, 19, 29, 29, 86, 40, 67, 93, 20, 44, 78,
                33, 55, 91, 54, 97, 20, 88, 16, 97, 64, 51, 97, 24, 48, 74, 31, 13, 89, 26, 69, 63,
                90, 52, 80, 46, 62, 70, 94, 91, 25, 53, 14, 24, 87, 9, 22, 55, 64, 38, 62, 36, 30,
                1, 84, 59, 2, 16, 10, 86, 34, 46, 12, 87, 29, 75, 39, 58, 67, 56, 87, 41, 16, 86,
                18, 48, 34, 79, 55, 38, 86, 19, 1, 95, 12, 15, 44, 44, 99, 77, 92, 77, 7, 45, 94,
                100, 25, 68, 96, 92, 3, 65, 21, 14, 98, 1, 32, 91, 50, 32, 24, 49, 42, 55, 99, 73,
                17, 30, 80, 80, 33, 11, 68, 64, 8, 9, 6, 44, 32, 14, 35, 18, 25, 36, 80, 80, 64, 15,
                56, 40, 5, 49, 42, 39, 95, 53, 1, 17, 48, 38, 26, 30, 12, 94, 78, 16, 82, 69, 10,
                45, 84, 20, 51, 38, 44, 36, 15, 87, 11, 70, 13, 73, 95, 37, 64, 71, 64, 88, 38, 40,
                65, 67, 30, 57, 34, 32, 45, 91, 33, 19, 58, 3, 81, 25, 90, 98, 19, 10, 68, 100, 58,
                53, 35, 83, 43, 6, 85, 32, 70, 93, 15, 74, 45, 63, 2, 46, 18, 83, 91, 91, 2, 69, 97,
                42, 74, 83, 12, 64, 47, 46, 48, 77, 51, 69, 10, 91, 23, 81, 93, 75, 47, 15, 51, 91,
                82, 7, 75, 71, 84, 30, 15, 76, 67, 92, 87, 51, 3, 74, 77, 86, 15, 95, 17, 90, 49,
                40, 59, 25, 47, 54, 30, 63, 42, 56, 77, 19, 45, 44, 33, 29, 80, 89, 16, 75, 42, 7,
                81, 49, 25, 47, 99, 19, 95, 61, 9, 91, 4, 51, 89, 97, 64, 58, 26, 4, 6, 75, 61, 88,
                44, 98, 23, 30, 12, 57, 93, 32, 86, 98, 32, 78, 61, 17, 12, 50, 66, 54, 29, 15, 63,
                80, 74, 11, 68, 28, 14, 78, 80, 17, 28, 31, 98, 7, 88, 100, 90, 42, 7, 16, 84, 10,
                37, 97, 17, 29, 5, 10, 22, 2, 17, 98, 90, 66, 45, 29, 7, 14, 41, 27, 100, 79, 86,
                19, 62, 54, 82, 45, 100, 60, 15, 12, 61, 5, 38, 81, 15, 22, 4, 97, 7, 56, 87, 31,
                77, 20, 15, 8, 35, 86, 31, 27, 31, 80, 42, 89, 11, 26, 24, 12, 5, 41, 52, 21, 97,
                10, 78, 39, 60, 97, 32, 22, 88, 42, 75, 72, 47, 99, 16, 19, 39, 75, 62, 70, 65, 89,
                51, 58, 72, 42, 27, 64, 40, 43, 62, 15, 1, 52, 5, 86, 80, 24, 98, 57, 59, 69, 55,
                86, 5, 1 },
            new double[] { 20, 58, 39, 57, 25, 13, 51, 21, 29, 14, 37, 37, 33, 22, 55, 54, 44, 35,
                51, 30, 53, 48, 34, 57, 12, 20, 13, 42, 52, 33, 55, 55, 47, 47, 12, 22, 35, 13, 48,
                45, 20, 35, 15, 40, 58, 51, 34, 46, 22, 22, 29, 37, 26, 56, 17, 26, 18, 47, 53, 57,
                47, 48, 15, 18, 18, 27, 11, 28, 18, 28, 48, 32, 17, 11, 36, 40, 57, 22, 45, 30, 27,
                33, 27, 29, 15, 12, 49, 53, 11, 51, 33, 40, 28, 24, 26, 33, 17, 32, 15, 31, 46, 25,
                29, 13, 27, 27, 41, 57, 48, 23, 40, 35, 58, 31, 57, 17, 46, 51, 50, 11, 31, 44, 40,
                25, 46, 14, 44, 22, 52, 10, 38, 13, 59, 55, 51, 20, 35, 22, 54, 19, 27, 21, 31, 33,
                37, 38, 26, 31, 45, 51, 46, 53, 23, 17, 18, 38, 26, 18, 50, 35, 15, 55, 18, 39, 25,
                35, 43, 50, 51, 16, 58, 42, 38, 51, 27, 39, 49, 51, 12, 41, 37, 41, 38, 11, 44, 23,
                49, 22, 20, 33, 32, 25, 23, 51, 13, 12, 40, 44, 44, 38, 25, 42, 29, 58, 48, 50, 20,
                26, 33, 22, 14, 54, 22, 38, 49, 32, 19, 34, 35, 50, 53, 20, 27, 10, 42, 32, 30, 53,
                24, 12, 10, 36, 12, 55, 15, 23, 53, 58, 37, 30, 56, 12, 45, 36, 29, 53, 32, 17, 10,
                45, 54, 26, 54, 58, 29, 48, 38, 45, 56, 36, 57, 34, 34, 49, 53, 13, 33, 52, 46, 34,
                15, 54, 55, 56, 31, 17, 44, 47, 37, 44, 39, 42, 10, 18, 15, 12, 21, 40, 46, 48, 29,
                11, 36, 52, 45, 24, 27, 26, 33, 22, 25, 22, 18, 51, 11, 46, 22, 13, 13, 28, 30, 48,
                59, 28, 21, 20, 39, 17, 46, 25, 27, 35, 55, 46, 46, 34, 31, 45, 20, 11, 50, 34, 37,
                30, 23, 37, 27, 36, 12, 44, 28, 39, 33, 28, 31, 51, 40, 46, 14, 53, 32, 16, 58, 27,
                17, 44, 59, 22, 22, 37, 53, 48, 22, 19, 30, 36, 40, 53, 14, 29, 58, 57, 42, 36, 58,
                59, 16, 36, 24, 26, 30, 33, 56, 38, 31, 15, 21, 22, 16, 58, 21, 25, 46, 22, 39, 47,
                48, 21, 50, 10, 13, 32, 57, 49, 37, 15, 27, 37, 59, 38, 28, 59, 59, 50, 32, 31, 37,
                21, 11, 41, 11, 43, 21, 37, 32, 51, 14, 18, 13, 49, 38, 16, 17, 35, 33, 45, 12, 16,
                13, 30, 42, 41, 39, 20, 49, 26, 14, 25, 59, 34, 40, 27, 52, 23, 22, 32, 49, 39, 32,
                56, 17, 20, 38, 44, 55, 39, 30, 27, 50, 41, 34, 56, 17, 17, 12, 43, 24, 17, 17, 35,
                18, 46, 23, 30, 24, 11, 16, 27, 36, 40, 17, 11, 40, 35, 14, 39, 54, 32, 12, 15, 49,
                29, 13, 30, 31, 35, 29, 40, 24, 24, 48, 56, 43, 47, 52, 16, 29, 20, 30, 17, 16, 35,
                33, 37, 38, 49, 27, 11, 15, 49, 54, 53, 56, 18, 43, 14, 17, 55, 41, 47, 41, 21, 14,
                41, 39, 50, 35, 20, 24, 46, 55, 14, 28, 50, 57, 32, 53, 14, 54, 55, 34, 24, 25, 48,
                48, 55, 16, 54, 22, 32, 18, 19, 15, 11, 27, 47, 20, 27, 43, 10, 43, 26, 44, 43, 41,
                20, 29, 13, 35, 16, 25, 28, 22, 40, 37, 53, 13, 29, 51, 25, 18, 24, 18, 38, 38, 31,
                53, 16, 20, 37, 43, 16, 10, 56, 23, 35, 22, 46, 14, 46, 46, 11, 49, 43, 13, 30, 25,
                19, 41, 27, 46, 46, 28, 29, 56, 26, 18, 23, 47, 59, 23, 45, 39, 33, 43, 12, 42, 28,
                32, 43, 16, 25, 30, 10, 45, 10, 42, 44, 29, 28, 12, 40, 33, 16, 52, 41, 48, 28, 45,
                26, 18, 55, 11, 39, 21, 27, 44, 41, 30, 27, 36, 41, 41, 25, 49, 26, 39, 10, 55, 34,
                31, 54, 26, 58, 38, 13, 12, 55, 57, 30, 2147483647 },
            new int[] { 1 }, new int[] { 99 });
    }

    public Graph<Integer, DefaultWeightedEdge> generateDirectedGraph()
    {
        GraphGenerator<Integer, DefaultWeightedEdge, Integer> randomGraphGenerator =
            new GnmRandomGraphGenerator<>(100, 500);
        Random rand = new Random();
        SimpleDirectedWeightedGraph<Integer, DefaultWeightedEdge> directedGraph =
            new SimpleDirectedWeightedGraph<>(
                SupplierUtil.createIntegerSupplier(), SupplierUtil.DEFAULT_WEIGHTED_EDGE_SUPPLIER);
        randomGraphGenerator.generateGraph(directedGraph);
        directedGraph
            .edgeSet().stream().forEach(e -> directedGraph.setEdgeWeight(e, rand.nextInt(100)));
        return directedGraph;
    }

    /*************** TEST CASES FOR UNDIRECTED GRAPHS ***************/

    public Graph<Integer, DefaultWeightedEdge> getUndirectedN1()
    {
        int[][] edges =
            { { 0, 1, 12 }, { 0, 2, 15 }, { 0, 3, 20 }, { 1, 5, 5 }, { 1, 6, 2 }, { 1, 2, 5 },
                { 2, 6, 6 }, { 2, 4, 3 }, { 2, 3, 11 }, { 3, 4, 4 }, { 3, 7, 8 }, { 4, 6, 6 },
                { 4, 7, 1 }, { 5, 6, 9 }, { 5, 8, 18 }, { 6, 7, 7 }, { 6, 8, 13 }, { 7, 8, 10 } };
        return constructUndirectedGraph(edges);
    }

    public Graph<Integer, DefaultWeightedEdge> getUndirectedN2()
    {
        int[][] edges = { { 2, 1, 20 }, { 3, 4, 58 }, { 1, 3, 39 }, { 3, 2, 57 }, { 4, 2, 25 },
            { 5, 3, 13 }, { 5, 1, 51 }, { 5, 4, 21 } };
        return constructUndirectedGraph(edges);
    }

    public Graph<Integer, DefaultWeightedEdge> getUndirectedN3()
    {
        int[][] edges =
            { { 8, 15, 20 }, { 24, 30, 58 }, { 26, 33, 39 }, { 2, 22, 57 }, { 26, 5, 25 },
                { 39, 22, 13 }, { 37, 10, 51 }, { 15, 6, 21 }, { 28, 19, 29 }, { 24, 35, 14 },
                { 6, 20, 37 }, { 31, 17, 37 }, { 29, 9, 33 }, { 19, 40, 22 }, { 12, 32, 55 },
                { 15, 27, 54 }, { 47, 25, 44 }, { 43, 1, 35 }, { 31, 11, 51 }, { 17, 42, 30 },
                { 44, 36, 53 }, { 17, 25, 48 }, { 19, 46, 34 }, { 2, 44, 57 }, { 5, 32, 12 },
                { 50, 2, 20 }, { 14, 10, 13 }, { 5, 24, 42 }, { 15, 25, 52 }, { 44, 25, 33 },
                { 1, 28, 55 }, { 45, 10, 55 }, { 31, 20, 47 }, { 22, 34, 47 }, { 16, 5, 12 },
                { 42, 40, 22 }, { 39, 45, 35 }, { 37, 8, 13 }, { 47, 32, 48 }, { 7, 10, 45 },
                { 34, 8, 20 }, { 44, 41, 35 }, { 43, 30, 15 }, { 27, 43, 40 }, { 3, 7, 58 },
                { 19, 32, 51 }, { 10, 5, 34 }, { 15, 10, 46 }, { 31, 50, 22 }, { 24, 40, 22 },
                { 30, 13, 29 }, { 6, 10, 37 }, { 48, 29, 26 }, { 28, 26, 56 }, { 22, 48, 17 },
                { 4, 39, 26 }, { 21, 24, 18 }, { 2, 30, 47 }, { 32, 48, 53 }, { 48, 6, 57 },
                { 20, 22, 47 }, { 31, 18, 48 }, { 12, 15, 15 }, { 21, 14, 18 }, { 25, 49, 18 },
                { 2, 46, 27 }, { 45, 47, 11 }, { 37, 6, 28 }, { 28, 21, 18 }, { 42, 18, 28 },
                { 10, 1, 48 }, { 31, 30, 32 }, { 32, 16, 17 }, { 32, 28, 11 }, { 38, 35, 36 },
                { 20, 28, 40 }, { 29, 43, 57 }, { 7, 23, 22 }, { 28, 27, 45 }, { 23, 16, 30 },
                { 11, 44, 27 }, { 14, 49, 33 }, { 11, 7, 27 }, { 46, 39, 29 }, { 8, 26, 15 },
                { 14, 7, 12 }, { 47, 33, 49 }, { 49, 8, 53 }, { 29, 5, 11 }, { 36, 41, 51 },
                { 18, 14, 33 }, { 25, 39, 40 }, { 21, 39, 28 }, { 42, 15, 24 }, { 39, 11, 26 },
                { 16, 22, 33 }, { 10, 11, 17 }, { 38, 13, 32 }, { 34, 32, 15 }, { 31, 33, 31 },
                { 45, 7, 46 }, { 34, 3, 25 }, { 7, 29, 29 }, { 22, 13, 13 }, { 35, 25, 27 },
                { 16, 35, 27 }, { 33, 45, 41 }, { 6, 18, 57 }, { 35, 22, 48 }, { 8, 43, 23 },
                { 44, 45, 40 }, { 5, 20, 35 }, { 34, 14, 58 }, { 48, 45, 31 }, { 27, 24, 57 },
                { 3, 18, 17 }, { 6, 29, 46 }, { 6, 30, 51 }, { 28, 15, 50 }, { 13, 32, 11 } };
        return constructUndirectedGraph(edges);
    }

    public Graph<Integer, DefaultWeightedEdge> getUndirectedN4()
    {
        int[][] edges =
            { { 100, 23, 20 }, { 96, 70, 58 }, { 79, 97, 39 }, { 24, 22, 57 }, { 11, 64, 25 },
                { 64, 96, 13 }, { 87, 62, 51 }, { 23, 15, 21 }, { 12, 37, 29 }, { 68, 81, 14 },
                { 54, 59, 37 }, { 75, 29, 37 }, { 3, 24, 33 }, { 92, 43, 22 }, { 37, 23, 55 },
                { 31, 43, 54 }, { 37, 53, 44 }, { 3, 79, 35 }, { 50, 63, 51 }, { 1, 95, 30 },
                { 11, 12, 53 }, { 8, 100, 48 }, { 9, 92, 34 }, { 10, 89, 57 }, { 45, 47, 12 },
                { 85, 39, 20 }, { 54, 41, 13 }, { 89, 87, 42 }, { 44, 46, 52 }, { 67, 99, 33 },
                { 40, 87, 55 }, { 19, 50, 55 }, { 40, 73, 47 }, { 4, 92, 47 }, { 99, 46, 12 },
                { 87, 17, 22 }, { 90, 82, 35 }, { 66, 97, 13 }, { 78, 75, 48 }, { 43, 90, 45 },
                { 10, 25, 20 }, { 74, 24, 35 }, { 6, 77, 15 }, { 13, 39, 40 }, { 6, 13, 58 },
                { 70, 34, 51 }, { 76, 42, 34 }, { 99, 50, 46 }, { 53, 38, 22 }, { 21, 13, 22 },
                { 75, 33, 29 }, { 63, 74, 37 }, { 56, 51, 26 }, { 51, 99, 56 }, { 23, 66, 17 },
                { 99, 87, 26 }, { 16, 99, 18 }, { 74, 85, 47 }, { 82, 15, 53 }, { 90, 94, 57 },
                { 85, 65, 47 }, { 26, 55, 48 }, { 27, 3, 15 }, { 52, 19, 18 }, { 84, 89, 18 },
                { 16, 57, 27 }, { 29, 32, 11 }, { 29, 37, 28 }, { 93, 86, 18 }, { 94, 45, 28 },
                { 60, 31, 48 }, { 74, 98, 32 }, { 9, 3, 17 }, { 31, 32, 11 }, { 22, 25, 36 },
                { 17, 60, 40 }, { 5, 99, 57 }, { 98, 8, 22 }, { 73, 33, 45 }, { 69, 33, 30 },
                { 42, 70, 27 }, { 85, 70, 33 }, { 5, 63, 27 }, { 4, 8, 29 }, { 44, 45, 15 },
                { 12, 5, 12 }, { 83, 87, 49 }, { 4, 2, 53 }, { 17, 89, 11 }, { 83, 37, 51 },
                { 82, 64, 33 }, { 41, 50, 40 }, { 84, 16, 28 }, { 9, 10, 24 }, { 12, 6, 26 },
                { 14, 100, 33 }, { 25, 8, 17 }, { 79, 85, 32 }, { 52, 39, 15 }, { 63, 8, 31 },
                { 55, 43, 46 }, { 21, 79, 25 }, { 80, 91, 29 }, { 89, 53, 13 }, { 23, 9, 27 },
                { 95, 57, 27 }, { 4, 29, 41 }, { 93, 39, 57 }, { 62, 54, 48 }, { 85, 86, 23 },
                { 85, 1, 40 }, { 17, 55, 35 }, { 35, 43, 58 }, { 79, 93, 31 }, { 97, 80, 57 },
                { 90, 87, 17 }, { 68, 88, 46 }, { 22, 30, 51 }, { 42, 82, 50 }, { 49, 51, 11 },
                { 74, 44, 31 }, { 87, 14, 44 }, { 86, 71, 40 }, { 23, 55, 25 }, { 85, 60, 46 },
                { 28, 37, 14 }, { 97, 54, 44 }, { 7, 53, 22 }, { 87, 58, 52 }, { 44, 62, 10 },
                { 57, 11, 38 }, { 99, 28, 13 }, { 7, 64, 59 }, { 31, 84, 55 }, { 36, 91, 51 },
                { 22, 88, 20 }, { 84, 39, 35 }, { 23, 70, 22 }, { 85, 76, 54 }, { 18, 84, 19 },
                { 41, 7, 27 }, { 99, 73, 21 }, { 66, 79, 31 }, { 22, 31, 33 }, { 44, 5, 37 },
                { 95, 14, 38 }, { 92, 95, 26 }, { 37, 14, 31 }, { 74, 68, 45 }, { 4, 24, 51 },
                { 45, 81, 46 }, { 100, 47, 53 }, { 100, 59, 23 }, { 8, 52, 17 }, { 24, 94, 18 },
                { 70, 71, 38 }, { 83, 85, 26 }, { 59, 83, 18 }, { 83, 14, 50 }, { 76, 72, 35 },
                { 65, 82, 15 }, { 99, 9, 55 }, { 30, 55, 18 }, { 73, 21, 39 }, { 96, 100, 25 },
                { 88, 85, 35 }, { 11, 35, 43 }, { 45, 3, 50 }, { 93, 77, 51 }, { 98, 42, 16 },
                { 25, 34, 58 }, { 14, 73, 42 }, { 71, 59, 38 }, { 33, 94, 51 }, { 51, 87, 27 },
                { 42, 85, 39 }, { 18, 3, 49 }, { 44, 28, 51 }, { 11, 84, 12 }, { 81, 93, 41 },
                { 7, 29, 37 }, { 18, 42, 41 }, { 19, 98, 38 }, { 35, 26, 11 }, { 6, 96, 44 },
                { 96, 40, 23 }, { 67, 50, 49 }, { 30, 69, 22 }, { 14, 97, 20 }, { 43, 48, 33 },
                { 40, 79, 32 }, { 86, 89, 25 }, { 31, 44, 23 }, { 94, 8, 51 }, { 89, 1, 13 },
                { 50, 57, 12 }, { 38, 94, 40 }, { 7, 97, 44 }, { 33, 96, 44 }, { 59, 98, 38 },
                { 62, 100, 25 }, { 77, 15, 42 }, { 79, 7, 29 }, { 28, 35, 58 }, { 95, 25, 48 },
                { 76, 91, 50 }, { 99, 76, 20 }, { 28, 51, 26 }, { 84, 98, 33 }, { 4, 15, 22 },
                { 88, 44, 14 }, { 83, 79, 54 }, { 49, 45, 22 }, { 3, 93, 38 }, { 81, 43, 49 },
                { 77, 48, 32 }, { 38, 68, 19 }, { 11, 56, 34 }, { 6, 28, 35 }, { 34, 72, 50 },
                { 94, 16, 53 }, { 62, 55, 20 }, { 5, 50, 27 }, { 42, 92, 10 }, { 2, 1, 42 },
                { 46, 96, 32 }, { 15, 12, 30 }, { 57, 6, 53 }, { 84, 79, 24 }, { 49, 75, 12 },
                { 90, 9, 10 }, { 15, 84, 36 }, { 32, 58, 12 }, { 57, 94, 55 }, { 26, 77, 15 },
                { 6, 35, 23 }, { 48, 37, 53 }, { 15, 56, 58 }, { 45, 74, 37 }, { 10, 8, 30 },
                { 88, 15, 56 }, { 97, 67, 12 }, { 83, 98, 45 }, { 93, 45, 36 }, { 25, 81, 29 },
                { 73, 51, 53 }, { 24, 43, 32 }, { 26, 53, 17 }, { 84, 59, 10 }, { 97, 18, 45 },
                { 32, 26, 54 }, { 61, 30, 26 }, { 24, 45, 54 }, { 88, 25, 58 }, { 50, 54, 29 },
                { 62, 40, 48 }, { 95, 23, 38 }, { 95, 63, 45 }, { 63, 59, 56 }, { 99, 10, 36 },
                { 25, 84, 57 }, { 92, 29, 34 }, { 58, 14, 34 }, { 79, 12, 49 }, { 1, 70, 53 },
                { 35, 79, 13 }, { 91, 83, 33 }, { 45, 26, 52 }, { 72, 78, 46 }, { 52, 92, 34 },
                { 28, 25, 15 }, { 22, 46, 54 }, { 47, 8, 55 }, { 26, 52, 56 }, { 50, 34, 31 },
                { 41, 99, 17 }, { 2, 52, 44 }, { 84, 10, 47 }, { 77, 20, 37 }, { 22, 78, 44 },
                { 81, 67, 39 }, { 67, 56, 42 }, { 34, 95, 10 }, { 14, 39, 18 }, { 63, 86, 15 },
                { 88, 35, 12 }, { 3, 7, 21 }, { 14, 55, 40 }, { 25, 18, 46 }, { 81, 19, 48 },
                { 45, 98, 29 }, { 18, 38, 11 }, { 59, 55, 36 }, { 75, 85, 52 }, { 86, 95, 45 },
                { 50, 53, 24 }, { 82, 7, 27 }, { 30, 87, 26 }, { 2, 51, 33 }, { 11, 40, 22 },
                { 5, 64, 25 }, { 72, 15, 22 }, { 94, 25, 18 }, { 61, 10, 51 }, { 39, 2, 11 },
                { 86, 51, 46 }, { 56, 6, 22 }, { 15, 16, 13 }, { 11, 27, 13 }, { 58, 54, 28 },
                { 86, 62, 30 }, { 88, 52, 48 }, { 48, 63, 59 }, { 89, 96, 28 }, { 50, 33, 21 },
                { 36, 23, 20 }, { 83, 77, 39 }, { 70, 68, 17 }, { 60, 10, 46 }, { 20, 68, 25 },
                { 30, 81, 27 }, { 10, 23, 35 }, { 9, 40, 55 }, { 44, 23, 46 }, { 92, 17, 46 },
                { 98, 91, 34 }, { 64, 72, 31 }, { 47, 94, 45 }, { 56, 43, 20 }, { 39, 54, 11 },
                { 10, 26, 50 }, { 62, 80, 34 }, { 7, 19, 37 }, { 98, 46, 30 }, { 20, 23, 23 },
                { 1, 97, 37 }, { 12, 95, 27 }, { 40, 60, 36 }, { 12, 29, 12 }, { 67, 48, 44 },
                { 67, 31, 28 }, { 11, 85, 39 }, { 95, 66, 33 }, { 97, 100, 28 }, { 18, 12, 31 },
                { 84, 95, 51 }, { 70, 88, 40 }, { 17, 84, 46 }, { 97, 81, 14 }, { 91, 73, 53 },
                { 61, 8, 32 }, { 46, 56, 16 }, { 81, 71, 58 }, { 15, 39, 27 }, { 24, 96, 17 },
                { 82, 16, 44 }, { 20, 45, 59 }, { 96, 45, 22 }, { 11, 93, 22 }, { 13, 64, 37 },
                { 88, 38, 53 }, { 3, 51, 48 }, { 81, 16, 22 }, { 7, 40, 19 }, { 33, 13, 30 },
                { 18, 28, 36 }, { 86, 65, 40 }, { 52, 14, 53 }, { 21, 67, 14 }, { 99, 96, 29 },
                { 78, 80, 58 }, { 2, 99, 57 }, { 77, 78, 42 }, { 12, 65, 36 }, { 51, 98, 58 },
                { 48, 73, 59 }, { 36, 28, 16 }, { 57, 67, 36 }, { 21, 9, 24 }, { 15, 73, 26 },
                { 26, 54, 30 }, { 90, 65, 33 }, { 29, 100, 56 }, { 11, 79, 38 }, { 95, 61, 31 },
                { 64, 19, 15 }, { 66, 64, 21 }, { 92, 19, 22 }, { 70, 5, 16 }, { 36, 33, 58 },
                { 71, 48, 21 }, { 46, 89, 25 }, { 79, 48, 46 }, { 70, 86, 22 }, { 89, 77, 39 },
                { 42, 78, 47 }, { 99, 15, 48 }, { 88, 36, 21 }, { 88, 93, 50 }, { 49, 77, 10 },
                { 76, 62, 13 }, { 15, 67, 32 }, { 52, 3, 57 }, { 10, 6, 49 }, { 99, 81, 37 },
                { 22, 27, 15 }, { 61, 74, 27 }, { 45, 86, 37 }, { 57, 10, 59 }, { 91, 8, 38 },
                { 80, 48, 28 }, { 100, 99, 59 }, { 27, 83, 59 }, { 3, 54, 50 }, { 67, 66, 32 },
                { 47, 9, 31 }, { 39, 68, 37 }, { 63, 75, 21 }, { 46, 35, 11 }, { 25, 2, 41 },
                { 72, 80, 11 }, { 20, 48, 43 }, { 39, 96, 21 }, { 50, 76, 37 }, { 35, 17, 32 },
                { 10, 77, 51 }, { 50, 22, 14 }, { 8, 99, 18 }, { 93, 50, 13 }, { 23, 12, 49 },
                { 84, 42, 38 }, { 69, 1, 16 }, { 17, 90, 17 }, { 27, 62, 35 }, { 94, 72, 33 },
                { 33, 67, 45 }, { 15, 43, 12 }, { 99, 31, 16 }, { 7, 67, 13 }, { 91, 9, 30 },
                { 16, 24, 42 }, { 88, 37, 41 }, { 5, 29, 39 }, { 86, 49, 20 }, { 36, 37, 49 },
                { 93, 37, 26 }, { 89, 94, 14 }, { 53, 28, 25 }, { 34, 30, 59 }, { 7, 42, 34 },
                { 59, 27, 40 }, { 61, 64, 27 }, { 49, 66, 52 }, { 35, 60, 23 }, { 47, 85, 22 },
                { 34, 42, 32 }, { 4, 10, 49 }, { 51, 66, 39 }, { 36, 61, 32 }, { 9, 11, 56 },
                { 91, 86, 17 }, { 53, 45, 20 }, { 71, 53, 38 }, { 75, 61, 44 }, { 85, 82, 55 },
                { 97, 96, 39 }, { 52, 44, 30 }, { 6, 14, 27 }, { 31, 89, 50 }, { 93, 25, 41 },
                { 52, 95, 34 }, { 60, 27, 56 }, { 34, 82, 17 }, { 10, 20, 17 }, { 17, 6, 12 },
                { 54, 5, 43 }, { 74, 88, 24 }, { 91, 96, 17 }, { 10, 58, 17 }, { 62, 95, 35 },
                { 91, 58, 18 }, { 30, 67, 46 }, { 53, 91, 23 }, { 62, 47, 30 }, { 30, 39, 24 },
                { 39, 65, 11 }, { 86, 11, 16 }, { 41, 72, 27 }, { 78, 49, 36 }, { 14, 92, 40 },
                { 65, 99, 17 }, { 15, 59, 11 }, { 44, 98, 40 }, { 50, 18, 35 }, { 46, 21, 14 },
                { 11, 7, 39 }, { 50, 44, 54 }, { 1, 46, 32 }, { 45, 4, 12 }, { 21, 82, 15 },
                { 5, 4, 49 }, { 31, 74, 29 }, { 87, 24, 13 }, { 72, 68, 30 }, { 45, 33, 31 },
                { 25, 91, 35 }, { 12, 66, 29 }, { 2, 68, 40 }, { 3, 19, 24 }, { 5, 22, 24 },
                { 24, 10, 48 }, { 72, 63, 56 }, { 99, 13, 43 }, { 67, 9, 47 }, { 64, 100, 52 },
                { 57, 60, 16 }, { 71, 84, 29 }, { 82, 97, 20 }, { 62, 15, 30 }, { 73, 41, 17 },
                { 2, 17, 16 }, { 99, 85, 35 }, { 5, 42, 33 }, { 3, 58, 37 }, { 70, 8, 38 },
                { 71, 24, 49 }, { 10, 51, 27 }, { 85, 55, 11 }, { 68, 100, 15 }, { 41, 3, 49 },
                { 30, 92, 54 }, { 39, 81, 53 }, { 86, 83, 56 }, { 18, 78, 18 }, { 27, 71, 43 },
                { 10, 12, 14 }, { 7, 90, 17 }, { 8, 45, 55 }, { 10, 54, 41 }, { 32, 76, 47 },
                { 7, 32, 41 }, { 3, 16, 21 }, { 5, 52, 14 }, { 4, 93, 41 }, { 60, 37, 39 },
                { 75, 80, 50 }, { 88, 57, 35 }, { 80, 17, 20 }, { 67, 8, 24 }, { 18, 92, 46 },
                { 66, 75, 55 }, { 56, 17, 14 }, { 36, 19, 28 }, { 38, 61, 50 }, { 19, 84, 57 },
                { 17, 51, 32 }, { 81, 84, 53 }, { 80, 56, 14 }, { 42, 58, 54 }, { 26, 99, 55 },
                { 9, 29, 34 }, { 58, 2, 24 }, { 24, 8, 25 }, { 21, 52, 48 }, { 93, 83, 48 },
                { 16, 59, 55 }, { 67, 88, 16 }, { 1, 12, 54 }, { 90, 80, 22 }, { 57, 78, 32 },
                { 58, 71, 18 }, { 81, 91, 19 }, { 2, 11, 15 }, { 73, 96, 11 }, { 70, 2, 27 },
                { 1, 52, 47 }, { 37, 70, 20 }, { 76, 80, 27 }, { 50, 1, 43 }, { 14, 71, 10 },
                { 7, 80, 43 }, { 97, 12, 26 }, { 58, 26, 44 }, { 8, 75, 43 }, { 65, 33, 41 },
                { 77, 3, 20 }, { 86, 17, 29 }, { 99, 55, 13 }, { 92, 32, 35 }, { 12, 91, 16 },
                { 82, 55, 25 }, { 41, 6, 28 }, { 62, 84, 22 }, { 6, 70, 40 }, { 79, 72, 37 },
                { 90, 12, 53 }, { 76, 92, 13 }, { 94, 99, 29 }, { 48, 7, 51 }, { 90, 68, 25 },
                { 46, 91, 18 }, { 20, 32, 24 }, { 13, 98, 18 }, { 10, 86, 38 }, { 78, 64, 38 },
                { 24, 47, 31 }, { 55, 56, 53 }, { 87, 88, 16 }, { 88, 27, 20 }, { 89, 3, 37 },
                { 72, 67, 43 }, { 34, 15, 16 }, { 6, 2, 10 }, { 92, 57, 56 }, { 55, 72, 23 },
                { 9, 70, 35 }, { 83, 74, 22 }, { 67, 98, 46 }, { 20, 37, 14 }, { 46, 76, 46 },
                { 93, 10, 46 }, { 7, 49, 11 }, { 25, 11, 49 }, { 18, 62, 43 }, { 5, 88, 13 },
                { 88, 58, 30 }, { 62, 6, 25 }, { 29, 93, 19 }, { 88, 43, 41 }, { 35, 97, 27 },
                { 2, 74, 46 }, { 28, 98, 46 }, { 7, 15, 28 }, { 74, 67, 29 }, { 38, 10, 56 },
                { 11, 18, 26 }, { 16, 34, 18 }, { 8, 19, 23 }, { 43, 83, 47 }, { 86, 72, 59 },
                { 19, 73, 23 }, { 42, 80, 45 }, { 41, 5, 39 }, { 71, 8, 33 }, { 87, 65, 43 },
                { 72, 24, 12 }, { 2, 57, 42 }, { 9, 83, 28 }, { 84, 29, 32 }, { 83, 48, 43 },
                { 60, 14, 16 }, { 79, 38, 25 }, { 74, 77, 30 }, { 80, 40, 10 }, { 78, 51, 45 },
                { 86, 3, 10 }, { 73, 27, 42 }, { 89, 43, 44 }, { 97, 49, 29 }, { 82, 67, 28 },
                { 14, 34, 12 }, { 60, 94, 40 }, { 48, 97, 33 }, { 50, 21, 16 }, { 87, 50, 52 },
                { 2, 14, 41 }, { 38, 17, 48 }, { 27, 82, 28 }, { 58, 41, 45 }, { 45, 55, 26 },
                { 5, 26, 18 }, { 71, 94, 55 }, { 1, 17, 11 }, { 54, 91, 39 }, { 43, 70, 21 },
                { 34, 52, 27 }, { 18, 47, 44 }, { 30, 89, 41 }, { 100, 15, 30 }, { 70, 79, 27 },
                { 100, 48, 36 }, { 86, 87, 41 }, { 75, 77, 41 }, { 33, 32, 25 }, { 64, 31, 49 },
                { 95, 83, 26 }, { 50, 82, 39 }, { 55, 33, 10 }, { 24, 15, 55 }, { 25, 69, 34 },
                { 68, 3, 31 }, { 53, 41, 54 }, { 70, 48, 26 }, { 62, 46, 58 }, { 63, 32, 38 },
                { 12, 49, 13 }, { 55, 91, 12 }, { 9, 50, 55 }, { 78, 55, 57 }, { 46, 23, 30 },
                { 52, 75, 27 }, { 34, 22, 42 }, { 11, 4, 17 }, { 20, 57, 18 }, { 64, 95, 19 },
                { 100, 4, 17 }, { 56, 16, 14 }, { 44, 94, 16 }, { 94, 86, 35 }, { 7, 8, 26 },
                { 80, 2, 36 }, { 88, 49, 21 }, { 37, 32, 25 }, { 33, 20, 25 }, { 85, 90, 46 },
                { 55, 36, 39 }, { 35, 53, 46 }, { 60, 49, 19 }, { 7, 66, 43 }, { 56, 76, 18 },
                { 47, 46, 13 }, { 54, 64, 37 }, { 74, 6, 44 }, { 76, 67, 47 }, { 64, 56, 55 },
                { 35, 16, 14 }, { 19, 6, 55 }, { 52, 57, 10 }, { 30, 90, 19 }, { 73, 75, 40 },
                { 17, 43, 18 }, { 58, 56, 16 }, { 73, 80, 17 }, { 95, 26, 23 }, { 79, 2, 24 },
                { 42, 13, 49 }, { 30, 43, 46 }, { 59, 30, 26 }, { 67, 78, 20 }, { 55, 65, 56 },
                { 99, 82, 19 }, { 50, 47, 37 }, { 14, 75, 21 }, { 21, 27, 50 }, { 80, 98, 14 },
                { 26, 47, 23 }, { 44, 99, 16 }, { 47, 6, 16 }, { 89, 6, 21 }, { 38, 72, 31 },
                { 70, 95, 16 }, { 8, 84, 36 }, { 33, 30, 23 }, { 18, 9, 53 }, { 61, 19, 16 },
                { 21, 35, 54 }, { 16, 7, 35 }, { 64, 12, 47 }, { 57, 56, 36 }, { 80, 69, 26 },
                { 14, 26, 45 }, { 68, 65, 56 }, { 97, 37, 48 }, { 16, 47, 54 }, { 61, 72, 13 },
                { 94, 63, 38 }, { 92, 2, 17 }, { 76, 48, 53 }, { 97, 11, 37 }, { 37, 73, 36 },
                { 87, 60, 31 }, { 44, 48, 58 }, { 97, 28, 50 }, { 87, 11, 11 }, { 6, 94, 13 },
                { 3, 1, 23 }, { 47, 83, 58 }, { 52, 18, 31 }, { 18, 88, 15 }, { 62, 45, 42 },
                { 12, 75, 43 }, { 96, 56, 41 }, { 88, 40, 37 }, { 30, 76, 47 }, { 62, 51, 26 },
                { 94, 58, 30 }, { 96, 71, 28 }, { 25, 85, 51 }, { 8, 48, 57 }, { 16, 100, 43 },
                { 83, 1, 24 }, { 87, 82, 19 }, { 51, 92, 42 }, { 57, 34, 41 }, { 25, 35, 56 },
                { 25, 66, 21 }, { 45, 97, 56 }, { 88, 79, 56 }, { 70, 30, 43 }, { 90, 15, 40 },
                { 57, 93, 16 }, { 42, 90, 55 }, { 64, 68, 22 }, { 10, 96, 42 }, { 46, 24, 21 },
                { 54, 1, 57 }, { 35, 85, 28 }, { 8, 53, 17 }, { 6, 90, 17 }, { 52, 48, 49 },
                { 96, 90, 53 }, { 11, 43, 32 }, { 54, 98, 12 }, { 31, 10, 23 }, { 29, 98, 32 },
                { 64, 22, 27 }, { 32, 79, 27 }, { 29, 94, 52 }, { 10, 27, 24 }, { 30, 45, 26 },
                { 39, 69, 56 }, { 63, 15, 31 }, { 21, 83, 16 }, { 86, 77, 12 }, { 85, 66, 17 },
                { 42, 37, 51 }, { 83, 24, 27 }, { 91, 31, 11 }, { 1, 29, 50 }, { 46, 3, 51 },
                { 68, 56, 19 }, { 88, 100, 41 }, { 11, 80, 38 }, { 2, 44, 21 }, { 42, 53, 59 },
                { 37, 64, 48 }, { 70, 76, 14 }, { 82, 11, 38 }, { 79, 23, 19 }, { 34, 73, 42 },
                { 57, 70, 16 }, { 57, 13, 41 }, { 48, 18, 39 }, { 50, 25, 29 }, { 88, 6, 55 },
                { 83, 25, 21 }, { 75, 55, 14 }, { 12, 93, 59 }, { 40, 65, 35 }, { 89, 63, 19 },
                { 27, 78, 56 }, { 63, 88, 18 }, { 69, 10, 32 }, { 79, 29, 43 }, { 17, 24, 15 },
                { 44, 16, 46 }, { 2, 8, 31 }, { 73, 26, 50 }, { 25, 37, 52 }, { 54, 87, 12 },
                { 33, 6, 16 }, { 67, 41, 36 }, { 72, 6, 29 }, { 67, 95, 55 }, { 35, 98, 45 },
                { 49, 34, 52 }, { 63, 18, 18 }, { 70, 4, 13 }, { 17, 8, 17 }, { 53, 36, 45 },
                { 37, 100, 50 }, { 74, 81, 49 }, { 38, 39, 51 }, { 28, 76, 54 }, { 93, 75, 59 },
                { 18, 81, 59 }, { 3, 53, 40 }, { 23, 87, 33 }, { 75, 27, 40 }, { 7, 99, 32 },
                { 48, 69, 33 }, { 65, 25, 41 }, { 31, 69, 33 }, { 96, 50, 39 }, { 40, 70, 26 },
                { 93, 22, 10 }, { 20, 2, 59 }, { 67, 19, 28 }, { 84, 73, 59 }, { 33, 2, 13 },
                { 71, 62, 40 }, { 70, 11, 49 }, { 50, 27, 32 }, { 98, 56, 14 }, { 45, 57, 11 },
                { 45, 90, 16 }, { 9, 65, 46 }, { 64, 39, 10 }, { 41, 55, 35 }, { 8, 33, 57 },
                { 18, 2, 47 }, { 14, 85, 35 }, { 77, 41, 32 }, { 20, 27, 21 }, { 65, 13, 29 },
                { 84, 23, 30 }, { 5, 20, 20 }, { 73, 90, 15 }, { 10, 92, 53 }, { 46, 13, 48 },
                { 91, 3, 24 }, { 98, 30, 27 }, { 89, 49, 22 }, { 74, 3, 12 }, { 30, 42, 32 },
                { 5, 23, 39 }, { 9, 96, 32 }, { 77, 90, 48 }, { 21, 57, 48 }, { 34, 24, 18 },
                { 73, 36, 18 }, { 68, 57, 44 }, { 13, 72, 26 }, { 6, 80, 18 }, { 22, 84, 56 },
                { 49, 30, 56 }, { 79, 37, 42 }, { 61, 62, 39 }, { 58, 36, 41 }, { 79, 30, 32 },
                { 46, 32, 18 }, { 26, 81, 57 }, { 28, 19, 48 }, { 20, 76, 50 }, { 18, 20, 44 },
                { 41, 20, 31 }, { 64, 53, 48 }, { 85, 97, 58 }, { 83, 60, 32 }, { 13, 32, 12 },
                { 84, 47, 12 }, { 2, 24, 58 }, { 80, 4, 10 }, { 73, 47, 31 }, { 47, 51, 33 },
                { 77, 85, 17 }, { 37, 43, 53 }, { 96, 77, 23 }, { 84, 3, 56 }, { 53, 98, 35 },
                { 28, 66, 40 }, { 16, 11, 19 }, { 6, 7, 32 }, { 84, 33, 35 }, { 91, 43, 59 },
                { 91, 42, 52 }, { 3, 82, 24 }, { 12, 55, 47 }, { 75, 65, 45 }, { 55, 92, 37 },
                { 96, 78, 15 }, { 62, 3, 26 }, { 81, 5, 25 }, { 70, 75, 39 }, { 17, 13, 49 },
                { 17, 91, 37 }, { 28, 91, 10 }, { 89, 23, 43 }, { 28, 75, 37 }, { 35, 19, 23 },
                { 2, 35, 45 }, { 48, 82, 51 }, { 19, 53, 34 }, { 56, 75, 12 }, { 55, 1, 25 },
                { 15, 64, 56 }, { 53, 4, 27 }, { 43, 82, 54 }, { 76, 45, 52 }, { 37, 55, 15 },
                { 59, 5, 11 }, { 48, 25, 34 }, { 62, 72, 46 }, { 61, 5, 53 }, { 4, 91, 55 },
                { 46, 45, 52 }, { 40, 100, 31 }, { 98, 16, 21 }, { 31, 34, 50 }, { 42, 12, 14 },
                { 86, 81, 12 }, { 14, 13, 14 }, { 24, 13, 46 }, { 43, 47, 14 }, { 32, 4, 21 },
                { 20, 87, 53 }, { 1, 44, 37 }, { 25, 97, 57 }, { 94, 66, 23 }, { 35, 99, 36 } };
        return constructUndirectedGraph(edges);
    }

    public Graph<Integer, DefaultWeightedEdge> getUndirectedN5()
    {
        int[][] edges = { { 18, 3, 20 }, { 31, 30, 58 }, { 19, 8, 39 }, { 38, 22, 57 },
            { 21, 34, 25 }, { 47, 21, 13 }, { 49, 40, 51 }, { 29, 10, 21 }, { 20, 23, 29 },
            { 46, 42, 14 }, { 20, 32, 37 }, { 1, 28, 37 }, { 24, 44, 33 }, { 34, 39, 22 },
            { 31, 27, 55 }, { 17, 23, 54 }, { 15, 44, 44 }, { 40, 6, 35 }, { 49, 4, 51 },
            { 14, 35, 30 }, { 25, 35, 53 }, { 34, 4, 48 }, { 17, 29, 34 }, { 24, 21, 57 },
            { 41, 33, 12 }, { 8, 18, 20 }, { 10, 8, 13 }, { 27, 6, 42 }, { 3, 45, 52 },
            { 8, 40, 33 }, { 31, 45, 55 }, { 4, 28, 55 }, { 34, 20, 47 }, { 47, 33, 47 },
            { 30, 27, 12 }, { 34, 40, 22 }, { 5, 36, 35 }, { 25, 9, 13 }, { 10, 1, 48 },
            { 1, 27, 45 }, { 19, 40, 20 }, { 30, 41, 35 }, { 34, 36, 15 }, { 42, 47, 40 },
            { 29, 36, 58 }, { 5, 39, 51 }, { 47, 29, 34 }, { 23, 44, 46 }, { 3, 48, 22 },
            { 13, 1, 22 }, { 19, 23, 29 }, { 27, 19, 37 }, { 41, 2, 26 }, { 46, 50, 56 },
            { 41, 12, 17 }, { 4, 14, 26 }, { 34, 33, 18 }, { 11, 2, 47 }, { 21, 22, 53 },
            { 25, 32, 57 }, { 39, 31, 47 }, { 30, 20, 48 }, { 34, 47, 15 }, { 29, 16, 18 },
            { 4, 27, 18 }, { 28, 48, 27 }, { 30, 2, 11 }, { 35, 12, 28 }, { 45, 28, 18 },
            { 35, 32, 28 }, { 29, 22, 48 }, { 2, 39, 32 }, { 38, 13, 17 }, { 46, 4, 11 },
            { 3, 39, 36 }, { 35, 16, 40 }, { 8, 43, 57 }, { 38, 40, 22 }, { 39, 43, 45 },
            { 41, 44, 30 }, { 50, 43, 27 }, { 4, 47, 33 }, { 46, 40, 27 }, { 18, 32, 29 },
            { 37, 3, 15 }, { 34, 16, 12 }, { 41, 45, 49 }, { 33, 11, 53 }, { 16, 4, 11 },
            { 25, 3, 51 }, { 26, 21, 33 }, { 19, 32, 40 }, { 38, 9, 28 }, { 26, 30, 24 },
            { 41, 17, 26 }, { 29, 21, 33 }, { 39, 23, 17 }, { 32, 46, 32 }, { 40, 9, 15 },
            { 17, 16, 31 }, { 6, 47, 46 }, { 8, 22, 25 }, { 41, 36, 29 }, { 33, 3, 13 },
            { 47, 35, 27 }, { 4, 21, 27 }, { 45, 46, 41 }, { 10, 19, 57 }, { 48, 14, 48 },
            { 50, 16, 23 }, { 25, 23, 40 }, { 6, 48, 35 }, { 44, 26, 58 }, { 33, 7, 31 },
            { 39, 33, 57 }, { 37, 44, 17 }, { 26, 50, 46 }, { 7, 41, 51 }, { 19, 38, 50 },
            { 38, 30, 11 }, { 2, 25, 31 }, { 22, 32, 44 }, { 32, 15, 40 }, { 12, 42, 25 },
            { 48, 36, 46 }, { 40, 43, 14 }, { 19, 5, 44 }, { 37, 23, 22 }, { 10, 18, 52 },
            { 11, 49, 10 }, { 33, 29, 38 }, { 4, 8, 13 }, { 44, 31, 59 }, { 48, 30, 55 },
            { 22, 28, 51 }, { 47, 1, 20 }, { 14, 30, 35 }, { 24, 46, 22 }, { 33, 46, 54 },
            { 13, 14, 19 }, { 39, 15, 27 }, { 27, 14, 21 }, { 24, 47, 31 }, { 33, 17, 33 },
            { 9, 12, 37 }, { 9, 39, 38 }, { 10, 24, 26 }, { 50, 36, 31 }, { 11, 48, 45 },
            { 10, 36, 51 }, { 47, 11, 46 }, { 7, 22, 53 }, { 13, 39, 23 }, { 28, 32, 17 },
            { 50, 30, 18 }, { 21, 41, 38 }, { 44, 46, 26 }, { 1, 2, 18 }, { 43, 19, 50 },
            { 27, 42, 35 }, { 32, 48, 15 }, { 23, 48, 55 }, { 28, 50, 18 }, { 24, 3, 39 },
            { 36, 35, 25 }, { 15, 5, 35 }, { 30, 3, 43 }, { 29, 27, 50 }, { 17, 35, 51 },
            { 14, 45, 16 }, { 18, 17, 58 }, { 1, 30, 42 }, { 17, 22, 38 }, { 18, 12, 51 },
            { 12, 8, 27 }, { 15, 22, 39 }, { 11, 50, 49 }, { 19, 29, 51 }, { 25, 22, 12 },
            { 45, 5, 41 }, { 6, 36, 37 }, { 2, 4, 41 }, { 46, 7, 38 }, { 3, 31, 11 }, { 3, 10, 44 },
            { 31, 26, 23 }, { 44, 21, 49 }, { 37, 36, 22 }, { 20, 39, 20 }, { 22, 3, 33 },
            { 39, 19, 32 }, { 28, 10, 25 }, { 44, 4, 23 }, { 10, 46, 51 }, { 27, 50, 13 },
            { 12, 3, 12 }, { 3, 29, 40 }, { 48, 4, 44 }, { 32, 43, 44 }, { 48, 12, 38 } };
        return constructUndirectedGraph(edges);
    }

    public Graph<Integer, DefaultWeightedEdge> getUndirectedN6()
    {
        int[][] edges =
            { { 20, 51, 20 }, { 43, 61, 58 }, { 27, 6, 39 }, { 47, 13, 57 }, { 63, 4, 25 },
                { 37, 68, 13 }, { 31, 60, 51 }, { 67, 57, 21 }, { 46, 44, 29 }, { 47, 60, 14 },
                { 7, 51, 37 }, { 61, 58, 37 }, { 21, 28, 33 }, { 62, 36, 22 }, { 18, 47, 55 },
                { 43, 57, 54 }, { 52, 28, 44 }, { 9, 47, 35 }, { 48, 19, 51 }, { 56, 50, 30 },
                { 26, 15, 53 }, { 8, 61, 48 }, { 37, 9, 34 }, { 9, 20, 57 }, { 2, 58, 12 },
                { 57, 19, 20 }, { 19, 6, 13 }, { 27, 54, 42 }, { 28, 6, 52 }, { 50, 67, 33 },
                { 37, 64, 55 }, { 21, 55, 55 }, { 22, 39, 47 }, { 24, 16, 47 }, { 19, 20, 12 },
                { 65, 69, 22 }, { 6, 12, 35 }, { 6, 25, 13 }, { 2, 32, 48 }, { 65, 12, 45 },
                { 63, 59, 20 }, { 41, 7, 35 }, { 53, 41, 15 }, { 44, 39, 40 }, { 31, 44, 58 },
                { 24, 58, 51 }, { 49, 4, 34 }, { 26, 67, 46 }, { 60, 8, 22 }, { 5, 7, 22 },
                { 7, 64, 29 }, { 33, 1, 37 }, { 38, 29, 26 }, { 5, 52, 56 }, { 62, 37, 17 },
                { 69, 46, 26 }, { 54, 22, 18 }, { 32, 55, 47 }, { 47, 29, 53 }, { 59, 36, 57 },
                { 40, 66, 47 }, { 35, 53, 48 }, { 57, 49, 15 }, { 2, 7, 18 }, { 70, 42, 18 },
                { 12, 54, 27 }, { 49, 1, 11 }, { 31, 34, 28 }, { 27, 64, 18 }, { 35, 52, 28 },
                { 51, 28, 48 }, { 48, 60, 32 }, { 57, 17, 17 }, { 57, 1, 11 }, { 9, 33, 36 },
                { 13, 46, 40 }, { 10, 59, 57 }, { 67, 53, 22 }, { 26, 48, 45 }, { 69, 53, 30 },
                { 59, 64, 27 }, { 48, 1, 33 }, { 13, 33, 27 }, { 60, 64, 29 }, { 51, 50, 15 },
                { 4, 43, 12 }, { 25, 10, 49 }, { 65, 11, 53 }, { 9, 2, 11 }, { 16, 23, 51 },
                { 1, 59, 33 }, { 68, 5, 40 }, { 1, 2, 28 }, { 43, 37, 24 }, { 39, 27, 26 },
                { 21, 19, 33 }, { 33, 61, 17 }, { 60, 55, 32 }, { 42, 55, 15 }, { 64, 16, 31 },
                { 17, 13, 46 }, { 23, 52, 25 }, { 23, 29, 29 }, { 52, 7, 13 }, { 6, 50, 27 },
                { 13, 36, 27 }, { 64, 67, 41 }, { 3, 65, 57 }, { 17, 69, 48 }, { 3, 22, 23 },
                { 31, 27, 40 }, { 62, 42, 35 }, { 15, 11, 58 }, { 61, 45, 31 }, { 43, 65, 57 },
                { 14, 40, 17 }, { 37, 6, 46 }, { 55, 43, 51 }, { 54, 29, 50 }, { 33, 4, 11 },
                { 39, 68, 31 }, { 62, 47, 44 }, { 65, 63, 40 }, { 6, 38, 25 }, { 32, 69, 46 },
                { 5, 8, 14 }, { 68, 55, 44 }, { 32, 7, 22 }, { 33, 53, 52 }, { 15, 2, 10 },
                { 50, 61, 38 }, { 22, 57, 13 }, { 7, 26, 59 }, { 63, 25, 55 }, { 67, 32, 51 },
                { 26, 63, 20 }, { 29, 2, 35 }, { 21, 43, 22 }, { 55, 35, 54 }, { 15, 45, 19 },
                { 13, 2, 27 }, { 47, 16, 21 }, { 66, 50, 31 }, { 13, 49, 33 }, { 38, 63, 37 },
                { 26, 12, 38 }, { 5, 36, 26 }, { 49, 62, 31 }, { 2, 56, 45 }, { 64, 63, 51 },
                { 49, 18, 46 }, { 18, 5, 53 }, { 28, 29, 23 }, { 24, 12, 17 }, { 26, 55, 18 },
                { 21, 18, 38 }, { 14, 33, 26 }, { 26, 1, 18 }, { 68, 64, 50 }, { 30, 53, 35 },
                { 58, 5, 15 }, { 22, 55, 55 }, { 38, 14, 18 }, { 8, 30, 39 }, { 57, 4, 25 },
                { 44, 27, 35 }, { 10, 62, 43 }, { 5, 57, 50 }, { 63, 28, 51 }, { 33, 63, 16 },
                { 36, 3, 58 }, { 57, 44, 42 }, { 28, 61, 38 }, { 45, 57, 51 }, { 20, 15, 27 },
                { 40, 34, 39 }, { 9, 26, 49 }, { 62, 18, 51 }, { 34, 24, 12 }, { 66, 36, 41 },
                { 45, 56, 37 }, { 40, 32, 41 }, { 57, 51, 38 }, { 47, 45, 11 }, { 6, 20, 44 },
                { 1, 8, 23 }, { 30, 47, 49 }, { 62, 69, 22 }, { 53, 58, 20 }, { 19, 1, 33 },
                { 41, 12, 32 }, { 7, 3, 25 }, { 41, 42, 23 }, { 5, 10, 51 }, { 55, 58, 13 },
                { 8, 40, 12 }, { 5, 30, 40 }, { 22, 15, 44 }, { 38, 68, 44 }, { 62, 14, 38 } };
        return constructUndirectedGraph(edges);
    }

    public Graph<Integer, DefaultWeightedEdge> getUndirectedN7()
    {
        int[][] edges =
            { { 5, 40, 20 }, { 5, 38, 58 }, { 17, 40, 39 }, { 61, 2, 57 }, { 25, 38, 25 },
                { 23, 47, 13 }, { 40, 26, 51 }, { 7, 57, 21 }, { 34, 40, 29 }, { 1, 33, 14 },
                { 32, 37, 37 }, { 57, 40, 37 }, { 42, 9, 33 }, { 11, 70, 22 }, { 8, 28, 55 },
                { 5, 18, 54 }, { 41, 36, 44 }, { 2, 19, 35 }, { 3, 5, 51 }, { 11, 25, 30 },
                { 34, 18, 53 }, { 6, 59, 48 }, { 48, 58, 34 }, { 39, 21, 57 }, { 42, 2, 12 },
                { 27, 7, 20 }, { 62, 11, 13 }, { 43, 48, 42 }, { 56, 40, 52 }, { 21, 25, 33 },
                { 67, 59, 55 }, { 67, 32, 55 }, { 23, 65, 47 }, { 12, 8, 47 }, { 58, 10, 12 },
                { 59, 34, 22 }, { 9, 69, 35 }, { 22, 54, 13 }, { 55, 33, 48 }, { 39, 16, 45 },
                { 1, 14, 20 }, { 52, 65, 35 }, { 65, 42, 15 }, { 23, 42, 40 }, { 63, 8, 58 },
                { 34, 46, 51 }, { 66, 33, 34 }, { 30, 53, 46 }, { 39, 63, 22 }, { 2, 53, 22 },
                { 24, 27, 29 }, { 49, 44, 37 }, { 27, 1, 26 }, { 3, 60, 56 }, { 53, 22, 17 },
                { 58, 15, 26 }, { 45, 20, 18 }, { 20, 14, 47 }, { 30, 61, 53 }, { 46, 65, 57 },
                { 33, 44, 47 }, { 13, 45, 48 }, { 63, 24, 15 }, { 39, 44, 18 }, { 18, 44, 18 },
                { 19, 61, 27 }, { 4, 43, 11 }, { 26, 20, 28 }, { 32, 66, 18 }, { 30, 47, 28 },
                { 53, 35, 48 }, { 50, 10, 32 }, { 27, 35, 17 }, { 67, 65, 11 }, { 46, 36, 36 },
                { 24, 35, 40 }, { 55, 68, 57 }, { 47, 32, 22 }, { 55, 36, 45 }, { 19, 20, 30 } };
        return constructUndirectedGraph(edges);
    }

    public Graph<Integer, DefaultWeightedEdge> getUndirectedN8()
    {
        int[][] edges =
            { { 22, 65, 20 }, { 59, 32, 58 }, { 45, 13, 39 }, { 29, 75, 57 }, { 100, 54, 25 },
                { 96, 59, 13 }, { 83, 43, 51 }, { 76, 84, 21 }, { 77, 88, 29 }, { 95, 8, 14 },
                { 79, 70, 37 }, { 99, 78, 37 }, { 7, 59, 33 }, { 89, 39, 22 }, { 53, 64, 55 },
                { 38, 95, 54 }, { 17, 69, 44 }, { 83, 28, 35 }, { 72, 16, 51 }, { 77, 13, 30 },
                { 6, 70, 53 }, { 97, 67, 48 }, { 48, 88, 34 }, { 87, 73, 57 }, { 83, 38, 12 },
                { 45, 80, 20 }, { 3, 7, 13 }, { 24, 74, 42 }, { 96, 98, 52 }, { 47, 26, 33 },
                { 1, 72, 55 }, { 31, 51, 55 }, { 21, 56, 47 }, { 67, 71, 47 }, { 49, 86, 12 },
                { 71, 29, 22 }, { 47, 11, 35 }, { 31, 13, 13 }, { 80, 8, 48 }, { 46, 30, 45 },
                { 64, 74, 20 }, { 6, 54, 35 }, { 11, 57, 15 }, { 59, 26, 40 }, { 92, 9, 58 },
                { 71, 44, 51 }, { 63, 55, 34 }, { 84, 60, 46 }, { 29, 65, 22 }, { 94, 35, 22 },
                { 31, 2, 29 }, { 13, 87, 37 }, { 92, 21, 26 }, { 86, 80, 56 }, { 99, 32, 17 },
                { 54, 63, 26 }, { 81, 12, 18 }, { 36, 78, 47 }, { 64, 28, 53 }, { 100, 75, 57 },
                { 18, 80, 47 }, { 66, 47, 48 }, { 85, 57, 15 }, { 64, 15, 18 }, { 55, 98, 18 },
                { 60, 93, 27 }, { 80, 10, 11 }, { 78, 33, 28 }, { 95, 29, 18 }, { 6, 82, 28 },
                { 66, 20, 48 }, { 10, 50, 32 }, { 95, 97, 17 }, { 58, 57, 11 }, { 79, 62, 36 },
                { 56, 6, 40 }, { 31, 25, 57 }, { 74, 85, 22 }, { 96, 32, 45 }, { 88, 5, 30 },
                { 70, 59, 27 }, { 28, 92, 33 }, { 60, 62, 27 }, { 6, 38, 29 }, { 78, 53, 15 },
                { 85, 97, 12 }, { 26, 43, 49 }, { 23, 6, 53 }, { 4, 39, 11 }, { 25, 34, 51 },
                { 29, 87, 33 }, { 20, 100, 40 }, { 99, 87, 28 }, { 21, 55, 24 }, { 46, 92, 26 },
                { 39, 67, 33 }, { 47, 6, 17 }, { 74, 82, 32 }, { 74, 63, 15 }, { 28, 41, 31 },
                { 23, 43, 46 }, { 64, 21, 25 }, { 13, 46, 29 }, { 13, 8, 13 }, { 34, 72, 27 },
                { 64, 69, 27 }, { 76, 69, 41 }, { 94, 21, 57 }, { 54, 64, 48 }, { 51, 86, 23 },
                { 16, 83, 40 }, { 83, 50, 35 }, { 63, 10, 58 }, { 31, 55, 31 }, { 2, 34, 57 },
                { 100, 94, 17 }, { 47, 94, 46 }, { 53, 36, 51 }, { 82, 34, 50 }, { 54, 8, 11 },
                { 52, 37, 31 }, { 41, 32, 44 }, { 16, 62, 40 }, { 39, 46, 25 }, { 63, 13, 46 },
                { 52, 51, 14 }, { 56, 23, 44 }, { 100, 79, 22 }, { 92, 66, 52 }, { 9, 70, 10 },
                { 38, 65, 38 }, { 70, 51, 13 }, { 97, 43, 59 }, { 11, 44, 55 }, { 13, 34, 51 },
                { 92, 47, 20 }, { 61, 18, 35 }, { 60, 68, 22 }, { 20, 91, 54 }, { 54, 84, 19 },
                { 23, 69, 27 }, { 55, 46, 21 }, { 50, 72, 31 }, { 59, 48, 33 }, { 42, 21, 37 },
                { 35, 25, 38 }, { 70, 41, 26 }, { 56, 98, 31 }, { 8, 4, 45 }, { 37, 56, 51 },
                { 83, 60, 46 }, { 51, 93, 53 }, { 11, 24, 23 }, { 53, 82, 17 }, { 97, 100, 18 },
                { 8, 61, 38 }, { 44, 92, 26 }, { 29, 43, 18 }, { 90, 29, 50 }, { 11, 15, 35 },
                { 12, 80, 15 }, { 1, 71, 55 }, { 58, 62, 18 }, { 3, 83, 39 }, { 14, 17, 25 },
                { 37, 48, 35 }, { 58, 18, 43 }, { 52, 30, 50 }, { 78, 75, 51 }, { 69, 18, 16 },
                { 44, 52, 58 }, { 49, 35, 42 }, { 24, 9, 38 }, { 95, 35, 51 }, { 44, 59, 27 },
                { 98, 75, 39 }, { 1, 53, 49 }, { 50, 21, 51 }, { 92, 14, 12 }, { 14, 26, 41 },
                { 15, 82, 37 }, { 37, 32, 41 }, { 1, 66, 38 }, { 66, 69, 11 }, { 81, 48, 44 },
                { 50, 59, 23 }, { 68, 7, 49 }, { 84, 100, 22 }, { 65, 19, 20 }, { 3, 54, 33 },
                { 17, 18, 32 }, { 60, 99, 25 }, { 100, 77, 23 }, { 56, 10, 51 }, { 98, 35, 13 },
                { 70, 47, 12 }, { 29, 15, 40 }, { 10, 37, 44 }, { 88, 20, 44 }, { 25, 4, 38 },
                { 77, 37, 25 }, { 1, 5, 42 }, { 69, 19, 29 }, { 74, 22, 58 }, { 12, 84, 48 },
                { 87, 26, 50 }, { 76, 34, 20 }, { 90, 85, 26 }, { 58, 97, 33 }, { 58, 70, 22 },
                { 35, 50, 14 }, { 63, 98, 54 }, { 30, 25, 22 }, { 53, 83, 38 }, { 12, 73, 49 },
                { 90, 73, 32 }, { 86, 40, 19 }, { 67, 44, 34 }, { 55, 14, 35 }, { 18, 31, 50 },
                { 41, 96, 53 }, { 17, 81, 20 }, { 65, 6, 27 }, { 19, 35, 10 }, { 23, 20, 42 },
                { 28, 30, 32 }, { 24, 18, 30 }, { 84, 55, 53 }, { 75, 43, 24 }, { 41, 25, 12 },
                { 64, 25, 10 }, { 69, 49, 36 }, { 24, 32, 12 }, { 94, 77, 55 }, { 2, 65, 15 },
                { 29, 4, 23 }, { 43, 9, 53 }, { 95, 79, 58 }, { 100, 3, 37 }, { 32, 58, 30 },
                { 70, 62, 56 }, { 34, 1, 12 }, { 71, 8, 45 }, { 13, 57, 36 }, { 36, 5, 29 },
                { 5, 67, 53 }, { 20, 8, 32 }, { 13, 32, 17 }, { 96, 56, 10 }, { 27, 79, 45 },
                { 74, 41, 54 }, { 23, 60, 26 }, { 56, 71, 54 }, { 82, 88, 58 }, { 49, 54, 29 },
                { 57, 71, 48 }, { 52, 76, 38 }, { 62, 32, 45 }, { 75, 70, 56 }, { 29, 7, 36 },
                { 71, 53, 57 }, { 72, 21, 34 }, { 15, 6, 34 }, { 34, 59, 49 }, { 67, 60, 53 },
                { 85, 73, 13 }, { 14, 90, 33 }, { 21, 34, 52 }, { 43, 13, 46 }, { 94, 50, 34 },
                { 29, 19, 15 }, { 14, 11, 54 }, { 22, 2, 55 }, { 57, 49, 56 }, { 30, 36, 31 },
                { 25, 5, 17 }, { 41, 76, 44 }, { 77, 97, 47 }, { 14, 71, 37 }, { 10, 34, 44 },
                { 93, 41, 39 }, { 81, 31, 42 }, { 40, 67, 10 }, { 69, 82, 18 }, { 62, 50, 15 },
                { 60, 29, 12 }, { 74, 47, 21 }, { 76, 47, 40 }, { 46, 100, 46 }, { 37, 4, 48 },
                { 26, 35, 29 }, { 41, 8, 11 }, { 26, 92, 36 }, { 71, 63, 52 }, { 78, 71, 45 },
                { 48, 68, 24 }, { 26, 44, 27 }, { 93, 11, 26 }, { 67, 20, 33 }, { 99, 37, 22 },
                { 29, 72, 25 }, { 64, 100, 22 }, { 56, 77, 18 }, { 98, 16, 51 }, { 71, 89, 11 },
                { 64, 90, 46 }, { 38, 20, 22 }, { 79, 45, 13 }, { 54, 26, 13 }, { 12, 49, 28 },
                { 70, 68, 30 }, { 27, 65, 48 }, { 84, 4, 59 }, { 34, 48, 28 }, { 100, 60, 21 },
                { 18, 35, 20 }, { 92, 4, 39 }, { 3, 95, 17 }, { 56, 79, 46 }, { 13, 65, 25 },
                { 43, 40, 27 }, { 61, 69, 35 }, { 22, 100, 55 }, { 54, 52, 46 }, { 97, 41, 46 },
                { 58, 5, 34 }, { 31, 98, 31 }, { 80, 78, 45 }, { 33, 73, 20 }, { 62, 24, 11 },
                { 6, 36, 50 }, { 75, 55, 34 }, { 57, 93, 37 }, { 37, 27, 30 }, { 42, 84, 23 },
                { 75, 63, 37 }, { 43, 24, 27 }, { 57, 20, 36 }, { 63, 48, 12 }, { 51, 82, 44 },
                { 38, 9, 28 }, { 75, 13, 39 }, { 69, 78, 33 }, { 81, 16, 28 }, { 12, 7, 31 },
                { 77, 65, 51 }, { 78, 56, 40 }, { 4, 86, 46 }, { 66, 2, 14 }, { 41, 44, 53 },
                { 50, 52, 32 }, { 61, 57, 16 }, { 71, 40, 58 }, { 25, 26, 27 }, { 77, 41, 17 },
                { 87, 77, 44 }, { 9, 49, 59 }, { 3, 60, 22 }, { 83, 92, 22 }, { 39, 47, 37 },
                { 67, 58, 53 }, { 19, 76, 48 }, { 44, 90, 22 }, { 72, 98, 19 }, { 79, 20, 30 },
                { 70, 74, 36 }, { 66, 63, 40 }, { 36, 70, 53 }, { 35, 30, 14 }, { 46, 42, 29 },
                { 85, 93, 58 }, { 31, 63, 57 }, { 36, 18, 42 }, { 45, 99, 36 }, { 88, 53, 58 },
                { 66, 78, 59 }, { 86, 13, 16 }, { 33, 49, 36 }, { 86, 55, 24 }, { 68, 76, 26 },
                { 48, 23, 30 }, { 100, 5, 33 }, { 65, 5, 56 }, { 23, 46, 38 }, { 10, 81, 31 },
                { 77, 98, 15 }, { 25, 78, 21 }, { 28, 4, 22 }, { 35, 55, 16 }, { 71, 65, 58 },
                { 81, 56, 21 }, { 89, 80, 25 }, { 97, 64, 46 }, { 36, 79, 22 }, { 40, 20, 39 },
                { 66, 67, 47 }, { 70, 21, 48 }, { 57, 30, 21 }, { 60, 38, 50 }, { 28, 34, 10 },
                { 90, 30, 13 }, { 100, 85, 32 }, { 49, 52, 57 }, { 45, 69, 49 }, { 29, 68, 37 },
                { 88, 63, 15 }, { 64, 96, 27 }, { 3, 98, 37 }, { 69, 91, 59 }, { 69, 89, 38 },
                { 89, 38, 28 }, { 43, 54, 59 }, { 4, 73, 59 }, { 94, 49, 50 }, { 80, 27, 32 },
                { 86, 92, 31 }, { 24, 54, 37 }, { 100, 82, 21 }, { 16, 33, 11 }, { 42, 8, 41 },
                { 68, 89, 11 }, { 51, 74, 43 }, { 16, 1, 21 }, { 41, 88, 37 }, { 16, 23, 32 },
                { 25, 49, 51 }, { 58, 50, 14 }, { 94, 46, 18 }, { 43, 93, 13 }, { 18, 12, 49 },
                { 46, 82, 38 }, { 90, 99, 16 }, { 54, 13, 17 }, { 98, 57, 35 }, { 9, 72, 33 },
                { 32, 64, 45 }, { 2, 4, 12 }, { 75, 15, 16 }, { 76, 18, 13 }, { 49, 72, 30 },
                { 56, 75, 42 }, { 11, 18, 41 }, { 76, 50, 39 }, { 49, 30, 20 }, { 23, 76, 49 },
                { 25, 6, 26 }, { 81, 66, 14 }, { 100, 53, 25 }, { 32, 39, 59 }, { 54, 93, 34 },
                { 30, 44, 40 }, { 29, 24, 27 }, { 67, 31, 52 }, { 94, 38, 23 }, { 67, 69, 22 },
                { 61, 9, 32 }, { 58, 88, 49 }, { 89, 61, 39 }, { 58, 84, 32 }, { 66, 50, 56 },
                { 95, 73, 17 }, { 63, 51, 20 }, { 10, 59, 38 }, { 87, 58, 44 }, { 25, 57, 55 },
                { 43, 88, 39 }, { 99, 24, 30 }, { 28, 5, 27 }, { 9, 44, 50 }, { 17, 16, 41 },
                { 69, 73, 34 }, { 98, 29, 56 }, { 44, 97, 17 }, { 9, 23, 17 }, { 75, 28, 12 },
                { 68, 41, 43 }, { 86, 67, 24 }, { 65, 44, 17 }, { 17, 82, 17 }, { 5, 41, 35 },
                { 35, 73, 18 }, { 64, 26, 46 }, { 25, 66, 23 }, { 88, 55, 30 }, { 99, 22, 24 },
                { 83, 26, 11 }, { 78, 88, 16 }, { 50, 25, 27 }, { 1, 96, 36 }, { 45, 48, 40 },
                { 21, 3, 17 }, { 84, 31, 11 }, { 84, 65, 40 }, { 24, 100, 35 }, { 94, 73, 14 },
                { 29, 70, 39 }, { 57, 6, 54 }, { 80, 99, 32 }, { 52, 96, 12 }, { 19, 16, 15 },
                { 10, 47, 49 }, { 82, 56, 29 }, { 5, 86, 13 }, { 21, 13, 30 }, { 26, 39, 31 },
                { 94, 97, 35 }, { 10, 62, 29 }, { 7, 28, 40 }, { 83, 6, 24 }, { 72, 92, 24 },
                { 95, 93, 48 }, { 48, 94, 56 }, { 76, 94, 43 }, { 49, 28, 47 }, { 74, 87, 52 },
                { 53, 3, 16 }, { 89, 32, 29 }, { 41, 33, 20 }, { 12, 94, 30 }, { 34, 91, 17 },
                { 86, 12, 16 }, { 96, 67, 35 }, { 44, 70, 33 }, { 60, 50, 37 }, { 25, 21, 38 },
                { 34, 89, 49 }, { 17, 9, 27 }, { 27, 99, 11 }, { 62, 38, 15 }, { 35, 12, 49 },
                { 3, 58, 54 }, { 66, 7, 53 }, { 17, 57, 56 }, { 90, 53, 18 }, { 47, 1, 43 },
                { 11, 5, 14 }, { 78, 45, 17 }, { 68, 52, 55 }, { 61, 100, 41 }, { 46, 57, 47 },
                { 100, 66, 41 }, { 40, 73, 21 }, { 95, 84, 14 }, { 29, 46, 41 }, { 86, 57, 39 },
                { 99, 55, 50 }, { 64, 70, 35 }, { 9, 6, 20 }, { 32, 47, 24 }, { 84, 57, 46 },
                { 74, 60, 55 }, { 87, 78, 14 }, { 7, 43, 28 }, { 16, 74, 50 }, { 11, 54, 57 },
                { 68, 92, 32 }, { 1, 41, 53 }, { 57, 48, 14 }, { 37, 96, 54 }, { 96, 25, 55 },
                { 43, 53, 34 }, { 60, 8, 24 }, { 37, 2, 25 }, { 87, 16, 48 }, { 92, 56, 48 },
                { 69, 2, 55 }, { 39, 38, 16 }, { 2, 29, 54 }, { 18, 37, 22 }, { 90, 97, 32 },
                { 10, 33, 18 }, { 22, 20, 19 }, { 34, 63, 15 }, { 64, 87, 11 }, { 31, 9, 27 },
                { 71, 81, 47 }, { 1, 18, 20 }, { 71, 96, 27 }, { 34, 44, 43 }, { 60, 40, 10 },
                { 67, 50, 43 }, { 85, 68, 26 }, { 33, 99, 44 }, { 87, 25, 43 }, { 82, 24, 41 },
                { 72, 87, 20 }, { 77, 66, 29 }, { 82, 12, 13 }, { 46, 21, 35 }, { 65, 70, 16 },
                { 23, 83, 25 }, { 54, 37, 28 }, { 42, 97, 22 }, { 21, 14, 40 }, { 16, 14, 37 },
                { 84, 33, 53 }, { 92, 82, 13 }, { 48, 91, 29 }, { 30, 24, 51 }, { 44, 45, 25 },
                { 28, 44, 18 }, { 76, 93, 24 }, { 67, 49, 18 }, { 37, 35, 38 }, { 55, 22, 38 },
                { 77, 63, 31 }, { 92, 77, 53 }, { 99, 21, 16 }, { 2, 33, 20 }, { 91, 5, 37 },
                { 28, 79, 43 }, { 95, 54, 16 }, { 15, 28, 10 }, { 14, 19, 56 }, { 60, 98, 23 },
                { 93, 59, 35 }, { 12, 8, 22 }, { 49, 20, 46 }, { 39, 62, 14 }, { 52, 80, 46 },
                { 57, 68, 46 }, { 35, 58, 11 }, { 99, 77, 49 }, { 99, 95, 43 }, { 74, 9, 13 },
                { 55, 65, 30 }, { 6, 71, 25 }, { 10, 15, 19 }, { 83, 37, 41 }, { 43, 44, 27 },
                { 87, 54, 46 }, { 72, 56, 46 }, { 41, 43, 28 }, { 85, 91, 29 }, { 90, 33, 56 },
                { 38, 44, 26 }, { 16, 53, 18 }, { 82, 55, 23 }, { 62, 59, 47 }, { 94, 32, 59 },
                { 98, 18, 23 }, { 87, 92, 45 }, { 40, 7, 39 }, { 99, 9, 33 }, { 10, 97, 43 },
                { 92, 37, 12 }, { 9, 58, 42 }, { 29, 18, 28 }, { 89, 12, 32 }, { 75, 99, 43 },
                { 42, 15, 16 }, { 20, 25, 25 }, { 50, 79, 30 }, { 30, 93, 10 }, { 16, 64, 45 },
                { 13, 47, 10 }, { 47, 34, 42 }, { 76, 79, 44 }, { 7, 97, 29 }, { 15, 85, 28 },
                { 98, 66, 12 }, { 17, 49, 40 }, { 84, 69, 33 }, { 80, 44, 16 }, { 90, 68, 52 },
                { 15, 25, 41 }, { 75, 36, 48 }, { 94, 3, 28 }, { 74, 42, 45 }, { 58, 80, 26 },
                { 86, 91, 18 }, { 77, 86, 55 }, { 15, 1, 11 }, { 78, 35, 39 }, { 15, 39, 21 },
                { 95, 17, 27 }, { 99, 34, 44 }, { 96, 91, 41 }, { 1, 88, 30 }, { 49, 10, 27 },
                { 59, 1, 36 }, { 33, 63, 41 }, { 24, 79, 41 }, { 44, 62, 25 }, { 84, 40, 49 },
                { 16, 60, 26 }, { 99, 47, 39 }, { 69, 55, 10 }, { 26, 11, 55 }, { 40, 97, 34 },
                { 6, 11, 31 }, { 14, 54, 54 }, { 23, 81, 26 }, { 18, 53, 58 }, { 80, 69, 38 },
                { 52, 32, 13 }, { 74, 48, 12 }, { 12, 17, 55 }, { 57, 38, 57 }, { 96, 82, 30 },
                { 9, 13, 27 }, { 21, 37, 42 }, { 81, 36, 17 }, { 77, 31, 18 }, { 97, 38, 19 },
                { 61, 16, 17 }, { 80, 40, 14 }, { 49, 44, 16 }, { 10, 38, 35 }, { 63, 84, 26 },
                { 87, 37, 36 }, { 18, 92, 21 }, { 33, 62, 25 }, { 76, 8, 25 }, { 49, 56, 46 },
                { 16, 54, 39 }, { 16, 7, 46 }, { 16, 55, 19 }, { 19, 43, 43 }, { 71, 79, 18 },
                { 70, 15, 13 }, { 52, 99, 37 }, { 22, 45, 44 }, { 2, 88, 47 }, { 28, 97, 55 },
                { 8, 90, 14 }, { 17, 7, 55 }, { 14, 38, 10 }, { 91, 32, 19 }, { 31, 78, 40 },
                { 14, 95, 18 }, { 67, 74, 16 }, { 58, 47, 17 }, { 60, 28, 23 }, { 41, 13, 24 },
                { 71, 13, 49 }, { 10, 99, 46 }, { 10, 3, 26 }, { 34, 24, 20 }, { 61, 30, 56 },
                { 79, 47, 19 }, { 32, 30, 37 }, { 51, 65, 21 }, { 63, 68, 50 }, { 98, 4, 14 },
                { 72, 59, 23 }, { 100, 56, 16 }, { 10, 74, 16 }, { 3, 8, 21 }, { 54, 36, 31 } };
        return constructUndirectedGraph(edges);
    }

    public Graph<Integer, DefaultWeightedEdge> getUndirectedN9()
    {
        int[][] edges = { { 1, 2, 0 }, { 2, 3, 1 }, { 3, 4, 0 }, { 4, 1, 1 }, { 1, 5, 1 },
            { 4, 5, 1 }, { 6, 2, 1 }, { 3, 6, 1 } };
        return constructUndirectedGraph(edges);
    }

    public Graph<Integer, DefaultWeightedEdge> generateUndirectedGraph()
    {
        GraphGenerator<Integer, DefaultWeightedEdge, Integer> randomGraphGenerator =
            new GnmRandomGraphGenerator<>(100, 500);
        Random rand = new Random();
        SimpleWeightedGraph<Integer, DefaultWeightedEdge> undirectedGraph =
            new SimpleWeightedGraph<>(
                SupplierUtil.createIntegerSupplier(), SupplierUtil.DEFAULT_WEIGHTED_EDGE_SUPPLIER);
        randomGraphGenerator.generateGraph(undirectedGraph);
        undirectedGraph
            .edgeSet().stream().forEach(e -> undirectedGraph.setEdgeWeight(e, rand.nextInt(100)));
        return undirectedGraph;
    }
}
