/* PROGRAM:     eggsh
 * FILE:        $Header: /home/egg/src/RCS/genlib.c,v 1.6 1999/02/28 20:05:18 ghn Exp $
 * PURPOSE:     General library
 * AUTHOR:      Greg Nelson
 * DATE:        98-05-09
 *
 * REVISED:
 * $Log: genlib.c,v $
 * Revision 1.6  1999/02/28 20:05:18  ghn
 * Version 5.1: Changed dquad2sockaddr interface to support ip/mm mask,
 * created hl2dquad to translate host-long to dotted quad, and modified
 * sockaddr2dquad to use this.
 *
 * Revision 1.5  1998/12/31 22:07:56  ghn
 * Rev 5 code: includes multi-reg support, HTML, etc.
 *
 * Revision 1.4  1998/08/03 20:43:35  kelvin
 * File byte-order independence.
 *
 * Revision 1.3  1998/08/01  18:51:25  ghn
 * Added John's byte-order-independence changes.
 *
 * Revision 1.2  1998/08/01 17:07:29  ghn
 * Casting fixes from John plus better parsing.
 *
 * Revision 1.1  1998/07/21 11:38:15  ghn
 * Initial revision
 * 
 * Copyright 1998 - Greg Nelson
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/utsname.h>
#include "global.h"
#include "genlib.h"

int32 getzulutime(struct timeval *ztv) {
  struct timeval ttv;

  if (ztv) {
    gettimeofday(ztv, NULL);
    return ztv->tv_sec;
  } else {
    gettimeofday(&ttv, NULL);
    return ttv.tv_sec;
  }
}

/* Get time difference (tv1-tv2) in msec.  Only works if less than 24
   days have passed. */
int32 deltams(struct timeval *tv1, struct timeval *tv2) {
  return (tv1->tv_sec - tv2->tv_sec) * 1000L +
         (tv1->tv_usec - tv2->tv_usec) / 1000L;
}

/* Look up an EGG id.  If we can't get it from environment, do a nutty
   thing using CRC of uname block, which should be pretty unique. */
uint16 GetID(void) {
  char *idstr;
  struct utsname utsname;

  if ((idstr = getenv("EGGID")) != NULL) {
    return atoi(idstr);
  } else {
    uname(&utsname);
    return BlockCRC16((byte *)(&utsname), sizeof(struct utsname));
  }
}

char *Packetize(EggCarton *src) {
  char *rbuf;
  uint32 lbuf;
  uint16 pktsize, rec;
  char *pktP;

  pktsize = (7 * sizeof(uint16)) + sizeof(trial) +	    /* Header */
	    sizeof(char) +				    /* ...including pad byte for trialsz */
	    (src->hdr.numrec * (sizeof(uint32) +	    /* Trial data */
		src->hdr.samp_rec * sizeof(trial))) +
	    sizeof(uint32);				    /* CRC and terminator */
  rbuf = pktP = (char *) malloc(pktsize);

  /* Assemble header fields into data packet. */

  packShort(src->hdr.type = DATA_PACKET);
  packShort(src->hdr.pktsize = pktsize);
  packShort(src->hdr.eggid);
  packShort(src->hdr.samp_rec);
  packShort(src->hdr.sec_rec);
  packShort(src->hdr.rec_pkt);
  packByte(0);			      /* Pad byte in case we want to expand trialsz */
  packByte(src->hdr.trialsz);
  packShort(src->hdr.numrec);

  /* Append data records to packet. */

  for (rec = 0; rec < src->hdr.numrec; rec++) {
    packLong(src->records[rec].timestamp);
    packBytes(&(src->records[rec].trials), src->hdr.samp_rec);
  }
  /* Get CRC, pack into base(32,32,64) notation, and add tag byte (0xFF) */
  lbuf = BlockCRC16((byte *) rbuf, pktP - rbuf);
  lbuf = ((lbuf & 0xF800) << 13) |
	 ((lbuf & 0x07C0) << 10) |
	 ((lbuf & 0x003F) << 8) |
	 (0x00FF);
  packLong(lbuf);
  if ((pktP - rbuf) != pktsize) {
    fprintf(stderr, "Length mismatch assembling packet.  Computed: %d, actually packed: %d.\n",
	pktsize, pktP - rbuf);
  }
  return rbuf;
}

int32 Unpacketize(EggCarton *dst, char *src) {
  char *pktP = src;
  char pad;
  uint16 rec;
  uint32 lbuf, filecrc;

  /* Unpack the portable header into a host-order and aligned
     EggHeader packet. */

  unpackShort(dst->hdr.type);
  unpackShort(dst->hdr.pktsize);
  unpackShort(dst->hdr.eggid);
  unpackShort(dst->hdr.samp_rec);
  unpackShort(dst->hdr.sec_rec);
  unpackShort(dst->hdr.rec_pkt);
  unpackByte(pad);		      /* Pad in case we later grow trialsz */
  unpackByte(dst->hdr.trialsz);
  unpackShort(dst->hdr.numrec);

  if (dst->hdr.type != DATA_PACKET) {
#ifdef DEBUG
    fprintf(stderr, "Invalid header type 0x%04X in packet read from file.\n", dst->hdr.type);
#endif
    return -1;
  }

  /* Unpack the data records from the file packet. */

  for (rec = 0; rec < dst->hdr.numrec; rec++) {
    unpackLong(dst->records[rec].timestamp);
    /* Assumes sizeof(trial) = 1 */
    unpackBytes(&(dst->records[rec].trials), dst->hdr.samp_rec);
  }

  /* Compute the CRC, reassemble into record terminator,
     and compare with terminator in file. */

  lbuf = BlockCRC16((byte *) src, pktP - src);
  lbuf = ((lbuf & 0xF800) << 13) |
	 ((lbuf & 0x07C0) << 10) |
	 ((lbuf & 0x003F) << 8) |
	 (0x00FF);

  unpackLong(filecrc);

  if (lbuf != filecrc) {
#ifdef DEBUG
    fprintf(stderr, "Bad CRC in packet read from file.  Read 0x%08lX, computed 0x%08lX.\n", filecrc, lbuf);
#endif
    return -2;
  }

  if (dst->hdr.pktsize != (pktP - src)) {
#ifdef DEBUG
    fprintf(stderr, "Length mismatch decoding packet.  Header: %d, length decoded: %d.\n",
      dst->hdr.pktsize, pktP - src);
#endif
    return -1;
  }

  /* One final little tweak.  Since we included a pad byte to allow
     for growth in trialsz, hdr.pktsize will include it.  Subtract
     one to hide the existence of the pad in the file from the
     caller.  In all probability the caller isn't going to look at
     pktsize, but you can't be too careful. */

  dst->hdr.pktsize--;

  return 0;
}

void Parse(char *input, int *argc, char *argv[]) {
  char *tp;

  *argc = 0;

  tp = strtok(input, " \t\n");
  while (tp != NULL && *argc < MAX_PARSE) {
    argv[*argc] = tp;
    *argc += 1;
    tp = strtok(NULL, " \t\n");
  }
}

char *mallocpy(char *input) {
  char *res;

  res = (char *)malloc(1+strlen(input));
  if (res) strcpy(res, input);
  return res;
}

void dquad2sockaddr(struct sockaddr_in *sinp, int16 *mask, char *dquad) {
  char *tp, *loser;
  long saddr;
  short qcount;

  loser = mallocpy(dquad);

  tp = strtok(loser, ".");
  for (qcount = 0, saddr = 0; qcount < 4 && tp != NULL; qcount++) {
    saddr = (saddr << 8) | (atoi(tp) & 0xFF);
    tp = strtok(NULL, ".");
  }
  if (mask) {
    *mask = 32;
    strcpy(loser, dquad);
    tp = strtok(loser, "/");
    if (tp) {
      tp = strtok(NULL, "/");
      if (tp) *mask = atoi(tp);
    }
  }
  free(loser);

  sinp->sin_family = AF_INET;
  sinp->sin_port = 0;		/* To be filled in later */
  sinp->sin_addr.s_addr = htonl(saddr);
}

char *sockaddr2dquad(struct sockaddr_in *sinp) {
  long saddr;
  
  saddr = ntohl(sinp->sin_addr.s_addr);
  return hl2dquad(saddr);
}

char *hl2dquad(long addr) {
  static char resout[16];

  sprintf(resout, "%ld.%ld.%ld.%ld",
	  (addr >> 24) & 0xFF, (addr >> 16) & 0xFF,
	  (addr >> 8) & 0xFF, addr & 0xFF);
  
  return resout;
}
