/*
 * AAC Spectral Band Replication decoding data
 * Copyright (c) 2008-2009 Robert Swain ( rob opendot cl )
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file
 * AAC Spectral Band Replication decoding data
 * @author Robert Swain ( rob opendot cl )
 */

#ifndef AVCODEC_AACSBRDATA_H
#define AVCODEC_AACSBRDATA_H

#include <stdint.h>
#include "libavutil/mem_internal.h"
#include "aac_defines.h"

///< Huffman tables for SBR

static const uint8_t t_huffman_env_1_5dB_bits[121] = {
    18, 18, 18, 18, 18, 18, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19,
    19, 19, 17, 18, 16, 17, 18, 17,
    16, 16, 16, 16, 15, 14, 14, 13,
    13, 12, 11, 10,  9,  8,  7,  6,
     5,  4,  3,  2,  2,  3,  4,  5,
     6,  7,  8,  9, 10, 12, 13, 14,
    14, 15, 16, 17, 16, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19,
    19,
};

static const uint32_t t_huffman_env_1_5dB_codes[121] = {
    0x3ffd6, 0x3ffd7, 0x3ffd8, 0x3ffd9, 0x3ffda, 0x3ffdb, 0x7ffb8, 0x7ffb9,
    0x7ffba, 0x7ffbb, 0x7ffbc, 0x7ffbd, 0x7ffbe, 0x7ffbf, 0x7ffc0, 0x7ffc1,
    0x7ffc2, 0x7ffc3, 0x7ffc4, 0x7ffc5, 0x7ffc6, 0x7ffc7, 0x7ffc8, 0x7ffc9,
    0x7ffca, 0x7ffcb, 0x7ffcc, 0x7ffcd, 0x7ffce, 0x7ffcf, 0x7ffd0, 0x7ffd1,
    0x7ffd2, 0x7ffd3, 0x1ffe6, 0x3ffd4, 0x0fff0, 0x1ffe9, 0x3ffd5, 0x1ffe7,
    0x0fff1, 0x0ffec, 0x0ffed, 0x0ffee, 0x07ff4, 0x03ff9, 0x03ff7, 0x01ffa,
    0x01ff9, 0x00ffb, 0x007fc, 0x003fc, 0x001fd, 0x000fd, 0x0007d, 0x0003d,
    0x0001d, 0x0000d, 0x00005, 0x00001, 0x00000, 0x00004, 0x0000c, 0x0001c,
    0x0003c, 0x0007c, 0x000fc, 0x001fc, 0x003fd, 0x00ffa, 0x01ff8, 0x03ff6,
    0x03ff8, 0x07ff5, 0x0ffef, 0x1ffe8, 0x0fff2, 0x7ffd4, 0x7ffd5, 0x7ffd6,
    0x7ffd7, 0x7ffd8, 0x7ffd9, 0x7ffda, 0x7ffdb, 0x7ffdc, 0x7ffdd, 0x7ffde,
    0x7ffdf, 0x7ffe0, 0x7ffe1, 0x7ffe2, 0x7ffe3, 0x7ffe4, 0x7ffe5, 0x7ffe6,
    0x7ffe7, 0x7ffe8, 0x7ffe9, 0x7ffea, 0x7ffeb, 0x7ffec, 0x7ffed, 0x7ffee,
    0x7ffef, 0x7fff0, 0x7fff1, 0x7fff2, 0x7fff3, 0x7fff4, 0x7fff5, 0x7fff6,
    0x7fff7, 0x7fff8, 0x7fff9, 0x7fffa, 0x7fffb, 0x7fffc, 0x7fffd, 0x7fffe,
    0x7ffff,
};

static const uint8_t f_huffman_env_1_5dB_bits[121] = {
    19, 19, 20, 20, 20, 20, 20, 20,
    20, 19, 20, 20, 20, 20, 19, 20,
    19, 19, 20, 18, 20, 20, 20, 19,
    20, 20, 20, 19, 20, 19, 18, 19,
    18, 18, 17, 18, 17, 17, 17, 16,
    16, 16, 15, 15, 14, 13, 13, 12,
    12, 11, 10,  9,  9,  8,  7,  6,
     5,  4,  3,  2,  2,  3,  4,  5,
     6,  8,  8,  9, 10, 11, 11, 11,
    12, 12, 13, 13, 14, 14, 16, 16,
    17, 17, 18, 18, 18, 18, 18, 18,
    18, 20, 19, 20, 20, 20, 20, 20,
    20, 19, 20, 20, 20, 20, 19, 20,
    18, 20, 20, 19, 19, 20, 20, 20,
    20, 20, 20, 20, 20, 20, 20, 20,
    20,
};

static const uint32_t f_huffman_env_1_5dB_codes[121] = {
    0x7ffe7, 0x7ffe8, 0xfffd2, 0xfffd3, 0xfffd4, 0xfffd5, 0xfffd6, 0xfffd7,
    0xfffd8, 0x7ffda, 0xfffd9, 0xfffda, 0xfffdb, 0xfffdc, 0x7ffdb, 0xfffdd,
    0x7ffdc, 0x7ffdd, 0xfffde, 0x3ffe4, 0xfffdf, 0xfffe0, 0xfffe1, 0x7ffde,
    0xfffe2, 0xfffe3, 0xfffe4, 0x7ffdf, 0xfffe5, 0x7ffe0, 0x3ffe8, 0x7ffe1,
    0x3ffe0, 0x3ffe9, 0x1ffef, 0x3ffe5, 0x1ffec, 0x1ffed, 0x1ffee, 0x0fff4,
    0x0fff3, 0x0fff0, 0x07ff7, 0x07ff6, 0x03ffa, 0x01ffa, 0x01ff9, 0x00ffa,
    0x00ff8, 0x007f9, 0x003fb, 0x001fc, 0x001fa, 0x000fb, 0x0007c, 0x0003c,
    0x0001c, 0x0000c, 0x00005, 0x00001, 0x00000, 0x00004, 0x0000d, 0x0001d,
    0x0003d, 0x000fa, 0x000fc, 0x001fb, 0x003fa, 0x007f8, 0x007fa, 0x007fb,
    0x00ff9, 0x00ffb, 0x01ff8, 0x01ffb, 0x03ff8, 0x03ff9, 0x0fff1, 0x0fff2,
    0x1ffea, 0x1ffeb, 0x3ffe1, 0x3ffe2, 0x3ffea, 0x3ffe3, 0x3ffe6, 0x3ffe7,
    0x3ffeb, 0xfffe6, 0x7ffe2, 0xfffe7, 0xfffe8, 0xfffe9, 0xfffea, 0xfffeb,
    0xfffec, 0x7ffe3, 0xfffed, 0xfffee, 0xfffef, 0xffff0, 0x7ffe4, 0xffff1,
    0x3ffec, 0xffff2, 0xffff3, 0x7ffe5, 0x7ffe6, 0xffff4, 0xffff5, 0xffff6,
    0xffff7, 0xffff8, 0xffff9, 0xffffa, 0xffffb, 0xffffc, 0xffffd, 0xffffe,
    0xfffff,
};

static const uint8_t t_huffman_env_bal_1_5dB_bits[49] = {
    16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 12, 11,  9,  7,  5,  3,
     1,  2,  4,  6,  8, 11, 12, 15,
    16, 16, 16, 16, 16, 16, 16, 17,
    17, 17, 17, 17, 17, 17, 17, 17,
    17,
};

static const uint32_t t_huffman_env_bal_1_5dB_codes[49] = {
    0x0ffe4, 0x0ffe5, 0x0ffe6, 0x0ffe7, 0x0ffe8, 0x0ffe9, 0x0ffea, 0x0ffeb,
    0x0ffec, 0x0ffed, 0x0ffee, 0x0ffef, 0x0fff0, 0x0fff1, 0x0fff2, 0x0fff3,
    0x0fff4, 0x0ffe2, 0x00ffc, 0x007fc, 0x001fe, 0x0007e, 0x0001e, 0x00006,
    0x00000, 0x00002, 0x0000e, 0x0003e, 0x000fe, 0x007fd, 0x00ffd, 0x07ff0,
    0x0ffe3, 0x0fff5, 0x0fff6, 0x0fff7, 0x0fff8, 0x0fff9, 0x0fffa, 0x1fff6,
    0x1fff7, 0x1fff8, 0x1fff9, 0x1fffa, 0x1fffb, 0x1fffc, 0x1fffd, 0x1fffe,
    0x1ffff,
};

static const uint8_t f_huffman_env_bal_1_5dB_bits[49] = {
    18, 18, 18, 18, 18, 18, 18, 18,
    18, 18, 18, 18, 18, 18, 18, 16,
    17, 14, 11, 11,  8,  7,  4,  2,
     1,  3,  5,  6,  9, 11, 12, 15,
    16, 18, 18, 18, 18, 18, 18, 18,
    18, 18, 18, 18, 18, 18, 18, 19,
    19,
};

static const uint32_t f_huffman_env_bal_1_5dB_codes[49] = {
    0x3ffe2, 0x3ffe3, 0x3ffe4, 0x3ffe5, 0x3ffe6, 0x3ffe7, 0x3ffe8, 0x3ffe9,
    0x3ffea, 0x3ffeb, 0x3ffec, 0x3ffed, 0x3ffee, 0x3ffef, 0x3fff0, 0x0fff7,
    0x1fff0, 0x03ffc, 0x007fe, 0x007fc, 0x000fe, 0x0007e, 0x0000e, 0x00002,
    0x00000, 0x00006, 0x0001e, 0x0003e, 0x001fe, 0x007fd, 0x00ffe, 0x07ffa,
    0x0fff6, 0x3fff1, 0x3fff2, 0x3fff3, 0x3fff4, 0x3fff5, 0x3fff6, 0x3fff7,
    0x3fff8, 0x3fff9, 0x3fffa, 0x3fffb, 0x3fffc, 0x3fffd, 0x3fffe, 0x7fffe,
    0x7ffff,
};

static const uint8_t t_huffman_env_3_0dB_bits[63] = {
    18, 18, 19, 19, 19, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19,
    19, 17, 16, 16, 16, 14, 14, 14,
    13, 12, 11,  8,  6,  4,  2,  1,
     3,  5,  7,  9, 11, 13, 14, 14,
    15, 16, 17, 18, 19, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19, 19,
    19, 19, 19, 19, 19, 19, 19,
};

static const uint32_t t_huffman_env_3_0dB_codes[63] = {
    0x3ffed, 0x3ffee, 0x7ffde, 0x7ffdf, 0x7ffe0, 0x7ffe1, 0x7ffe2, 0x7ffe3,
    0x7ffe4, 0x7ffe5, 0x7ffe6, 0x7ffe7, 0x7ffe8, 0x7ffe9, 0x7ffea, 0x7ffeb,
    0x7ffec, 0x1fff4, 0x0fff7, 0x0fff9, 0x0fff8, 0x03ffb, 0x03ffa, 0x03ff8,
    0x01ffa, 0x00ffc, 0x007fc, 0x000fe, 0x0003e, 0x0000e, 0x00002, 0x00000,
    0x00006, 0x0001e, 0x0007e, 0x001fe, 0x007fd, 0x01ffb, 0x03ff9, 0x03ffc,
    0x07ffa, 0x0fff6, 0x1fff5, 0x3ffec, 0x7ffed, 0x7ffee, 0x7ffef, 0x7fff0,
    0x7fff1, 0x7fff2, 0x7fff3, 0x7fff4, 0x7fff5, 0x7fff6, 0x7fff7, 0x7fff8,
    0x7fff9, 0x7fffa, 0x7fffb, 0x7fffc, 0x7fffd, 0x7fffe, 0x7ffff,
};

static const uint8_t f_huffman_env_3_0dB_bits[63] = {
    20, 20, 20, 20, 20, 20, 20, 18,
    19, 19, 19, 19, 18, 18, 20, 19,
    17, 18, 17, 16, 16, 15, 14, 12,
    11, 10,  9,  8,  6,  4,  2,  1,
     3,  5,  8,  9, 10, 11, 12, 13,
    14, 15, 15, 16, 16, 17, 17, 18,
    18, 18, 20, 19, 19, 19, 20, 19,
    19, 20, 20, 20, 20, 20, 20,
};

static const uint32_t f_huffman_env_3_0dB_codes[63] = {
    0xffff0, 0xffff1, 0xffff2, 0xffff3, 0xffff4, 0xffff5, 0xffff6, 0x3fff3,
    0x7fff5, 0x7ffee, 0x7ffef, 0x7fff6, 0x3fff4, 0x3fff2, 0xffff7, 0x7fff0,
    0x1fff5, 0x3fff0, 0x1fff4, 0x0fff7, 0x0fff6, 0x07ff8, 0x03ffb, 0x00ffd,
    0x007fd, 0x003fd, 0x001fd, 0x000fd, 0x0003e, 0x0000e, 0x00002, 0x00000,
    0x00006, 0x0001e, 0x000fc, 0x001fc, 0x003fc, 0x007fc, 0x00ffc, 0x01ffc,
    0x03ffa, 0x07ff9, 0x07ffa, 0x0fff8, 0x0fff9, 0x1fff6, 0x1fff7, 0x3fff5,
    0x3fff6, 0x3fff1, 0xffff8, 0x7fff1, 0x7fff2, 0x7fff3, 0xffff9, 0x7fff7,
    0x7fff4, 0xffffa, 0xffffb, 0xffffc, 0xffffd, 0xffffe, 0xfffff,
};

static const uint8_t t_huffman_env_bal_3_0dB_bits[25] = {
    13, 13, 13, 13, 13, 13, 13, 12,
     8,  7,  4,  3,  1,  2,  5,  6,
     9, 13, 13, 13, 13, 13, 13, 14,
    14,
};

static const uint16_t t_huffman_env_bal_3_0dB_codes[25] = {
    0x1ff2, 0x1ff3, 0x1ff4, 0x1ff5, 0x1ff6, 0x1ff7, 0x1ff8, 0x0ff8,
    0x00fe, 0x007e, 0x000e, 0x0006, 0x0000, 0x0002, 0x001e, 0x003e,
    0x01fe, 0x1ff9, 0x1ffa, 0x1ffb, 0x1ffc, 0x1ffd, 0x1ffe, 0x3ffe,
    0x3fff,
};

static const uint8_t f_huffman_env_bal_3_0dB_bits[25] = {
    13, 13, 13, 13, 13, 14, 14, 11,
     8,  7,  4,  2,  1,  3,  5,  6,
     9, 12, 13, 14, 14, 14, 14, 14,
    14,
};

static const uint16_t f_huffman_env_bal_3_0dB_codes[25] = {
    0x1ff7, 0x1ff8, 0x1ff9, 0x1ffa, 0x1ffb, 0x3ff8, 0x3ff9, 0x07fc,
    0x00fe, 0x007e, 0x000e, 0x0002, 0x0000, 0x0006, 0x001e, 0x003e,
    0x01fe, 0x0ffa, 0x1ff6, 0x3ffa, 0x3ffb, 0x3ffc, 0x3ffd, 0x3ffe,
    0x3fff,
};

static const uint8_t t_huffman_noise_3_0dB_bits[63] = {
    13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 11,  8,  6,  4,  3,  1,
     2,  5,  8, 10, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 14, 14,
};

static const uint16_t t_huffman_noise_3_0dB_codes[63] = {
    0x1fce, 0x1fcf, 0x1fd0, 0x1fd1, 0x1fd2, 0x1fd3, 0x1fd4, 0x1fd5,
    0x1fd6, 0x1fd7, 0x1fd8, 0x1fd9, 0x1fda, 0x1fdb, 0x1fdc, 0x1fdd,
    0x1fde, 0x1fdf, 0x1fe0, 0x1fe1, 0x1fe2, 0x1fe3, 0x1fe4, 0x1fe5,
    0x1fe6, 0x1fe7, 0x07f2, 0x00fd, 0x003e, 0x000e, 0x0006, 0x0000,
    0x0002, 0x001e, 0x00fc, 0x03f8, 0x1fcc, 0x1fe8, 0x1fe9, 0x1fea,
    0x1feb, 0x1fec, 0x1fcd, 0x1fed, 0x1fee, 0x1fef, 0x1ff0, 0x1ff1,
    0x1ff2, 0x1ff3, 0x1ff4, 0x1ff5, 0x1ff6, 0x1ff7, 0x1ff8, 0x1ff9,
    0x1ffa, 0x1ffb, 0x1ffc, 0x1ffd, 0x1ffe, 0x3ffe, 0x3fff,
};

static const uint8_t t_huffman_noise_bal_3_0dB_bits[25] = {
    8, 8, 8, 8, 8, 8, 8, 8,
    8, 8, 5, 2, 1, 3, 6, 8,
    8, 8, 8, 8, 8, 8, 8, 8,
    8,
};

static const uint8_t t_huffman_noise_bal_3_0dB_codes[25] = {
    0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3,
    0xf4, 0xf5, 0x1c, 0x02, 0x00, 0x06, 0x3a, 0xf6,
    0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe,
    0xff,
};

static const int8_t sbr_offset[6][16] = {
    {-8, -7, -6, -5, -4, -3, -2, -1,  0,  1,  2,  3,  4,  5,  6,  7}, //             fs_sbr  = 16000 Hz
    {-5, -4, -3, -2, -1,  0,  1,  2,  3,  4,  5,  6,  7,  9, 11, 13}, //             fs_sbr  = 22050 Hz
    {-5, -3, -2, -1,  0,  1,  2,  3,  4,  5,  6,  7,  9, 11, 13, 16}, //             fs_sbr  = 24000 Hz
    {-6, -4, -2, -1,  0,  1,  2,  3,  4,  5,  6,  7,  9, 11, 13, 16}, //             fs_sbr  = 32000 Hz
    {-4, -2, -1,  0,  1,  2,  3,  4,  5,  6,  7,  9, 11, 13, 16, 20}, // 44100 Hz <= fs_sbr <= 64000 Hz
    {-2, -1,  0,  1,  2,  3,  4,  5,  6,  7,  9, 11, 13, 16, 20, 24}, // 64000 Hz <  fs_sbr
};

///< window coefficients for analysis/synthesis QMF banks
static DECLARE_ALIGNED(32, INTFLOAT, sbr_qmf_window_ds)[320];
/* This table contains redundancy: It is symmetric about the entry #320
 * with the exception of entries 384 and 512 which are negated. */
static const DECLARE_ALIGNED(32, INTFLOAT, sbr_qmf_window_us)[640] = {
     Q31( 0.0000000000f), Q31(-0.0005525286f), Q31(-0.0005617692f), Q31(-0.0004947518f),
     Q31(-0.0004875227f), Q31(-0.0004893791f), Q31(-0.0005040714f), Q31(-0.0005226564f),
     Q31(-0.0005466565f), Q31(-0.0005677802f), Q31(-0.0005870930f), Q31(-0.0006132747f),
     Q31(-0.0006312493f), Q31(-0.0006540333f), Q31(-0.0006777690f), Q31(-0.0006941614f),
     Q31(-0.0007157736f), Q31(-0.0007255043f), Q31(-0.0007440941f), Q31(-0.0007490598f),
     Q31(-0.0007681371f), Q31(-0.0007724848f), Q31(-0.0007834332f), Q31(-0.0007779869f),
     Q31(-0.0007803664f), Q31(-0.0007801449f), Q31(-0.0007757977f), Q31(-0.0007630793f),
     Q31(-0.0007530001f), Q31(-0.0007319357f), Q31(-0.0007215391f), Q31(-0.0006917937f),
     Q31(-0.0006650415f), Q31(-0.0006341594f), Q31(-0.0005946118f), Q31(-0.0005564576f),
     Q31(-0.0005145572f), Q31(-0.0004606325f), Q31(-0.0004095121f), Q31(-0.0003501175f),
     Q31(-0.0002896981f), Q31(-0.0002098337f), Q31(-0.0001446380f), Q31(-0.0000617334f),
     Q31( 0.0000134949f), Q31( 0.0001094383f), Q31( 0.0002043017f), Q31( 0.0002949531f),
     Q31( 0.0004026540f), Q31( 0.0005107388f), Q31( 0.0006239376f), Q31( 0.0007458025f),
     Q31( 0.0008608443f), Q31( 0.0009885988f), Q31( 0.0011250155f), Q31( 0.0012577884f),
     Q31( 0.0013902494f), Q31( 0.0015443219f), Q31( 0.0016868083f), Q31( 0.0018348265f),
     Q31( 0.0019841140f), Q31( 0.0021461583f), Q31( 0.0023017254f), Q31( 0.0024625616f),
     Q31( 0.0026201758f), Q31( 0.0027870464f), Q31( 0.0029469447f), Q31( 0.0031125420f),
     Q31( 0.0032739613f), Q31( 0.0034418874f), Q31( 0.0036008268f), Q31( 0.0037603922f),
     Q31( 0.0039207432f), Q31( 0.0040819753f), Q31( 0.0042264269f), Q31( 0.0043730719f),
     Q31( 0.0045209852f), Q31( 0.0046606460f), Q31( 0.0047932560f), Q31( 0.0049137603f),
     Q31( 0.0050393022f), Q31( 0.0051407353f), Q31( 0.0052461166f), Q31( 0.0053471681f),
     Q31( 0.0054196775f), Q31( 0.0054876040f), Q31( 0.0055475714f), Q31( 0.0055938023f),
     Q31( 0.0056220643f), Q31( 0.0056455196f), Q31( 0.0056389199f), Q31( 0.0056266114f),
     Q31( 0.0055917128f), Q31( 0.0055404363f), Q31( 0.0054753783f), Q31( 0.0053838975f),
     Q31( 0.0052715758f), Q31( 0.0051382275f), Q31( 0.0049839687f), Q31( 0.0048109469f),
     Q31( 0.0046039530f), Q31( 0.0043801861f), Q31( 0.0041251642f), Q31( 0.0038456408f),
     Q31( 0.0035401246f), Q31( 0.0032091885f), Q31( 0.0028446757f), Q31( 0.0024508540f),
     Q31( 0.0020274176f), Q31( 0.0015784682f), Q31( 0.0010902329f), Q31( 0.0005832264f),
     Q31( 0.0000276045f), Q31(-0.0005464280f), Q31(-0.0011568135f), Q31(-0.0018039472f),
     Q31(-0.0024826723f), Q31(-0.0031933778f), Q31(-0.0039401124f), Q31(-0.0047222596f),
     Q31(-0.0055337211f), Q31(-0.0063792293f), Q31(-0.0072615816f), Q31(-0.0081798233f),
     Q31(-0.0091325329f), Q31(-0.0101150215f), Q31(-0.0111315548f), Q31(-0.0121849995f),
     Q31( 0.0132718220f), Q31( 0.0143904666f), Q31( 0.0155405553f), Q31( 0.0167324712f),
     Q31( 0.0179433381f), Q31( 0.0191872431f), Q31( 0.0204531793f), Q31( 0.0217467550f),
     Q31( 0.0230680169f), Q31( 0.0244160992f), Q31( 0.0257875847f), Q31( 0.0271859429f),
     Q31( 0.0286072173f), Q31( 0.0300502657f), Q31( 0.0315017608f), Q31( 0.0329754081f),
     Q31( 0.0344620948f), Q31( 0.0359697560f), Q31( 0.0374812850f), Q31( 0.0390053679f),
     Q31( 0.0405349170f), Q31( 0.0420649094f), Q31( 0.0436097542f), Q31( 0.0451488405f),
     Q31( 0.0466843027f), Q31( 0.0482165720f), Q31( 0.0497385755f), Q31( 0.0512556155f),
     Q31( 0.0527630746f), Q31( 0.0542452768f), Q31( 0.0557173648f), Q31( 0.0571616450f),
     Q31( 0.0585915683f), Q31( 0.0599837480f), Q31( 0.0613455171f), Q31( 0.0626857808f),
     Q31( 0.0639715898f), Q31( 0.0652247106f), Q31( 0.0664367512f), Q31( 0.0676075985f),
     Q31( 0.0687043828f), Q31( 0.0697630244f), Q31( 0.0707628710f), Q31( 0.0717002673f),
     Q31( 0.0725682583f), Q31( 0.0733620255f), Q31( 0.0741003642f), Q31( 0.0747452558f),
     Q31( 0.0753137336f), Q31( 0.0758008358f), Q31( 0.0761992479f), Q31( 0.0764992170f),
     Q31( 0.0767093490f), Q31( 0.0768173975f), Q31( 0.0768230011f), Q31( 0.0767204924f),
     Q31( 0.0765050718f), Q31( 0.0761748321f), Q31( 0.0757305756f), Q31( 0.0751576255f),
     Q31( 0.0744664394f), Q31( 0.0736406005f), Q31( 0.0726774642f), Q31( 0.0715826364f),
     Q31( 0.0703533073f), Q31( 0.0689664013f), Q31( 0.0674525021f), Q31( 0.0657690668f),
     Q31( 0.0639444805f), Q31( 0.0619602779f), Q31( 0.0598166570f), Q31( 0.0575152691f),
     Q31( 0.0550460034f), Q31( 0.0524093821f), Q31( 0.0495978676f), Q31( 0.0466303305f),
     Q31( 0.0434768782f), Q31( 0.0401458278f), Q31( 0.0366418116f), Q31( 0.0329583930f),
     Q31( 0.0290824006f), Q31( 0.0250307561f), Q31( 0.0207997072f), Q31( 0.0163701258f),
     Q31( 0.0117623832f), Q31( 0.0069636862f), Q31( 0.0019765601f), Q31(-0.0032086896f),
     Q31(-0.0085711749f), Q31(-0.0141288827f), Q31(-0.0198834129f), Q31(-0.0258227288f),
     Q31(-0.0319531274f), Q31(-0.0382776572f), Q31(-0.0447806821f), Q31(-0.0514804176f),
     Q31(-0.0583705326f), Q31(-0.0654409853f), Q31(-0.0726943300f), Q31(-0.0801372934f),
     Q31(-0.0877547536f), Q31(-0.0955533352f), Q31(-0.1035329531f), Q31(-0.1116826931f),
     Q31(-0.1200077984f), Q31(-0.1285002850f), Q31(-0.1371551761f), Q31(-0.1459766491f),
     Q31(-0.1549607071f), Q31(-0.1640958855f), Q31(-0.1733808172f), Q31(-0.1828172548f),
     Q31(-0.1923966745f), Q31(-0.2021250176f), Q31(-0.2119735853f), Q31(-0.2219652696f),
     Q31(-0.2320690870f), Q31(-0.2423016884f), Q31(-0.2526480309f), Q31(-0.2631053299f),
     Q31(-0.2736634040f), Q31(-0.2843214189f), Q31(-0.2950716717f), Q31(-0.3059098575f),
     Q31(-0.3168278913f), Q31(-0.3278113727f), Q31(-0.3388722693f), Q31(-0.3499914122f),
     Q31( 0.3611589903f), Q31( 0.3723795546f), Q31( 0.3836350013f), Q31( 0.3949211761f),
     Q31( 0.4062317676f), Q31( 0.4175696896f), Q31( 0.4289119920f), Q31( 0.4402553754f),
     Q31( 0.4515996535f), Q31( 0.4629308085f), Q31( 0.4742453214f), Q31( 0.4855253091f),
     Q31( 0.4967708254f), Q31( 0.5079817500f), Q31( 0.5191234970f), Q31( 0.5302240895f),
     Q31( 0.5412553448f), Q31( 0.5522051258f), Q31( 0.5630789140f), Q31( 0.5738524131f),
     Q31( 0.5845403235f), Q31( 0.5951123086f), Q31( 0.6055783538f), Q31( 0.6159109932f),
     Q31( 0.6261242695f), Q31( 0.6361980107f), Q31( 0.6461269695f), Q31( 0.6559016302f),
     Q31( 0.6655139880f), Q31( 0.6749663190f), Q31( 0.6842353293f), Q31( 0.6933282376f),
     Q31( 0.7022388719f), Q31( 0.7109410426f), Q31( 0.7194462634f), Q31( 0.7277448900f),
     Q31( 0.7358211758f), Q31( 0.7436827863f), Q31( 0.7513137456f), Q31( 0.7587080760f),
     Q31( 0.7658674865f), Q31( 0.7727780881f), Q31( 0.7794287519f), Q31( 0.7858353120f),
     Q31( 0.7919735841f), Q31( 0.7978466413f), Q31( 0.8034485751f), Q31( 0.8087695004f),
     Q31( 0.8138191270f), Q31( 0.8185776004f), Q31( 0.8230419890f), Q31( 0.8272275347f),
     Q31( 0.8311038457f), Q31( 0.8346937361f), Q31( 0.8379717337f), Q31( 0.8409541392f),
     Q31( 0.8436238281f), Q31( 0.8459818469f), Q31( 0.8480315777f), Q31( 0.8497805198f),
     Q31( 0.8511971524f), Q31( 0.8523047035f), Q31( 0.8531020949f), Q31( 0.8535720573f),
     Q31( 0.8537385600f), Q31( 0.8535720573f), Q31( 0.8531020949f), Q31( 0.8523047035f),
     Q31( 0.8511971524f), Q31( 0.8497805198f), Q31( 0.8480315777f), Q31( 0.8459818469f),
     Q31( 0.8436238281f), Q31( 0.8409541392f), Q31( 0.8379717337f), Q31( 0.8346937361f),
     Q31( 0.8311038457f), Q31( 0.8272275347f), Q31( 0.8230419890f), Q31( 0.8185776004f),
     Q31( 0.8138191270f), Q31( 0.8087695004f), Q31( 0.8034485751f), Q31( 0.7978466413f),
     Q31( 0.7919735841f), Q31( 0.7858353120f), Q31( 0.7794287519f), Q31( 0.7727780881f),
     Q31( 0.7658674865f), Q31( 0.7587080760f), Q31( 0.7513137456f), Q31( 0.7436827863f),
     Q31( 0.7358211758f), Q31( 0.7277448900f), Q31( 0.7194462634f), Q31( 0.7109410426f),
     Q31( 0.7022388719f), Q31( 0.6933282376f), Q31( 0.6842353293f), Q31( 0.6749663190f),
     Q31( 0.6655139880f), Q31( 0.6559016302f), Q31( 0.6461269695f), Q31( 0.6361980107f),
     Q31( 0.6261242695f), Q31( 0.6159109932f), Q31( 0.6055783538f), Q31( 0.5951123086f),
     Q31( 0.5845403235f), Q31( 0.5738524131f), Q31( 0.5630789140f), Q31( 0.5522051258f),
     Q31( 0.5412553448f), Q31( 0.5302240895f), Q31( 0.5191234970f), Q31( 0.5079817500f),
     Q31( 0.4967708254f), Q31( 0.4855253091f), Q31( 0.4742453214f), Q31( 0.4629308085f),
     Q31( 0.4515996535f), Q31( 0.4402553754f), Q31( 0.4289119920f), Q31( 0.4175696896f),
     Q31( 0.4062317676f), Q31( 0.3949211761f), Q31( 0.3836350013f), Q31( 0.3723795546f),
    -Q31( 0.3611589903f), Q31(-0.3499914122f), Q31(-0.3388722693f), Q31(-0.3278113727f),
     Q31(-0.3168278913f), Q31(-0.3059098575f), Q31(-0.2950716717f), Q31(-0.2843214189f),
     Q31(-0.2736634040f), Q31(-0.2631053299f), Q31(-0.2526480309f), Q31(-0.2423016884f),
     Q31(-0.2320690870f), Q31(-0.2219652696f), Q31(-0.2119735853f), Q31(-0.2021250176f),
     Q31(-0.1923966745f), Q31(-0.1828172548f), Q31(-0.1733808172f), Q31(-0.1640958855f),
     Q31(-0.1549607071f), Q31(-0.1459766491f), Q31(-0.1371551761f), Q31(-0.1285002850f),
     Q31(-0.1200077984f), Q31(-0.1116826931f), Q31(-0.1035329531f), Q31(-0.0955533352f),
     Q31(-0.0877547536f), Q31(-0.0801372934f), Q31(-0.0726943300f), Q31(-0.0654409853f),
     Q31(-0.0583705326f), Q31(-0.0514804176f), Q31(-0.0447806821f), Q31(-0.0382776572f),
     Q31(-0.0319531274f), Q31(-0.0258227288f), Q31(-0.0198834129f), Q31(-0.0141288827f),
     Q31(-0.0085711749f), Q31(-0.0032086896f), Q31( 0.0019765601f), Q31( 0.0069636862f),
     Q31( 0.0117623832f), Q31( 0.0163701258f), Q31( 0.0207997072f), Q31( 0.0250307561f),
     Q31( 0.0290824006f), Q31( 0.0329583930f), Q31( 0.0366418116f), Q31( 0.0401458278f),
     Q31( 0.0434768782f), Q31( 0.0466303305f), Q31( 0.0495978676f), Q31( 0.0524093821f),
     Q31( 0.0550460034f), Q31( 0.0575152691f), Q31( 0.0598166570f), Q31( 0.0619602779f),
     Q31( 0.0639444805f), Q31( 0.0657690668f), Q31( 0.0674525021f), Q31( 0.0689664013f),
     Q31( 0.0703533073f), Q31( 0.0715826364f), Q31( 0.0726774642f), Q31( 0.0736406005f),
     Q31( 0.0744664394f), Q31( 0.0751576255f), Q31( 0.0757305756f), Q31( 0.0761748321f),
     Q31( 0.0765050718f), Q31( 0.0767204924f), Q31( 0.0768230011f), Q31( 0.0768173975f),
     Q31( 0.0767093490f), Q31( 0.0764992170f), Q31( 0.0761992479f), Q31( 0.0758008358f),
     Q31( 0.0753137336f), Q31( 0.0747452558f), Q31( 0.0741003642f), Q31( 0.0733620255f),
     Q31( 0.0725682583f), Q31( 0.0717002673f), Q31( 0.0707628710f), Q31( 0.0697630244f),
     Q31( 0.0687043828f), Q31( 0.0676075985f), Q31( 0.0664367512f), Q31( 0.0652247106f),
     Q31( 0.0639715898f), Q31( 0.0626857808f), Q31( 0.0613455171f), Q31( 0.0599837480f),
     Q31( 0.0585915683f), Q31( 0.0571616450f), Q31( 0.0557173648f), Q31( 0.0542452768f),
     Q31( 0.0527630746f), Q31( 0.0512556155f), Q31( 0.0497385755f), Q31( 0.0482165720f),
     Q31( 0.0466843027f), Q31( 0.0451488405f), Q31( 0.0436097542f), Q31( 0.0420649094f),
     Q31( 0.0405349170f), Q31( 0.0390053679f), Q31( 0.0374812850f), Q31( 0.0359697560f),
     Q31( 0.0344620948f), Q31( 0.0329754081f), Q31( 0.0315017608f), Q31( 0.0300502657f),
     Q31( 0.0286072173f), Q31( 0.0271859429f), Q31( 0.0257875847f), Q31( 0.0244160992f),
     Q31( 0.0230680169f), Q31( 0.0217467550f), Q31( 0.0204531793f), Q31( 0.0191872431f),
     Q31( 0.0179433381f), Q31( 0.0167324712f), Q31( 0.0155405553f), Q31( 0.0143904666f),
    -Q31( 0.0132718220f), Q31(-0.0121849995f), Q31(-0.0111315548f), Q31(-0.0101150215f),
     Q31(-0.0091325329f), Q31(-0.0081798233f), Q31(-0.0072615816f), Q31(-0.0063792293f),
     Q31(-0.0055337211f), Q31(-0.0047222596f), Q31(-0.0039401124f), Q31(-0.0031933778f),
     Q31(-0.0024826723f), Q31(-0.0018039472f), Q31(-0.0011568135f), Q31(-0.0005464280f),
     Q31( 0.0000276045f), Q31( 0.0005832264f), Q31( 0.0010902329f), Q31( 0.0015784682f),
     Q31( 0.0020274176f), Q31( 0.0024508540f), Q31( 0.0028446757f), Q31( 0.0032091885f),
     Q31( 0.0035401246f), Q31( 0.0038456408f), Q31( 0.0041251642f), Q31( 0.0043801861f),
     Q31( 0.0046039530f), Q31( 0.0048109469f), Q31( 0.0049839687f), Q31( 0.0051382275f),
     Q31( 0.0052715758f), Q31( 0.0053838975f), Q31( 0.0054753783f), Q31( 0.0055404363f),
     Q31( 0.0055917128f), Q31( 0.0056266114f), Q31( 0.0056389199f), Q31( 0.0056455196f),
     Q31( 0.0056220643f), Q31( 0.0055938023f), Q31( 0.0055475714f), Q31( 0.0054876040f),
     Q31( 0.0054196775f), Q31( 0.0053471681f), Q31( 0.0052461166f), Q31( 0.0051407353f),
     Q31( 0.0050393022f), Q31( 0.0049137603f), Q31( 0.0047932560f), Q31( 0.0046606460f),
     Q31( 0.0045209852f), Q31( 0.0043730719f), Q31( 0.0042264269f), Q31( 0.0040819753f),
     Q31( 0.0039207432f), Q31( 0.0037603922f), Q31( 0.0036008268f), Q31( 0.0034418874f),
     Q31( 0.0032739613f), Q31( 0.0031125420f), Q31( 0.0029469447f), Q31( 0.0027870464f),
     Q31( 0.0026201758f), Q31( 0.0024625616f), Q31( 0.0023017254f), Q31( 0.0021461583f),
     Q31( 0.0019841140f), Q31( 0.0018348265f), Q31( 0.0016868083f), Q31( 0.0015443219f),
     Q31( 0.0013902494f), Q31( 0.0012577884f), Q31( 0.0011250155f), Q31( 0.0009885988f),
     Q31( 0.0008608443f), Q31( 0.0007458025f), Q31( 0.0006239376f), Q31( 0.0005107388f),
     Q31( 0.0004026540f), Q31( 0.0002949531f), Q31( 0.0002043017f), Q31( 0.0001094383f),
     Q31( 0.0000134949f), Q31(-0.0000617334f), Q31(-0.0001446380f), Q31(-0.0002098337f),
     Q31(-0.0002896981f), Q31(-0.0003501175f), Q31(-0.0004095121f), Q31(-0.0004606325f),
     Q31(-0.0005145572f), Q31(-0.0005564576f), Q31(-0.0005946118f), Q31(-0.0006341594f),
     Q31(-0.0006650415f), Q31(-0.0006917937f), Q31(-0.0007215391f), Q31(-0.0007319357f),
     Q31(-0.0007530001f), Q31(-0.0007630793f), Q31(-0.0007757977f), Q31(-0.0007801449f),
     Q31(-0.0007803664f), Q31(-0.0007779869f), Q31(-0.0007834332f), Q31(-0.0007724848f),
     Q31(-0.0007681371f), Q31(-0.0007490598f), Q31(-0.0007440941f), Q31(-0.0007255043f),
     Q31(-0.0007157736f), Q31(-0.0006941614f), Q31(-0.0006777690f), Q31(-0.0006540333f),
     Q31(-0.0006312493f), Q31(-0.0006132747f), Q31(-0.0005870930f), Q31(-0.0005677802f),
     Q31(-0.0005466565f), Q31(-0.0005226564f), Q31(-0.0005040714f), Q31(-0.0004893791f),
     Q31(-0.0004875227f), Q31(-0.0004947518f), Q31(-0.0005617692f), Q31(-0.0005525286f),
};

#endif /* AVCODEC_AACSBRDATA_H */
