/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "PreviewInfo.h"

#include <QByteArray>
#include <QImage>
#include <QQueue>

#include <KDebug>

#include "RawImageInfo.h"

struct PreviewInfo::Private
{
  RawImageInfoSP rawImageInfo;
  int width;
  int height;
  QByteArray data;
  static QQueue< const PreviewInfo* > cache;
};

QQueue< const PreviewInfo* > PreviewInfo::Private::cache;


PreviewInfo::PreviewInfo( RawImageInfoSP _rawImageInfo, int _width, int _height, const QByteArray& _data) : d(new Private)
{
  d->rawImageInfo = _rawImageInfo;
  d->width = _width;
  d->height = _height;
  d->data = _data;
}

PreviewInfo::~PreviewInfo()
{
  kDebug() << "============== Delete preview info for " << d->rawImageInfo.data();
  Private::cache.removeAll(this);
  d->rawImageInfo->setPreviewInfo(0);
  delete d;
}

int PreviewInfo::width() const
{
  return d->width;
}

int PreviewInfo::height() const
{
  return d->height;
}

const QByteArray& PreviewInfo::data() const
{
  return d->data;
}

QImage PreviewInfo::asQImage() const
{
  QImage image( d->width, d->height, QImage::Format_RGB32);
  for( int y = 0; y < d->height; ++y)
  {
    for(int x = 0; x < d->width; ++x )
    {
      quint16* ptr = (quint16*)( d->data.data() + (y * d->width + x )* 3 *sizeof(quint16));
      image.setPixel( x,y, qRgb( ptr[0] >> 8, ptr[1] >> 8, ptr[2] >> 8 ));
    }
  }
  Private::cache.removeAll( this );
  Private::cache.enqueue( this );
  return image;
}

void PreviewInfo::appendToCache( )
{
  Private::cache.enqueue( this );
  if( Private::cache.size() > 4 ) // TODO make that configurable
  {
    const PreviewInfo* first = Private::cache.dequeue( );
    delete first;
  }
}
