/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include <stdlib.h>  // qsort
#include <algorithm>
#include <cassert>

#include "percentiles.h"
#include "array.h"
#include "field.h"
#include "functs.h"
#include "cdo_output.h"

Field::Field() : fpeRaised(0), nwpv(1), floatPrecision(-1), grid(-1), size(0), nsamp(0), nmiss(0), missval(0) { m_count = 0; }

void
Field::resize(size_t count)
{
  m_count = count;
  vec.resize(m_count);
  if (!size) size = m_count;
}

void
Field::resize(size_t count, double value)
{
  m_count = count;
  vec.resize(m_count, value);
  if (!size) size = m_count;
}

void
Field::resizef(size_t count)
{
  m_count = count;
  vecf.resize(m_count);
  if (!size) size = m_count;
}

void
Field::resizef(size_t count, float value)
{
  m_count = count;
  vecf.resize(m_count, value);
  if (!size) size = m_count;
}

bool
Field::empty() const
{
  return m_count == 0;
}

void
Field::check_gridsize() const
{
  if (size == 0) fprintf(stderr, "Internal problem, size of field not set!");
  if (size > m_count) fprintf(stderr, "Internal problem, size of field is greater than allocated size of field!");
}

void
fieldFill(Field &field, double value)
{
  field.check_gridsize();

  std::fill(field.vec.begin(), field.vec.begin() + field.size, value);
}

// functor that returns true if value is equal to the value of the constructor parameter provided
class valueDblIsEqual
{
  double _missval;

public:
  valueDblIsEqual(double missval) : _missval(missval) {}
  bool
  operator()(double value) const
  {
    return DBL_IS_EQUAL(value, _missval);
  }
};

// functor that returns true if value is equal to the value of the constructor parameter provided
class valueIsEqual
{
  double _missval;

public:
  valueIsEqual(double missval) : _missval(missval) {}
  bool
  operator()(double value) const
  {
    return IS_EQUAL(value, _missval);
  }
};

size_t
fieldNumMiss(const Field &field)
{
  field.check_gridsize();

  size_t nummiss = 0;

  if (std::isnan(field.missval))
    {
      nummiss = std::count_if(field.vec.begin(), field.vec.begin() + field.size, valueDblIsEqual(field.missval));
    }
  else
    {
      nummiss = std::count_if(field.vec.begin(), field.vec.begin() + field.size, valueIsEqual(field.missval));
    }

  return nummiss;
}

static double
vfldrange(const Field &field)
{
  return field.nmiss ? varrayRangeMV(field.size, field.vec, field.missval) : varrayRange(field.size, field.vec);
}

double
vfldmin(const Field &field)
{
  return field.nmiss ? varrayMinMV(field.size, field.vec, field.missval) : varrayMin(field.size, field.vec);
}

double
vfldmax(const Field &field)
{
  return field.nmiss ? varrayMaxMV(field.size, field.vec, field.missval) : varrayMax(field.size, field.vec);
}

double
vfldsum(const Field &field)
{
  return field.nmiss ? varraySumMV(field.size, field.vec, field.missval) : varraySum(field.size, field.vec);
}

double
vfldmean(const Field &field)
{
  return field.nmiss ? varrayMeanMV(field.size, field.vec, field.missval) : varrayMean(field.size, field.vec);
}

double
vfldmeanw(const Field &field)
{
  return field.nmiss ? varrayWeightedMeanMV(field.size, field.vec, field.weightv, field.missval)
                     : varrayWeightedMean(field.size, field.vec, field.weightv, field.missval);
}

double
vfldavg(const Field &field)
{
  return field.nmiss ? varrayAvgMV(field.size, field.vec, field.missval) : varrayMean(field.size, field.vec);
}

double
vfldavgw(const Field &field)
{
  return field.nmiss ? varrayWeightedAvgMV(field.size, field.vec, field.weightv, field.missval)
                     : varrayWeightedMean(field.size, field.vec, field.weightv, field.missval);
}

static void
prevarsum(const Field &field, double &rsum, double &rsumw, double &rsumq, double &rsumwq)
{
  const auto len = field.size;
  const auto missval = field.missval;
  const auto &array = field.vec;

  assert(array.size() >= len);

  rsum = 0;
  rsumq = 0;
  rsumw = 0;
  rsumwq = 0;

  if (field.nmiss)
    {
      for (size_t i = 0; i < len; ++i)
        if (!DBL_IS_EQUAL(array[i], missval))
          {
            rsum += array[i];
            rsumq += array[i] * array[i];
            rsumw += 1;
            rsumwq += 1;
          }
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        {
          rsum += array[i];
          rsumq += array[i] * array[i];
        }
      rsumw = len;
      rsumwq = len;
    }
}

static void
preskewsum(const Field &field, const double mean, double &rsum3w, double &rsum4w, double &rsum3diff, double &rsum2diff)
{
  const auto len = field.size;
  const auto missval = field.missval;
  const auto &array = field.vec;

  assert(array.size() >= len);

  double xsum3w = 0, xsum3diff = 0;
  double xsum4w = 0, xsum2diff = 0;

  if (field.nmiss)
    {
      for (size_t i = 0; i < len; ++i)
        if (!DBL_IS_EQUAL(array[i], missval))
          {
            xsum3diff += (array[i] - mean) * (array[i] - mean) * (array[i] - mean);
            xsum2diff += (array[i] - mean) * (array[i] - mean);
            xsum3w += 1;
            xsum4w += 1;
          }
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        {
          xsum3diff += (array[i] - mean) * (array[i] - mean) * (array[i] - mean);
          xsum2diff += (array[i] - mean) * (array[i] - mean);
        }
      xsum3w = len;
      xsum4w = len;
    }

  rsum3diff = xsum3diff;
  rsum2diff = xsum2diff;
  rsum3w = xsum3w;
  rsum4w = xsum4w;
}

static void
prekurtsum(const Field &field, const double mean, double &rsum3w, double &rsum4w, double &rsum2diff, double &rsum4diff)
{
  const auto len = field.size;
  const auto missval = field.missval;
  const auto &array = field.vec;

  assert(array.size() >= len);

  double xsum3w = 0, xsum4diff = 0;
  double xsum4w = 0, xsum2diff = 0;

  if (field.nmiss)
    {
      for (size_t i = 0; i < len; ++i)
        if (!DBL_IS_EQUAL(array[i], missval))
          {
            xsum2diff += (array[i] - mean) * (array[i] - mean);
            xsum4diff += (array[i] - mean) * (array[i] - mean) * (array[i] - mean) * (array[i] - mean);
            xsum3w += 1;
            xsum4w += 1;
          }
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        {
          xsum2diff += (array[i] - mean) * (array[i] - mean);
          xsum4diff += (array[i] - mean) * (array[i] - mean) * (array[i] - mean) * (array[i] - mean);
        }
      xsum3w = len;
      xsum4w = len;
    }

  rsum4diff = xsum4diff;
  rsum2diff = xsum2diff;
  rsum3w = xsum3w;
  rsum4w = xsum4w;
}

double
vfldvar(const Field &field)
{
  double rsum, rsumw, rsumq, rsumwq;
  prevarsum(field, rsum, rsumw, rsumq, rsumwq);

  auto rvar = IS_NOT_EQUAL(rsumw, 0) ? (rsumq * rsumw - rsum * rsum) / (rsumw * rsumw) : field.missval;
  if (rvar < 0 && rvar > -1.e-5) rvar = 0;

  return rvar;
}

double
vfldvar1(const Field &field)
{
  double rsum, rsumw, rsumq, rsumwq;
  prevarsum(field, rsum, rsumw, rsumq, rsumwq);

  auto rvar = (rsumw * rsumw > rsumwq) ? (rsumq * rsumw - rsum * rsum) / (rsumw * rsumw - rsumwq) : field.missval;
  if (rvar < 0 && rvar > -1.e-5) rvar = 0;

  return rvar;
}

double
vfldkurt(const Field &field)
{
  double rsum3w;  // 3rd moment variables
  double rsum4w;  // 4th moment variables
  double rsum2diff, rsum4diff;

  double rsum, rsumw, rsumq, rsumwq;
  prevarsum(field, rsum, rsumw, rsumq, rsumwq);
  prekurtsum(field, (rsum / rsumw), rsum3w, rsum4w, rsum2diff, rsum4diff);

  if (IS_EQUAL(rsum3w, 0.0) || IS_EQUAL(rsum2diff, 0.0)) return field.missval;

  double rvar = ((rsum4diff / rsum3w) / std::pow((rsum2diff) / (rsum3w), 2)) - 3.0;
  if (rvar < 0 && rvar > -1.e-5) rvar = 0;

  return rvar;
}

double
vfldskew(const Field &field)
{
  double rsum3w;  // 3rd moment variables
  double rsum4w;  // 4th moment variables
  double rsum3diff, rsum2diff;

  double rsum, rsumw, rsumq, rsumwq;
  prevarsum(field, rsum, rsumw, rsumq, rsumwq);
  preskewsum(field, (rsum / rsumw), rsum3w, rsum4w, rsum3diff, rsum2diff);

  if (IS_EQUAL(rsum3w, 0.0) || IS_EQUAL(rsum3w, 1.0) || IS_EQUAL(rsum2diff, 0.0)) return field.missval;

  double rvar = (rsum3diff / rsum3w) / std::pow((rsum2diff) / (rsum3w - 1.0), 1.5);
  if (rvar < 0 && rvar > -1.e-5) rvar = 0;

  return rvar;
}

static void
prevarsumw(const Field &field, double &rsum, double &rsumw, double &rsumq, double &rsumwq)
{
  const auto len = field.size;
  const auto missval = field.missval;
  const auto &v = field.vec;
  const auto &w = field.weightv;

  assert(v.size() >= len);
  assert(w.size() >= len);

  rsum = 0;
  rsumq = 0;
  rsumw = 0;
  rsumwq = 0;

  if (field.nmiss)
    {
      for (size_t i = 0; i < len; ++i)
        if (!DBL_IS_EQUAL(v[i], missval) && !DBL_IS_EQUAL(w[i], missval))
          {
            rsum += w[i] * v[i];
            rsumq += w[i] * v[i] * v[i];
            rsumw += w[i];
            rsumwq += w[i] * w[i];
          }
    }
  else
    {
      for (size_t i = 0; i < len; ++i)
        {
          rsum += w[i] * v[i];
          rsumq += w[i] * v[i] * v[i];
          rsumw += w[i];
          rsumwq += w[i] * w[i];
        }
    }
}

double
vfldvarw(const Field &field)
{
  double rsum, rsumw, rsumq, rsumwq;
  prevarsumw(field, rsum, rsumw, rsumq, rsumwq);

  double rvar = IS_NOT_EQUAL(rsumw, 0) ? (rsumq * rsumw - rsum * rsum) / (rsumw * rsumw) : field.missval;
  if (rvar < 0 && rvar > -1.e-5) rvar = 0;

  return rvar;
}

double
vfldvar1w(const Field &field)
{
  double rsum, rsumw, rsumq, rsumwq;
  prevarsumw(field, rsum, rsumw, rsumq, rsumwq);

  double rvar = (rsumw * rsumw > rsumwq) ? (rsumq * rsumw - rsum * rsum) / (rsumw * rsumw - rsumwq) : field.missval;
  if (rvar < 0 && rvar > -1.e-5) rvar = 0;

  return rvar;
}

double
varToStd(double rvar, double missval)
{
  double rstd;

  if (DBL_IS_EQUAL(rvar, missval) || rvar < 0)
    {
      rstd = missval;
    }
  else
    {
      rstd = IS_NOT_EQUAL(rvar, 0) ? std::sqrt(rvar) : 0;
    }

  return rstd;
}

double
vfldstd(const Field &field)
{
  return varToStd(vfldvar(field), field.missval);
}

double
vfldstd1(const Field &field)
{
  return varToStd(vfldvar1(field), field.missval);
}

double
vfldstdw(const Field &field)
{
  return varToStd(vfldvarw(field), field.missval);
}

double
vfldstd1w(const Field &field)
{
  return varToStd(vfldvar1w(field), field.missval);
}

void
vfldrms(const Field &field, const Field &field2, Field &field3)
{
  size_t i;
  size_t rnmiss = 0;
  auto grid1 = field.grid;
  //  size_t nmiss1   = field.nmiss;
  const auto array1 = field.vec.data();
  auto grid2 = field2.grid;
  //  size_t nmiss2   = field2.nmiss;
  const auto array2 = field2.vec.data();
  const auto missval1 = field.missval;
  const auto missval2 = field2.missval;
  const auto &w = field.weightv;
  double rsum = 0, rsumw = 0;

  const auto len = gridInqSize(grid1);
  if (len != gridInqSize(grid2)) cdoAbort("fields have different size!");

  /*
  if ( nmiss1 )
  */
  {
    for (i = 0; i < len; i++)
      if (!DBL_IS_EQUAL(w[i], missval1))
        {
          rsum = ADDMN(rsum, MULMN(w[i], MULMN(SUBMN(array2[i], array1[i]), SUBMN(array2[i], array1[i]))));
          rsumw = ADDMN(rsumw, w[i]);
        }
  }
  /*
else
  {
    for ( i = 0; i < len; i++ )
      {
        rsum  += w[i] * array1[i];
        rsumw += w[i];
      }
  }
  */

  const double ravg = SQRTMN(DIVMN(rsum, rsumw));

  if (DBL_IS_EQUAL(ravg, missval1)) rnmiss++;

  field3.vec[0] = ravg;
  field3.nmiss = rnmiss;
}

double
vfldpctl(Field &field, const double pn)
{
  auto pctl = field.missval;

  if ((field.size - field.nmiss) > 0)
    {
      if (field.nmiss)
        {
          Varray<double> v(field.size - field.nmiss);

          size_t j = 0;
          for (size_t i = 0; i < field.size; i++)
            if (!DBL_IS_EQUAL(field.vec[i], field.missval)) v[j++] = field.vec[i];

          pctl = percentile(v.data(), j, pn);
        }
      else
        {
          pctl = percentile(field.vec.data(), field.size, pn);
        }
    }

  return pctl;
}

static int
compareDouble(const void *a, const void *b)
{
  const double *x = (const double *) a;
  const double *y = (const double *) b;
  return ((*x > *y) - (*x < *y)) * 2 + (*x > *y) - (*x < *y);
}

double
vfldrank(Field &field)
{
  double res = 0;
  // Using first value as reference (observation)
  double *array = &field.vec[1];
  const auto val = array[-1];
  const auto len = field.size - 1;

  if (field.nmiss) return field.missval;

  qsort(array, len, sizeof(double), compareDouble);

  if (val > array[len - 1])
    res = (double) len;
  else
    for (size_t j = 0; j < len; j++)
      if (array[j] >= val)
        {
          res = (double) j;
          break;
        }

  return res;
}

double fldbrs(Field &field)  // Not used!
{
  const auto nmiss = field.nmiss;
  const auto len = field.size;
  const auto array = field.vec.data();
  const auto missval = field.missval;

  double brs = 0;
  size_t i, count = 0;

  // Using first value as reference
  if (nmiss == 0)
    {
      for (i = 1; i < len; i++) brs += (array[i] - array[0]) * (array[i] - array[0]);
      count = i - 1;
    }
  else
    {
      if (DBL_IS_EQUAL(array[0], missval)) return missval;

      for (i = 1; i < len; i++)
        if (!DBL_IS_EQUAL(array[i], missval))
          {
            brs += (array[i] - array[0]) * (array[i] - array[0]);
            count++;
          }
    }

  return brs / count;
}

double
vfldfun(const Field &field, int function)
{
  double rval = 0;

  // clang-format off
  switch (function)
    {
    case func_range:  rval = vfldrange(field);  break;
    case func_min:    rval = vfldmin(field);    break;
    case func_max:    rval = vfldmax(field);    break;
    case func_sum:    rval = vfldsum(field);    break;
    case func_mean:   rval = vfldmean(field);   break;
    case func_avg:    rval = vfldavg(field);    break;
    case func_std:    rval = vfldstd(field);    break;
    case func_std1:   rval = vfldstd1(field);   break;
    case func_var:    rval = vfldvar(field);    break;
    case func_var1:   rval = vfldvar1(field);   break;
    case func_meanw:  rval = vfldmeanw(field);  break;
    case func_avgw:   rval = vfldavgw(field);   break;
    case func_stdw:   rval = vfldstdw(field);   break;
    case func_std1w:  rval = vfldstd1w(field);  break;
    case func_varw:   rval = vfldvarw(field);   break;
    case func_var1w:  rval = vfldvar1w(field);  break;
    case func_skew:   rval = vfldskew(field);   break;
    case func_kurt:   rval = vfldkurt(field);   break;
    default: cdoAbort("%s: function %d not implemented!", __func__, function);
    }
  // clang-format on

  return rval;
}
