!{\src2tex{textfont=tt}}
!!****f* ABINIT/pawxcm
!! NAME
!! pawxcm
!!
!! FUNCTION
!! PAW only
!! Start from the density or spin-density, and compute xc correlation
!! potential and energies inside a paw sphere.
!! LDA+GGA - USE A DEVELOPMENT OF THE DENSITY OVER (L,M) MOMENTS
!! Driver of XC functionals. Only treat colinear spins.
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!! This routine has been written from rhohxc_coll
!!
!! INPUTS
!!  corexc(pawrad%mesh_size)=core density on radial grid
!!  ixc= choice of exchange-correlation scheme (see above and below)
!!  lm_size=size of density array rhor (see below)
!!  lmselect(lm_size,nspden)=select the non-zero LM-moments of input density rhor
!!  nhat(pawrad%mesh_size,lm_size,nspden)=compensation density
!!                                        (total in 1st half and spin-up in 2nd half if nspden=2)
!!  nspden=number of spin-density components
!!  option=0 compute both XC energies (direct+double-counting) and potential
!!         1 compute only XC potential
!!         2 compute only XC energies (direct+double-counting)
!!         3 compute only XC energy by direct scheme
!!         4 compute only XC energy by direct scheme for spherical part of the density
!!  pawrad <type(pawrad_type)>=paw radial mesh and related data
!!  rhor(pawrad%mesh_size,lm_size,nspden)=electron density in real space in electrons/bohr**3
!!                                       (total in 1st half and spin-up in 2nd half if nspden=2)
!!  usecore= 1 if core density has to be used in Exc/Vxc ; 0 otherwise
!!  usexcnhat= 0 if compensation density does not have to be used
!!             1 if compensation density has to be used in double counting energy term only
!!             2 if compensation density (nhat) has to be used in Exc/Vxc and double counting energy term
!!
!! OUTPUT
!!  == if option==0, 2, 3, or 4 ==
!!    enxc=returned exchange and correlation energy (hartree)
!!  == if option==0 or 2 ==
!!    enxcdc=returned exchange-cor. contribution to double-counting energy
!!  == if option==0 or 1 ==
!!    vxc(pawrad%mesh_size,lm_size,nspden)=xc potential
!!       (spin up in 1st half and spin-down in 2nd half if nspden=2)
!!
!! PARENTS
!!      pawdenpot,psp7in
!!
!! CHILDREN
!!      drivexc,leave_new,mkdenpos,pawmkgrad,pawxcdenm,pawxcgrad,simp_gen
!!      size_dvxc,timab,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

 subroutine pawxcm(corexc,enxc,enxcdc,ixc,lm_size,lmselect,nhat,nspden,option,&
&                  pawrad,rhor,usecore,usexcnhat,vxc)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi
 use interfaces_11util
 use interfaces_13paw, except_this_one => pawxcm
 use interfaces_13xc
#else
 use defs_xc
#endif
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ixc,lm_size,nspden,option,usecore,usexcnhat
 real(dp),intent(out) :: enxc,enxcdc
 type(pawrad_type),intent(in) :: pawrad
!arrays
 integer,intent(in) :: lmselect(lm_size,nspden)
 real(dp),intent(in) :: corexc(pawrad%mesh_size)
 real(dp),intent(in) :: nhat(pawrad%mesh_size,lm_size,nspden)
 real(dp),intent(in) :: rhor(pawrad%mesh_size,lm_size,nspden)
 real(dp),intent(out) :: vxc(pawrad%mesh_size,lm_size,nspden)

!Local variables-------------------------------
!scalars
 integer :: ilm,ipts,ir,ir1,ir2,isel,ispden,jr,ndvxc,ngr2,ngrad,nrad,nspgrad
 integer :: nvxcdgr,order
 real(dp),parameter :: delta=1.d-4
 real(dp) :: pp,sqfpi,sqfpi2
 character(len=500) :: message
!arrays
 real(dp) :: drho_updn(3),tsec(2)
 real(dp),allocatable :: d1vxc(:,:),d2vxc(:,:),d2vxcar(:),dnexcdn(:,:)
 real(dp),allocatable :: dvxcdgr(:,:),dvxcdgr1(:,:),dvxcdgr2(:,:),dvxci(:,:)
 real(dp),allocatable :: exc1(:),exc2(:),exci(:),ff(:),gg(:),grho2_updn(:,:)
 real(dp),allocatable :: rho1(:,:),rho2(:,:),rho_updn(:,:),rhoeff(:,:,:)
 real(dp),allocatable :: rhoeffdc(:,:,:),rhoinv(:,:),rhonow(:,:,:)
 real(dp),allocatable :: rhonow_(:,:,:),rhosph(:,:),rhowork(:,:),vxc1(:,:)
 real(dp),allocatable :: vxc2(:,:),vxci(:,:)

! *************************************************************************

! DEBUG
! write(6,*)' pawxcm : enter with option, nspden ',option,nspden
! ENDDEBUG

 call timab(81,1,tsec)

!----------------------------------------------------------------------
!----- Check options
!----------------------------------------------------------------------

 if(nspden/=1 .and. nspden/=2)then
  write(message, '(a,a,a,a,a,a,i5)' ) ch10,&
&  ' pawxcm :  BUG -',ch10,&
&  '  Only non-spin-polarised or collinear spin-densities are allowed,',ch10,&
&  '  while the argument nspden=',nspden
  call wrtout(6,message,'COLL')
  call leave_new('COLL')
 end if
 if (nspden==2.and.(ixc/=0.and.ixc/=1.and.(ixc<7.or.ixc>16))) then
  write(message, '(a,a,a,a,a,a,i8,a,a,a)' ) ch10,&
&   ' pawxcm : ERROR -',ch10,&
&   '  With nspden=2, must use ixc=0, 1 or ixc between 7 and 16.',ch10,&
&   '  Found ixc=',ixc,'.',ch10,&
&   '  Action : change either nspden or ixc in your input file.'
  call wrtout(06,message,'COLL')
  call leave_new('COLL')
 end if

!----------------------------------------------------------------------
!----- Initializations
!----------------------------------------------------------------------

 nrad=pawrad%mesh_size
 ngrad=1                            ! ngrad=1 is for LDAs or LSDs
 if(ixc>=11 .and. ixc<=16 ) ngrad=2 ! ngrad=2 is for GGAs
 nspgrad=nspden*ngrad
 if(nspden==2.and.ngrad==2) nspgrad=5
 if (option/=1) enxc=zero
 if (option==0.or.option==2) enxcdc=zero
 if (option<3) vxc(:,:,:)=zero
 sqfpi=sqrt(four_pi)
 sqfpi2=half*sqfpi

 if (ixc==0) then ! No xc at all is applied (usually for testing)
  write(message, '(a,a,a,a)' ) ch10,&
&  ' pawxcm : WARNING -',ch10,&
&  '  Note that no xc is applied (ixc=0).'
  call wrtout(06,message,'COLL')

 else if (1<=ixc .and. ixc<=16)  then

! The different components of dnexcdn will be
! for nspden=1,           dnexcdn(:,1)=d(n.exc)/d(n)
!         and if ngrad=2, dnexcdn(:,2)=1/2*1/|grad n_up|*d(n.exc)/d(|grad n_up|)
!                                      +   1/|grad n|*d(n.exc)/d(|grad n|)
!                         do not forget : |grad n| /= |grad n_up| + |grad n_down|
! for nspden=2,           dnexcdn(:,1)=d(n.exc)/d(n_up)
!                         dnexcdn(:,2)=d(n.exc)/d(n_down)
!         and if ngrad=2, dnexcdn(:,3)=1/|grad n_up|*d(n.exc)/d(|grad n_up|)
!                         dnexcdn(:,4)=1/|grad n_down|*d(n.exc)/d(|grad n_down|)
!                         dnexcdn(:,5)=1/|grad n|*d(n.exc)/d(|grad n|)
! Density points of the spherical grid are in rhonow(:,:,1).
! For GGA calculations the gradient of the density will be in rhonow(:,:,2:4),
! The calculation of the gradient on the sphere is not yet implemented

  allocate(rhonow(nrad,nspden,ngrad*ngrad),dnexcdn(nrad,nspgrad))
  allocate(rhoeff(nrad,lm_size,nspden),rhosph(nrad,nspden))
  if ((usexcnhat==1.or.usecore==1).and.(option==0.or.option==2)) allocate(rhoeffdc(nrad,lm_size,nspden))

! Build several densities
  rhoeff(:,:,1)=rhor(:,:,nspden)
  if (nspden==2) rhoeff(:,:,2)=rhor(:,:,1)-rhor(:,:,2)
  if (usexcnhat==2) then
   rhoeff(:,:,1)=rhoeff(:,:,1)+nhat(:,:,nspden)
   if (nspden==2) rhoeff(:,:,2)=rhoeff(:,:,2)+nhat(:,:,1)-nhat(:,:,2)
  end if
  if (option==0.or.option==2) then
   if (usexcnhat==1) then
    rhoeffdc(:,:,1)=rhoeff(:,:,1)+nhat(:,:,nspden)
    if (nspden==2) rhoeffdc(:,:,2)=rhoeffdc(:,:,2)+nhat(:,:,1)-nhat(:,:,2)
   else if (usecore==1) then
    rhoeffdc(:,:,:)=rhoeff(:,:,:)
   end if
  end if
  if (usecore==1) then
   if (nspden==1) rhoeff(:,1,1)=rhoeff(:,1,1)+sqfpi*corexc(:)
   if (nspden==2) then
    rhoeff(:,1,1)=rhoeff(:,1,1)+sqfpi2*corexc(:)
    rhoeff(:,1,2)=rhoeff(:,1,2)+sqfpi2*corexc(:)
   end if
  end if

! Prepare spherical density data
  rhosph(:,nspden)=rhoeff(:,1,1)
  if (nspden==2) rhosph(:,1)=rhoeff(:,1,1)+rhoeff(:,1,2)
  call mkdenpos(nrad,1,nspden,rhosph)

! The variable order indicates to which derivative of the energy
! the computation must be done. Here, no derivative.
  order=1

!----------------------------------------------------------------------
!----- Compute Vxc
!----------------------------------------------------------------------

  !Allocation of mandatory arguments of drivexc
  allocate(exci(nrad),vxci(nrad,nspden),rho_updn(nrad,nspden))
  !Allocation of optional arguments
  call size_dvxc(ixc,ndvxc,ngr2,nspden,nvxcdgr,order)
  if (ndvxc/=0) allocate(dvxci(nrad,ndvxc))
  if (nvxcdgr/=0) allocate(dvxcdgr(nrad,nvxcdgr))
  if ((ixc==3 .or. (ixc>=7 .and. ixc<=15)) .and. order==3) allocate(d2vxcar(nrad))
  if (ngrad == 2) allocate(grho2_updn(nrad,ngr2))

!!$   Old version:
!!$   (Note : could be more selective in allocating memory !)
!!$   allocate(dvxci(nrad,15),dvxcdgr(nrad,3),d2vxcar(nrad),exci(nrad),vxci(nrad,nspden))
!!$   allocate(grho2_updn(nrad,2*nspden-1),rhoarr(nrad),rho_updn(nrad,nspden))

  call pawxcdenm(ngrad,nrad,nspden,pawrad,rhonow,rhosph)
  call pawmkgrad(grho2_updn,ixc,ngrad,nrad,nspden,rhonow,rho_updn)

! Cases with gradient
  if (ixc >= 11 .and. ixc <= 16) then
   if (order**2 <= 1 .or. ixc == 16) then
    if (ixc /= 13) then
     call drivexc(exci,ixc,nrad,nspden,order,rho_updn,vxci,ndvxc,ngr2,nvxcdgr,&
&                 grho2_updn=grho2_updn,vxcgr=dvxcdgr)
    else
     call drivexc(exci,ixc,nrad,nspden,order,rho_updn,vxci,ndvxc,ngr2,nvxcdgr,&
&                 grho2_updn=grho2_updn)
    end if
   else if (order /= 3) then
    if (ixc /= 13) then
     call drivexc(exci,ixc,nrad,nspden,order,rho_updn,vxci,ndvxc,ngr2,nvxcdgr,&
&                 dvxc=dvxci,grho2_updn=grho2_updn,vxcgr=dvxcdgr)
    else
     call drivexc(exci,ixc,nrad,nspden,order,rho_updn,vxci,ndvxc,ngr2,nvxcdgr,&
&                 dvxc=dvxci,grho2_updn=grho2_updn)
    end if
   else if (order == 3) then
    if (ixc /= 13) then
     call drivexc(exci,ixc,nrad,nspden,order,rho_updn,vxci,ndvxc,ngr2,nvxcdgr,&
&                 dvxci,d2vxcar,grho2_updn,dvxcdgr)
    else
     call drivexc(exci,ixc,nrad,nspden,order,rho_updn,vxci,ndvxc,ngr2,nvxcdgr,&
&                 dvxc=dvxci,d2vxc=d2vxcar,grho2_updn=grho2_updn)
    end if
   end if
! Cases without gradient
  else
   if (order**2 <=1 .or. ixc >= 31 .and. ixc<=34) then
    call drivexc(exci,ixc,nrad,nspden,order,rho_updn,vxci,ndvxc,ngr2,nvxcdgr)
   else if (order==3 .and. (ixc==3 .or. ixc>=7 .and. ixc<=10)) then
    call drivexc(exci,ixc,nrad,nspden,order,rho_updn,vxci,ndvxc,ngr2,nvxcdgr,&
&                dvxc=dvxci,d2vxc=d2vxcar)
   else
    call drivexc(exci,ixc,nrad,nspden,order,rho_updn,vxci,ndvxc,ngr2,nvxcdgr,&
&                dvxc=dvxci)
   end if
  end if

  if (ngrad==2) call pawxcgrad(dnexcdn,dvxcdgr,grho2_updn,ngrad,nrad,&
&                              nspden,nspgrad,nvxcdgr,pawrad,rhonow,vxci)

!----------------------------------------------------------------------
!----- Compute numerical derivatives of Vxc
!----------------------------------------------------------------------

  if (option/=4) then
   allocate(rhonow_(nrad,nspden,ngrad*ngrad),rhowork(nrad,nspden))

!  ===== Compute exc, vxc (and gradients) for rho+delta_rho

   !Allocation of mandatory arguments of drivexc
   allocate(exc1(nrad),vxc1(nrad,nspden),rho1(nrad,nspden))
   !Allocation of optional arguments
   if (nvxcdgr/=0) allocate(dvxcdgr1(nrad,nvxcdgr))

   if (ngrad==2) then
    rhowork(:,:)=(1+delta)*rhosph(:,:)
    call pawxcdenm(ngrad,nrad,nspden,pawrad,rhonow_,rhowork)
    call pawmkgrad(grho2_updn,ixc,ngrad,nrad,nspden,rhonow_,rho1)
   else
    rho1=(1+delta)*rho_updn
   end if

!  Case with gradient
    if (ixc >= 11 .and. ixc <= 16) then
     if (order**2 <= 1 .or. ixc == 16) then
      if (ixc /= 13) then
       call drivexc(exc1,ixc,nrad,nspden,order,rho1,vxc1,ndvxc,ngr2,nvxcdgr,&
&                   grho2_updn=grho2_updn,vxcgr=dvxcdgr1)
      else
       call drivexc(exc1,ixc,nrad,nspden,order,rho1,vxc1,ndvxc,ngr2,nvxcdgr,&
&                   grho2_updn=grho2_updn)
      end if
     else if (order /= 3) then
      if (ixc /= 13) then
       call drivexc(exc1,ixc,nrad,nspden,order,rho1,vxc1,ndvxc,ngr2,nvxcdgr,&
&                   dvxc=dvxci,grho2_updn=grho2_updn,vxcgr=dvxcdgr1)
      else
       call drivexc(exc1,ixc,nrad,nspden,order,rho1,vxc1,ndvxc,ngr2,nvxcdgr,&
&                   dvxc=dvxci,grho2_updn=grho2_updn)
      end if
     else if (order == 3) then
      if (ixc /= 13) then
       call drivexc(exc1,ixc,nrad,nspden,order,rho1,vxc1,ndvxc,ngr2,nvxcdgr,&
&                   dvxci,d2vxcar,grho2_updn,dvxcdgr1)
      else
       call drivexc(exc1,ixc,nrad,nspden,order,rho1,vxc1,ndvxc,ngr2,nvxcdgr,&
&                   dvxc=dvxci,d2vxc=d2vxcar,grho2_updn=grho2_updn)
      end if
     end if
    !Cases without gradient
    else
     if (order**2 <=1 .or. ixc >= 31 .and. ixc<=34) then
      call drivexc(exc1,ixc,nrad,nspden,order,rho1,vxc1,ndvxc,ngr2,nvxcdgr)
     else if (order==3 .and. (ixc==3 .or. ixc>=7 .and. ixc<=10)) then
      call drivexc(exc1,ixc,nrad,nspden,order,rho1,vxc1,ndvxc,ngr2,nvxcdgr,&
&                  dvxc=dvxci,d2vxc=d2vxcar)
     else
      call drivexc(exc1,ixc,nrad,nspden,order,rho1,vxc1,ndvxc,ngr2,nvxcdgr,&
&                  dvxc=dvxci)
     end if
    end if

!!$   Old version:
!!$   call drivexc(exc1,ixc,nrad,nspden,order,rho1,vxc1,dvxcdgr1,&
!!$&               dvxci,d2vxcar,grho2_updn)

   if (ngrad==2) call pawxcgrad(dnexcdn,dvxcdgr1,grho2_updn,ngrad,nrad,&
&                               nspden,nspgrad,nvxcdgr,pawrad,rhonow_,vxc1)

   deallocate(exc1,rho1)
   if (allocated(dvxcdgr1)) deallocate(dvxcdgr1)

!  ===== Compute exc, vxc (and gradients) for rho-delta_rho

   !Allocation of mandatory arguments of drivexc
   allocate(exc2(nrad),vxc2(nrad,nspden),rho2(nrad,nspden))
   !Allocation of optional arguments
   if (nvxcdgr/=0) allocate(dvxcdgr2(nrad,nvxcdgr))

   if (ngrad==2) then
    rhowork(:,:)=(1-delta)*rhosph(:,:)
    call pawxcdenm(ngrad,nrad,nspden,pawrad,rhonow_,rhowork)
    call pawmkgrad(grho2_updn,ixc,ngrad,nrad,nspden,rhonow_,rho2)
   else
    rho2=(1-delta)*rho_updn
   end if

!  Cases with gradient
   if (ixc >= 11 .and. ixc <= 16) then
    if (order**2 <= 1 .or. ixc == 16) then
     if (ixc /= 13) then
      call drivexc(exc2,ixc,nrad,nspden,order,rho2,vxc2,ndvxc,ngr2,nvxcdgr,&
&                  grho2_updn=grho2_updn,vxcgr=dvxcdgr2)
     else
      call drivexc(exc2,ixc,nrad,nspden,order,rho2,vxc2,ndvxc,ngr2,nvxcdgr,&
&                  grho2_updn=grho2_updn)
     end if
    else if (order /= 3) then
     if (ixc /= 13) then
      call drivexc(exc2,ixc,nrad,nspden,order,rho2,vxc2,ndvxc,ngr2,nvxcdgr,&
&                  dvxc=dvxci,grho2_updn=grho2_updn,vxcgr=dvxcdgr2)
     else
      call drivexc(exc2,ixc,nrad,nspden,order,rho2,vxc2,ndvxc,ngr2,nvxcdgr,&
&                  dvxc=dvxci,grho2_updn=grho2_updn)
     end if
    else if (order == 3) then
     if (ixc /= 13) then
      call drivexc(exc2,ixc,nrad,nspden,order,rho2,vxc2,ndvxc,ngr2,nvxcdgr,&
&                  dvxci,d2vxcar,grho2_updn,dvxcdgr2)
     else
      call drivexc(exc2,ixc,nrad,nspden,order,rho2,vxc2,ndvxc,ngr2,nvxcdgr,&
&                  dvxc=dvxci,d2vxc=d2vxcar,grho2_updn=grho2_updn)
     end if
    end if
!  Cases without gradient
   else
    if (order**2 <=1 .or. ixc >= 31 .and. ixc<=34) then
     call drivexc(exc2,ixc,nrad,nspden,order,rho2,vxc2,ndvxc,ngr2,nvxcdgr)
    else if (order==3 .and. (ixc==3 .or. ixc>=7 .and. ixc<=10)) then
     call drivexc(exc2,ixc,nrad,nspden,order,rho2,vxc2,ndvxc,ngr2,nvxcdgr,&
&                 dvxc=dvxci,d2vxc=d2vxcar)
    else
     call drivexc(exc2,ixc,nrad,nspden,order,rho2,vxc2,ndvxc,ngr2,nvxcdgr,&
&                 dvxc=dvxci)
    end if
   end if

!!$   Old version:
!!$   call drivexc(exc2,ixc,nrad,nspden,order,rho2,vxc2,dvxcdgr2,&
!!$&               dvxci,d2vxcar,grho2_updn)

   if (ngrad==2) call pawxcgrad(dnexcdn,dvxcdgr2,grho2_updn,ngrad,nrad,&
&                               nspden,nspgrad,nvxcdgr,pawrad,rhonow_,vxc2)

   deallocate(exc2,rho2)
   if (allocated(dvxcdgr2)) deallocate(dvxcdgr2)
   deallocate(rhonow_,rhowork)

!  ===== Compute numerical derivatives for vxc
   allocate(rhoinv(nrad,nspden))
   if (nspden==1) then
    do ir=1,nrad
     if (rhonow(ir,1,1)>tol14) then
      rhoinv(ir,1)=one/(delta*rhonow(ir,1,1))
     else
      rhoinv(ir,1)=zero
     end if
    end do
   else if (nspden==2) then
    do ispden=1,nspden
     do ir=1,nrad
      if (rho_updn(ir,ispden)>tol14) then
       rhoinv(ir,ispden)=one/(delta*rho_updn(ir,ispden))
      else
       rhoinv(ir,ispden)=zero
      end if
     end do
    end do
   end if
   if (option<3) then
    allocate(d1vxc(nrad,nspden),d2vxc(nrad,nspden))
    do ispden=1,nspden
     d1vxc(1:nrad,ispden)=(vxc1(1:nrad,ispden)-vxc2(1:nrad,ispden))*0.5_dp*rhoinv(1:nrad,ispden)
     d2vxc(1:nrad,ispden)=(vxc1(1:nrad,ispden)+vxc2(1:nrad,ispden)&
&                         -2._dp*vxci(1:nrad,ispden))*rhoinv(1:nrad,ispden)**2
    end do
   else
    allocate(d1vxc(nrad,nspden))
    do ispden=1,nspden
     d1vxc(1:nrad,ispden)=(vxc1(1:nrad,ispden)-vxc2(1:nrad,ispden))&
&                         *0.5_dp*rhoinv(1:nrad,ispden)
    end do
   end if ! option
   deallocate(rhoinv)

   deallocate(vxc1,vxc2)
  end if ! option

!----------------------------------------------------------------------
!----- Accumulate and store XC potential
!----------------------------------------------------------------------

  if (option<3) then
   do ispden=1,nspden

    do ilm=2,lm_size
     if (lmselect(ilm,ispden)>0) then
      vxc(1:nrad,1  ,ispden)=  vxc(1:nrad,1,ispden)+rhoeff(1:nrad,ilm,ispden)**2
      vxc(1:nrad,ilm,ispden)=d1vxc(1:nrad,ispden)  *rhoeff(1:nrad,ilm,ispden)
     end if
    end do
    vxc(1:nrad,1,ispden)=vxc(1:nrad,1,ispden)*d2vxc(1:nrad,ispden)/sqfpi*half+vxci(1:nrad,ispden)*sqfpi

!   Pathological case: if rho(r) is negative, interpolate vxc:
    do ilm=1,lm_size
     if (lmselect(ilm,ispden)>0) then
      ir1=0;ir2=0
      do ir=1,nrad
       if (rhoeff(ir,ilm,ispden)<tol14) then
        if (ir1==0) ir1=ir-1
        ir2=ir+1
       else if (ir1>0) then
        if (ir1>1.or.ir2<nrad) then
         pp=(vxc(ir2,ilm,ispden)-vxc(ir1,ilm,ispden))/(pawrad%rad(ir2)-pawrad%rad(ir1))
         do jr=ir1+1,ir2-1
          vxc(jr,ilm,ispden)=vxc(ir1,ilm,ispden)+pp*(pawrad%rad(jr)-pawrad%rad(ir1))
         end do
        end if
        ir1=0;ir2=0
       end if
      end do
     end if
    end do

   end do
   deallocate(d2vxc)
  end if !option

!----------------------------------------------------------------------
!----- Accumulate and store XC energies
!----------------------------------------------------------------------

  if (option/=1) allocate(ff(nrad))

!----- Calculate Exc (direct scheme) term
  if (option/=1) then
   if (nspden==1) ff(1:nrad)=rhoeff(1:nrad,1,1)*exci(1:nrad)*sqfpi
   if (nspden==2) ff(1:nrad)=(rhoeff(1:nrad,1,1)+rhoeff(1:nrad,1,2))*exci(1:nrad)*sqfpi
   if (option/=4) then
    allocate(gg(nrad));gg(1:nrad)=zero
    do ispden=1,nspden
     do ilm=2,lm_size
      if (lmselect(ilm,ispden)>0) gg(1:nrad)=gg(1:nrad)+rhoeff(1:nrad,ilm,ispden)**2
     end do
    end do
    ff(1:nrad)=ff(1:nrad)+half*gg(1:nrad)*d1vxc(1:nrad,1)
    deallocate(gg)
   end if
   ff(1:nrad)=ff(1:nrad)*pawrad%rad(1:nrad)**2
   call simp_gen(enxc,ff,pawrad)
  end if
  if (option/=4) deallocate(d1vxc)

!----- Calculate Excdc double counting term
  if (option==0.or.option==2) then
   ff(1:nrad)=zero
   do ispden=1,nspden
    if (usexcnhat==1.or.usecore==1) then
     do ilm=1,lm_size
      if (lmselect(ilm,ispden)>0) ff(1:nrad)=ff(1:nrad)+vxc(1:nrad,ilm,ispden)*rhoeffdc(1:nrad,ilm,ispden)
     end do
    else
     do ilm=1,lm_size
      if (lmselect(ilm,ispden)>0) ff(1:nrad)=ff(1:nrad)+vxc(1:nrad,ilm,ispden)*rhoeff(1:nrad,ilm,ispden)
     end do
    end if
   end do
   ff(1:nrad)=ff(1:nrad)*pawrad%rad(1:nrad)**2
   call simp_gen(enxcdc,ff,pawrad)
  end if

  if (option/=1) deallocate(ff)

!----- End of routine
  if (ndvxc/=0) deallocate(dvxci)
  if (nvxcdgr/=0) deallocate(dvxcdgr)
  if ((ixc==3 .or. (ixc>=7 .and. ixc<=15)) .and. order==3)deallocate(d2vxcar)
  if (ngrad == 2)deallocate(grho2_updn)
  deallocate(rho_updn,vxci)
  deallocate(exci)
  deallocate(dnexcdn,rhoeff,rhonow,rhosph)
  if ((usexcnhat==1.or.usecore==1).and.(option==0.or.option==2)) deallocate(rhoeffdc)

 end if  !End IF a xc part has to be computed

 call timab(81,2,tsec)

!DEBUG
!write(6,*)' pawxcm : exit '
!ENDDEBUG

 end subroutine pawxcm
!!***
