###############################################################################
#
# The MIT License (MIT)
#
# Copyright (c) Tavendo GmbH
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
###############################################################################

import sys
import six
import datetime

try:
    import asyncio
except ImportError:
    # Trollius >= 0.3 was renamed
    import trollius as asyncio

from autobahn.asyncio import wamp, websocket


class MyBackendComponent(wamp.ApplicationSession):

    """
    Application code goes here. This is an example component that provides
    a simple procedure which can be called remotely from any WAMP peer.
    It also publishes an event every second to some topic.
    """

    def onConnect(self):
        self.join(u"realm1")

    @asyncio.coroutine
    def onJoin(self, details):
        # register a procedure for remote calling
        ##
        def utcnow():
            print("Someone is calling me;)")
            now = datetime.datetime.utcnow()
            return six.u(now.strftime("%Y-%m-%dT%H:%M:%SZ"))

        reg = yield from self.register(utcnow, u'com.timeservice.now')
        print("Registered procedure with ID {}".format(reg.id))

        # publish events to a topic
        ##
        counter = 0
        while True:
            self.publish(u'com.myapp.topic1', counter)
            print("Published event.")
            counter += 1
            yield from asyncio.sleep(1)


if __name__ == '__main__':

    # 1) create a WAMP router factory
    router_factory = wamp.RouterFactory()

    # 2) create a WAMP router session factory
    session_factory = wamp.RouterSessionFactory(router_factory)

    # 3) Optionally, add embedded WAMP application sessions to the router
    session_factory.add(MyBackendComponent())

    # 4) create a WAMP-over-WebSocket transport server factory
    transport_factory = websocket.WampWebSocketServerFactory(session_factory,
                                                             debug=False,
                                                             debug_wamp=False)

    # 5) start the server
    loop = asyncio.get_event_loop()
    coro = loop.create_server(transport_factory, '127.0.0.1', 8080)
    server = loop.run_until_complete(coro)

    try:
        # 6) now enter the asyncio event loop
        loop.run_forever()
    except KeyboardInterrupt:
        pass
    finally:
        server.close()
        loop.close()
