"""ONETEP file parsers.

Implemented:
* Input file
* Output file
* Geometry output file
"""

from io import StringIO

import numpy as np
import pytest
from pytest import approx

from ase.io import read, write
from ase.io.onetep import get_onetep_keywords

eV2au = 27.2116529
angtobohr = 1.889726134583

# Very complex solvation input file with
# many custom defined species
# and constraints
onetep_input = """
task                   : SINGLEPOINT

ppd_npoints 7 5 5

is_implicit_solvent = T
is_smeared_ion_rep : t
is_sepaRate_restArt_fiLes T

 %block lattice_cart
 ang
 120.00000000   0.00000000   0.00000000
   0.00000000  90.00000000   0.00000000
   0.00000000   0.00000000  90.00000000
 %endblock lattice_cart

#==== ring-resolved PDOS
%block species_ldos_groups
H2  O2  Al2
H3  O3  Al3
H4  O4  Si4
H5  O5  Al5
H6  O6  Al6
H7  O7  Si7
H8  O8  Al8
H9  O9  Al9
H10 O10 Si10
  H11 O11 Al11
H12 O12 Al12
H13 O13 Si13
    H14 O14 Al14
H15 O15 Al15
H16 O16 Si16
H17 O17 Al17
H18 O18 Al18
H19 O19 Si19
    H20 O20 Al20
H21 O21 Al21
H22 O22 Si22
H23 O23 Al23
H24 O24 Al24
HOH OOH
%endblock species_ldos_groups

%block species_cond
H2    H   1   1  13.0
O2    O   8   4  13.0
Al2   Al  13  9  13.0
H3    H   1   1  13.0
O3    O   8   4  13.0
Al3   Al  13  9  13.0
H4    H   1   1  13.0
O4    O   8   4  13.0
Si4   Si  14  9  13.0
H5    H   1   1  13.0
O5    O   8   4  13.0
Al5   Al  13  9  13.0
H6    H   1   1  13.0
O6    O   8   4  13.0
Al6   Al  13  9  13.0
H7    H   1   1  13.0
O7    O   8   4  13.0
Si7   Si  14  9  13.0
H8    H   1   1  13.0
O8    O   8   4  13.0
Al8   Al  13  9  13.0
H9    H   1   1  13.0
O9    O   8   4  13.0
Al9   Al  13  9  13.0
H10    H   1   1  13.0
O10    O   8   4  13.0
Si10   Si  14  9  13.0
H11    H   1   1  13.0
O11    O   8   4  13.0
Al11   Al  13  9  13.0
H12    H   1   1  13.0
O12    O   8   4  13.0
Al12   Al  13  9  13.0
H13    H   1   1  13.0
O13    O   8   4  13.0
Si13   Si  14  9  13.0
H14    H   1   1  13.0
O14    O   8   4  13.0
Al14   Al  13  9  13.0
H15    H   1   1  13.0
O15    O   8   4  13.0
Al15   Al  13  9  13.0
H16    H   1   1  13.0
O16    O   8   4  13.0
Si16   Si  14  9  13.0
H17    H   1   1  13.0
O17    O   8   4  13.0
Al17   Al  13  9  13.0
H18    H   1   1  13.0
O18    O   8   4  13.0
Al18   Al  13  9  13.0
H19    H   1   1  13.0
O19    O   8   4  13.0
Si19   Si  14  9  13.0
H20    H   1   1  13.0
O20    O   8   4  13.0
Al20   Al  13  9  13.0
H21    H   1   1  13.0
O21    O   8   4  13.0
Al21   Al  13  9  13.0
H22    H   1   1  13.0
O22    O   8   4  13.0
Si22   Si  14  9  13.0
H23    H   1   1  13.0
O23    O   8   4  13.0
Al23   Al  13  9  13.0
H24    H   1   1  13.0
O24    O   8   4  13.0
Al24   Al  13  9  13.0
HOH    H   1   1  13.0
OOH    O   8   4  13.0
%endblock species_cond

%block species_atomic_set
H2      SOLVE
O2      SOLVE
Al2     SOLVE
H3      SOLVE
O3      SOLVE
Al3     SOLVE
H4      SOLVE
O4      SOLVE
Si4     SOLVE
H5      SOLVE
O5      SOLVE
Al5     SOLVE
H6      SOLVE
O6      SOLVE
Al6     SOLVE
H7      SOLVE
O7      SOLVE
Si7     SOLVE
H8      SOLVE
O8      SOLVE
Al8     SOLVE
H9      SOLVE
O9      SOLVE
Al9     SOLVE
H10      SOLVE
O10      SOLVE
Si10     SOLVE
H11      SOLVE
O11      SOLVE
Al11     SOLVE
H12      SOLVE
O12      SOLVE
Al12     SOLVE
H13      SOLVE
O13      SOLVE
Si13     SOLVE
H14      SOLVE
O14      SOLVE
Al14     SOLVE
H15      SOLVE
O15      SOLVE
Al15     SOLVE
H16      SOLVE
O16      SOLVE
Si16     SOLVE
H17      SOLVE
O17      SOLVE
Al17     SOLVE
H18      SOLVE
O18      SOLVE
Al18     SOLVE
H19      SOLVE
O19      SOLVE
Si19     SOLVE
H20      SOLVE
O20      SOLVE
Al20     SOLVE
H21      SOLVE
O21      SOLVE
Al21     SOLVE
H22      SOLVE
O22      SOLVE
Si22     SOLVE
H23      SOLVE
O23      SOLVE
Al23     SOLVE
H24      SOLVE
O24      SOLVE
Al24     SOLVE
HOH      SOLVE
OOH      SOLVE
%endblock species_atomic_set

%block species_pot
H2      h-optgga1.recpot
O2      o-optgga1.recpot
Al2    al-optgga1.recpot
H3      h-optgga1.recpot
O3      o-optgga1.recpot
Al3    al-optgga1.recpot
H4      h-optgga1.recpot
O4      o-optgga1.recpot
Si4     Si_op_07_nopcc.recpot
H5      h-optgga1.recpot
O5      o-optgga1.recpot
Al5    al-optgga1.recpot
H6      h-optgga1.recpot
O6      o-optgga1.recpot
Al6    al-optgga1.recpot
H7      h-optgga1.recpot
O7      o-optgga1.recpot
Si7     Si_op_07_nopcc.recpot
H8      h-optgga1.recpot
O8      o-optgga1.recpot
Al8    al-optgga1.recpot
H9      h-optgga1.recpot
O9      o-optgga1.recpot
Al9    al-optgga1.recpot
H10      h-optgga1.recpot
O10      o-optgga1.recpot
Si10     Si_op_07_nopcc.recpot
H11      h-optgga1.recpot
O11      o-optgga1.recpot
Al11    al-optgga1.recpot
H12      h-optgga1.recpot
O12      o-optgga1.recpot
Al12    al-optgga1.recpot
H13      h-optgga1.recpot
O13      o-optgga1.recpot
Si13     Si_op_07_nopcc.recpot
H14      h-optgga1.recpot
O14      o-optgga1.recpot
Al14    al-optgga1.recpot
H15      h-optgga1.recpot
O15      o-optgga1.recpot
Al15    al-optgga1.recpot
H16      h-optgga1.recpot
O16      o-optgga1.recpot
Si16     Si_op_07_nopcc.recpot
H17      h-optgga1.recpot
O17      o-optgga1.recpot
Al17    al-optgga1.recpot
H18      h-optgga1.recpot
O18      o-optgga1.recpot
Al18    al-optgga1.recpot
H19      h-optgga1.recpot
O19      o-optgga1.recpot
Si19     Si_op_07_nopcc.recpot
H20      h-optgga1.recpot
O20      o-optgga1.recpot
Al20    al-optgga1.recpot
H21      h-optgga1.recpot
O21      o-optgga1.recpot
Al21    al-optgga1.recpot
H22      h-optgga1.recpot
O22      o-optgga1.recpot
Si22     Si_op_07_nopcc.recpot
H23      h-optgga1.recpot
O23      o-optgga1.recpot
Al23    al-optgga1.recpot
H24      h-optgga1.recpot
O24      o-optgga1.recpot
Al24    al-optgga1.recpot
HOH      h-optgga1.recpot
OOH      o-optgga1.recpot
%endblock species_pot


%block species
H2    H   1   1  8.0
O2    O   8   4  8.0
Al2   Al  13  9  8.0
H3    H   1   1  8.0
O3    O   8   4  8.0
Al3   Al  13  9  8.0
H4    H   1   1  8.0
O4    O   8   4  8.0
Si4   Si  14  9  8.0
H5    H   1   1  8.0
O5    O   8   4  8.0
Al5   Al  13  9  8.0
H6    H   1   1  8.0
O6    O   8   4  8.0
Al6   Al  13  9  8.0
H7    H   1   1  8.0
O7    O   8   4  8.0
Si7   Si  14  9  8.0
H8    H   1   1  8.0
O8    O   8   4  8.0
Al8   Al  13  9  8.0
H9    H   1   1  8.0
O9    O   8   4  8.0
Al9   Al  13  9  8.0
H10    H   1   1  8.0
O10    O   8   4  8.0
Si10   Si  14  9  8.0
H11    H   1   1  8.0
O11    O   8   4  8.0
Al11   Al  13  9  8.0
H12    H   1   1  8.0
O12    O   8   4  8.0
Al12   Al  13  9  8.0
H13    H   1   1  8.0
O13    O   8   4  8.0
Si13   Si  14  9  8.0
H14    H   1   1  8.0
O14    O   8   4  8.0
Al14   Al  13  9  8.0
H15    H   1   1  8.0
O15    O   8   4  8.0
Al15   Al  13  9  8.0
H16    H   1   1  8.0
O16    O   8   4  8.0
Si16   Si  14  9  8.0
H17    H   1   1  8.0
O17    O   8   4  8.0
Al17   Al  13  9  8.0
H18    H   1   1  8.0
O18    O   8   4  8.0
Al18   Al  13  9  8.0
H19    H   1   1  8.0
O19    O   8   4  8.0
Si19   Si  14  9  8.0
H20    H   1   1  8.0
O20    O   8   4  8.0
Al20   Al  13  9  8.0
H21    H   1   1  8.0
O21    O   8   4  8.0
Al21   Al  13  9  8.0
H22    H   1   1  8.0
O22    O   8   4  8.0
Si22   Si  14  9  8.0
H23    H   1   1  8.0
O23    O   8   4  8.0
Al23   Al  13  9  8.0
H24    H   1   1  8.0
O24    O   8   4  8.0
Al24   Al  13  9  8.0
HOH    H   1   1  8.0
OOH    O   8   4  8.0
%endblock species


%block species_constraints
H2    NONE
O2    NONE
Al2   NONE
H3    NONE
O3    NONE
Al3   NONE
H4    NONE
O4    NONE
Si4   NONE
H5    NONE
O5    NONE
Al5   NONE
H6    NONE
O6    NONE
Al6   NONE
H7    NONE
O7    NONE
Si7   NONE
H8    NONE
O8    NONE
Al8   NONE
H9    NONE
O9    NONE
Al9   NONE
H10   NONE
O10   NONE
Si10  NONE
H11   NONE
O11   NONE
Al11  NONE
H12   NONE
O12   No:N
Al12  NONE
H13   NONE
O13   N=NE
Si13  NONE
H14   NONE
O14   NONE
Al14  NONE
H15   NONE
O15   NONE
Al15  NONE
H16   NONE
O16   NONE
Si16  NONE
H17   NONE
O17   NONE
Al17  NONE
H18   NONE
O18   NONE
Al18  NONE
H19   NONE
O19   NONE
Si19  NONE
H20   NONE
O20   NONE
Al20  NONE
H21   NONE
O21   NONE
Al21  NONE
H22   NONE
O22   NONE
Si22  NONE
H23   NONE
O23   NONE
Al23  NoNe
H24   NONE
O24   NONE
Al24  NONE
HOH   NONE
OOH   NONE
%endblock species_constraints

dos_smear               : 0.1 eV
kernel_cutoff          : 1000.0
cutoff_energy          : 1000 eV
odd_psinc_grid         : T
xc_functional          : B3LYP
timings_level          : 3
output_detail          : VERBOSE
check_atoms            : T

do_properties          : T
dx_format T
cube_format F
read_denskern          : T
read_tightbox_ngwfs    : T

write_denskern         : T
write_tightbox_ngwfs   : T

initial_dens_realspace  : T

# Construct initial density in real space from atomsolver density
# problems with kernel degeneracy: diagonalise it if broken
maxit_palser_mano   = -1
maxit_pen              : 0
minit_lnv              : 8
maxit_lnv              : 8
maxit_ngwf_cg          : 20
lnv_check_trial_steps  : T

tHREADS_NUM_FFTBOXES 8

! hfx_memory_limit 95000 ! przy populate swop OOM
! hfx_memory_limit 85000 ! po prodach
! hfx_memory_limit 75000 ! po prodach na 64
! hfx_memory_limit 60000 ! takze oom na 64
! hfx_memory_limit 50000 ! takze oom na 64
! hfx_memory_limit 50000 ! takze oom na 128
! hfx_memory_limit 40000 ! oom na 128
! hfx_memory_limit 35000 ! oom na 128 w drugim grad

hfx_memory_limit 32000

hfx_memory_weights 1 0 0

%block swri
  for_hfx 3 12 V 12 12 RE2
%endblock swri

hfx_use_ri for_hfx
hfx_max_l 3
hfx_max_q 12
hfx_metric ELECTROSTATIC
hfx_cutoff 20.0 bohr

%block species_swri-for_hfx
H2
O2
Al2
H3
O3
Al3
H4
O4
Si4
H5
O5
Al5
H6
O6
Al6
H7
O7
Si7
H8
O8
Al8
H9
O9
Al9
H10
O10
Si10
H11
O11
Al11
H12
O12
Al12
H13
O13
Si13
H14
O14
Al14
H15
O15
Al15
H16
O16
Si16
H17
O17
Al17
H18
O18
Al18
H19
O19
Si19
H20
O20
Al20
H21
O21
Al21
H22
O22
Si22
H23
O23
Al23
H24
O24
Al24
HOH
OOH
%endblock species_swri-for_hfx

 %block positions_abs
Al2      89.96156402      49.81122736      62.56876970
O2      92.92087542      49.47107662      65.41280780
H2      92.07238831      48.22763671      66.46727508
H2      94.38352359      48.67739157      64.65313782
O2      92.50135618      50.76742887      60.40125362
H2      92.82638911      49.48619443      59.14458562
Al2      89.96345374      57.94460941      57.80287994
O2      92.91709597      59.06899656      60.44849677
H2      92.06104995      58.52475539      61.98106481
H2      94.38730304      58.00697038      60.20472208
O2      92.50135618      57.68382718      55.45017068
H2      92.81694048      55.94527897      54.99663636
Al2      89.96912292      62.60467450      49.61658555
O2      92.93221378      64.89313306      51.34946458
H2      92.06293968      65.19737900      52.94061413
H2      94.38730304      63.83488633      51.68772559
O2      92.50324591      61.19682839      47.70607224
H2      92.81883020      59.46583909      48.18039355
Al2      89.96156402      40.18874097      27.43119863
O2      92.92087542      40.52889171      24.58716053
H2      92.07238831      41.77233162      23.53269325
H2      94.38352359      41.32257676      25.34683051
O2      92.50135618      39.23253946      29.59871471
H2      92.82638911      40.51377390      30.85538271
Al2      89.96345374      32.05535892      32.19708839
O2      92.91709597      30.93097176      29.55147156
H2      92.06104995      31.47521294      28.01890352
H2      94.38730304      31.99299795      29.79524625
O2      92.50135618      32.31614115      34.54979765
H2      92.81694048      34.05468936      35.00333197
Al2      89.96912292      27.39529383      40.38338278
O2      92.93221378      25.10683527      38.65050375
H2      92.06293968      24.80258933      37.05935420
H2      94.38730304      26.16508200      38.31224274
O2      92.50324591      28.80313994      42.29389608
H2      92.81883020      30.53412924      41.81957478
Al2      89.96156402      62.56876970      40.18874097
O2      92.92087542      65.41280780      40.52889171
H2      92.07238831      66.46727508      41.77233162
H2      94.38352359      64.65313782      41.32257676
O2      92.50135618      60.40125362      39.23253946
H2      92.82638911      59.14458562      40.51377390
Al2      89.96345374      57.80287994      32.05535892
O2      92.91709597      60.44849677      30.93097176
H2      92.06104995      61.98106481      31.47521294
H2      94.38730304      60.20472208      31.99299795
O2      92.50135618      55.45017068      32.31614115
H2      92.81694048      54.99663636      34.05468936
Al2      89.96912292      49.61658555      27.39529383
O2      92.93221378      51.34946458      25.10683527
H2      92.06293968      52.94061413      24.80258933
H2      94.38730304      51.68772559      26.16508200
O2      92.50324591      47.70607224      28.80313994
H2      92.81883020      48.18039355      30.53412924
Al2      89.96156402      27.43119863      49.81122736
O2      92.92087542      24.58716053      49.47107662
H2      92.07238831      23.53269325      48.22763671
H2      94.38352359      25.34683051      48.67739157
O2      92.50135618      29.59871471      50.76742887
H2      92.82638911      30.85538271      49.48619443
Al2      89.96345374      32.19708839      57.94460941
O2      92.91709597      29.55147156      59.06899656
H2      92.06104995      28.01890352      58.52475539
H2      94.38730304      29.79524625      58.00697038
O2      92.50135618      34.54979765      57.68382718
H2      92.81694048      35.00333197      55.94527897
Al2      89.96912292      40.38338278      62.60467450
O2      92.93221378      38.65050375      64.89502279
H2      92.06293968      37.05935420      65.19737900
H2      94.38730304      38.31224274      63.83488633
O2      92.50324591      42.29389608      61.19682839
H2      92.81883020      41.81957478      59.46583909
Al3      87.29704991      45.07368349      63.63646506
O3      89.84440098      46.44373507      61.59934010
H3      91.14264296      45.12848555      61.20060785
O3      88.21545690      48.15960656      65.40335917
H3      87.02114987      49.02321148      66.46538535
O3      89.50236052      42.68884888      65.34855710
H3      91.01225185      43.22364143      66.20271340
Al3      87.29327046      54.38625476      61.10045235
O3      89.83873181      54.54310205      58.64947732
H3      91.13886351      53.20895527      58.97262052
O3      89.48724271      53.18060937      63.78764317
H3      90.99146486      54.07067047      64.27897201
O3      88.21356718      57.94271968      61.08155509
H3      87.01737042      59.22395412      61.56154557
Al3      87.29327046      61.17415168      54.24830474
O3      89.84062153      60.08377959      52.04677359
H3      91.13886351      59.08600410      52.99541619
O3      89.49102216      61.47083871      57.17171135
H3      90.99335458      62.49129092      57.14336545
O3      88.22301581      64.24495694      52.46062365
H3      87.02870878      65.59611125      52.22629759
Al3      87.29704991      44.92628484      26.36350327
O3      89.84440098      43.55623326      28.40062823
H3      91.14264296      44.87148278      28.79936048
O3      88.21545690      41.84036177      24.59660916
H3      87.02114987      40.97675684      23.53458298
O3      89.50236052      47.31111945      24.65141123
H3      91.01225185      46.77632690      23.79725493
Al3      87.29327046      35.61371357      28.89951598
O3      89.83873181      35.45686628      31.35049101
H3      91.13886351      36.79101306      31.02734781
O3      89.48724271      36.81935896      26.21232516
H3      90.99146486      35.92929786      25.72099632
O3      88.21356718      32.05724865      28.91841324
H3      87.01737042      30.77601421      28.43842276
Al3      87.29327046      28.82581665      35.75166359
O3      89.84062153      29.91618873      37.95319474
H3      91.13886351      30.91396423      37.00455213
O3      89.49102216      28.52912962      32.82825698
H3      90.99335458      27.50867741      32.85660288
O3      88.22301581      25.75501139      37.53934468
H3      87.02870878      24.40385708      37.77367074
Al3      87.29704991      63.63646506      44.92628484
O3      89.50236052      65.34855710      47.31111945
H3      91.01225185      66.20271340      46.77632690
O3      89.84440098      61.59934010      43.55623326
H3      91.14264296      61.20060785      44.87148278
O3      88.21545690      65.40335917      41.84036177
H3      87.02114987      66.46538535      40.97675684
Al3      87.29327046      61.10045235      35.61371357
O3      89.83873181      58.64947732      35.45686628
H3      91.13886351      58.97262052      36.79101306
O3      89.48724271      63.78764317      36.81935896
H3      90.99146486      64.27897201      35.92929786
O3      88.21356718      61.08155509      32.05724865
H3      87.01737042      61.56154557      30.77601421
Al3      87.29327046      54.24830474      28.82581665
O3      89.84062153      52.04677359      29.91618873
H3      91.13886351      52.99541619      30.91396423
O3      89.49102216      57.17171135      28.52912962
H3      90.99335458      57.14336545      27.50867741
O3      88.22301581      52.46062365      25.75501139
H3      87.02870878      52.22629759      24.40385708
Al3      87.29704991      26.36350327      45.07368349
O3      89.50236052      24.65141123      42.68884888
H3      91.01225185      23.79725493      43.22364143
O3      89.84440098      28.40062823      46.44373507
H3      91.14264296      28.79936048      45.12848555
O3      88.21545690      24.59660916      48.15960656
H3      87.02114987      23.53458298      49.02321148
Al3      87.29327046      28.89951598      54.38625476
O3      89.83873181      31.35238073      54.54310205
H3      91.13886351      31.02734781      53.20895527
O3      89.48724271      26.21232516      53.18060937
H3      90.99146486      25.72099632      54.07067047
O3      88.21356718      28.91841324      57.94271968
H3      87.01737042      28.43842276      59.22395412
Al3      87.29327046      35.75166359      61.17415168
O3      89.84062153      37.95319474      60.08377959
H3      91.13886351      37.00455213      59.08600410
O3      89.49102216      32.82825698      61.47083871
H3      90.99335458      32.85660288      62.49129092
O3      88.22301581      37.53934468      64.24495694
H3      87.02870878      37.77367074      65.59611125
Si4      84.40765838      49.15927178      60.08755905
O4      84.39254057      48.54322100      57.03565105
H4      84.32828988      46.76120909      56.61046263
O4      86.95311973      50.77876723      60.84533930
Si4      84.40576866      56.13992078      55.97929404
O4      84.38309194      54.08011910      53.64548205
H4      84.33395906      52.32456335      54.17082596
O4      86.95123000      57.92193269      55.82622621
Si4      84.40576866      60.14425084      48.93628407
O4      84.37931249      57.19438806      47.94039831
H4      84.32451042      55.93205089      49.26887591
O4      86.95311973      61.60878873      47.91583187
Si4      84.40765838      40.84069655      29.91240928
O4      84.39254057      41.45674733      32.96431728
H4      84.32828988      43.23875924      33.38950570
O4      86.95311973      39.22120110      29.15462903
Si4      84.40576866      33.86004755      34.02067429
O4      84.38309194      35.91984923      36.35448628
H4      84.33395906      37.67540498      35.82914237
O4      86.95123000      32.07803564      34.17374212
Si4      84.40576866      29.85571749      41.06368425
O4      84.37931249      32.80558027      42.05957002
H4      84.32451042      34.06791744      40.73109242
O4      86.95311973      28.39117960      42.08413646
Si4      84.40765838      60.08755905      40.84069655
O4      84.39254057      57.03565105      41.45674733
H4      84.32828988      56.61046263      43.23875924
O4      86.95311973      60.84533930      39.22120110
Si4      84.40576866      55.97929404      33.86004755
O4      84.38309194      53.64548205      35.91984923
H4      84.33395906      54.17082596      37.67540498
O4      86.95123000      55.82622621      32.07803564
Si4      84.40576866      48.93628407      29.85571749
O4      84.37931249      47.94039831      32.80558027
H4      84.32451042      49.26887591      34.06791744
O4      86.95311973      47.91583187      28.39117960
Si4      84.40765838      29.91240928      49.15927178
O4      84.39254057      32.96431728      48.54322100
H4      84.32828988      33.38950570      46.76120909
O4      86.95311973      29.15462903      50.77876723
Si4      84.40576866      34.02067429      56.13992078
O4      84.38309194      36.35448628      54.08011910
H4      84.33395906      35.82914237      52.32456335
O4      86.95123000      34.17374212      57.92193269
Si4      84.40576866      41.06368425      60.14425084
O4      86.95311973      42.08413646      61.60878873
O4      84.37931249      42.05957002      57.19438806
H4      84.32451042      40.73109242      55.93205089
Al5      81.73558538      45.10202939      63.73095138
O5      84.43600428      46.55711865      61.78831273
O5      84.59474129      44.43117654      65.82854759
H5      84.58151320      42.97986674      66.91325049
Al5      81.73558538      54.45239518      61.16659277
O5      84.59096183      54.92860621      63.31899104
H5      84.58529266      54.21239994      64.98383992
O5      84.43411455      54.73774386      58.75341227
Al5      81.73558538      61.26863799      54.27287118
O5      84.59285156      62.75585260      55.89992554
H5      84.58907211      62.97128140      57.69894499
O5      84.43411455      60.31432621      52.03921468
Al5      81.73558538      44.89793894      26.26901695
O5      84.43600428      43.44284968      28.21165560
O5      84.59474129      45.56879178      24.17142074
H5      84.58151320      47.02010159      23.08671784
Al5      81.73558538      35.54757315      28.83337556
O5      84.59096183      35.07136212      26.68097729
H5      84.58529266      35.78756839      25.01612840
O5      84.43411455      35.26222447      31.24655606
Al5      81.73558538      28.73322006      35.72709715
O5      84.59285156      27.24411573      34.10004279
H5      84.58907211      27.02868693      32.30102334
O5      84.43411455      29.68564212      37.96075365
Al5      81.73558538      63.73095138      44.89793894
O5      84.59474129      65.82854759      45.56879178
H5      84.58151320      66.91325049      47.02010159
O5      84.43600428      61.78831273      43.44284968
Al5      81.73558538      61.16659277      35.54757315
O5      84.59096183      63.31899104      35.07136212
H5      84.58529266      64.98383992      35.78756839
O5      84.43411455      58.75341227      35.26222447
Al5      81.73558538      54.27287118      28.73133034
O5      84.59285156      55.89992554      27.24411573
H5      84.58907211      57.69894499      27.02868693
O5      84.43411455      52.03921468      29.68564212
Al5      81.73558538      26.26901695      45.10202939
O5      84.59474129      24.17142074      44.43117654
H5      84.58151320      23.08671784      42.97986674
O5      84.43600428      28.21165560      46.55711865
Al5      81.73558538      28.83337556      54.45239518
O5      84.59096183      26.68097729      54.92860621
H5      84.58529266      25.01612840      54.21239994
O5      84.43411455      31.24655606      54.73774386
Al5      81.73558538      35.72709715      61.26674827
O5      84.59285156      34.10004279      62.75585260
H5      84.58907211      32.30102334      62.97128140
O5      84.43411455      37.96075365      60.31432621
Al6      79.10886580      49.93972875      63.05064991
O6      81.87542512      50.81656176      60.86801601
O6      81.06284281      48.16338601      65.50540439
H6      82.42722521      48.91360736      66.43326001
O6      79.81751317      53.36013337      63.98417470
H6      78.56840408      54.35223969      64.84211045
Al6      79.10886580      58.29987796      58.16003821
O6      81.87542512      57.96917585      55.82811594
O6      81.06284281      57.98996284      61.17226195
H6      82.42533548      59.10679109      61.59934010
O6      79.82129262      61.72784149      57.26241821
H6      78.57029380      63.01474511      57.50619290
Al6      79.10697607      63.09978279      49.74697666
O6      81.87542512      61.65414216      47.89315515
O6      81.06284281      64.33377407      52.51353598
H6      82.43100466      65.50729411      52.32078390
O6      79.81751317      65.62256742      47.25631738
H6      78.56084517      66.85466897      46.82545978
Al6      79.10886580      40.06023958      26.94931842
O6      81.87542512      39.18340657      29.13195231
O6      81.06284281      41.83658232      24.49456394
H6      82.42722521      41.08636097      23.56670832
O6      79.81751317      36.63983496      26.01579362
H6      78.56840408      35.64772864      25.15785788
Al6      79.10886580      31.70009037      31.84181985
O6      81.87542512      32.03079248      34.17185239
O6      81.06284281      32.01000549      28.82770638
H6      82.42533548      30.89317724      28.40062823
O6      79.82129262      28.27212684      32.73755012
H6      78.57029380      26.98522322      32.49377543
Al6      79.10697607      26.90018554      40.25299167
O6      81.87542512      28.34582617      42.10681318
O6      81.06284281      25.66619426      37.48643234
H6      82.43100466      24.49267422      37.67918443
O6      79.81751317      24.37740091      42.74365095
H6      78.56084517      23.14529935      43.17450855
Al6      79.10886580      63.05064991      40.06023958
O6      81.87542512      60.86801601      39.18340657
O6      81.06284281      65.50540439      41.83658232
H6      82.42722521      66.43326001      41.08636097
O6      79.81751317      63.98417470      36.63983496
H6      78.56840408      64.84211045      35.64772864
Al6      79.10886580      58.15814848      31.70009037
O6      81.87542512      55.82811594      32.03079248
O6      81.06284281      61.17226195      32.01000549
H6      82.42533548      61.59934010      30.89317724
O6      79.82129262      57.26241821      28.27212684
H6      78.57029380      57.50619290      26.98522322
Al6      79.10697607      49.74697666      26.90018554
O6      81.87542512      47.89315515      28.34582617
O6      81.06284281      52.51353598      25.66619426
H6      82.43100466      52.32078390      24.49267422
O6      79.81751317      47.25631738      24.37740091
H6      78.56084517      46.82545978      23.14529935
Al6      79.10886580      26.94931842      49.93972875
O6      81.87542512      29.13195231      50.81656176
O6      81.06284281      24.49456394      48.16338601
H6      82.42722521      23.56670832      48.91360736
O6      79.81751317      26.01579362      53.36013337
H6      78.56840408      25.15785788      54.35223969
Al6      79.10886580      31.84181985      58.29987796
O6      81.87542512      34.17185239      57.96917585
O6      81.06284281      28.82770638      57.98996284
H6      82.42533548      28.40062823      59.10679109
O6      79.82129262      32.73755012      61.72784149
H6      78.57029380      32.49377543      63.01474511
Al6      79.10697607      40.25299167      63.09978279
O6      81.87542512      42.10681318      61.65414216
O6      81.06284281      37.48643234      64.33377407
H6      82.43100466      37.67918443      65.50729411
O6      79.81751317      42.74365095      65.62256742
H6      78.56084517      43.17450855      66.85466897
Si7      76.35931402      52.95195249      58.48318141
O7      76.39521882      51.59890845      55.68260701
H7      76.37254210      49.76398420      55.72040154
O7      78.90855481      54.73207468      58.84600886
Si7      76.36120374      58.62491088      52.69494971
O7      76.40844690      56.04732419      50.95073232
H7      76.37632155      54.47885135      51.90126466
O7      78.90855481      60.35023100      52.12047291
Si7      76.35931402      45.14927254      60.65447694
O7      78.90666509      46.50798576      61.86579151
O7      76.40466745      45.37792943      57.55154634
H7      76.37821128      43.77355179      56.66337497
Si7      76.35931402      44.85069579      29.34549139
O7      76.40466745      44.62203890      32.44842199
H7      76.37821128      46.22641654      33.33659336
O7      78.90666509      43.49198257      28.13417682
Si7      76.35931402      37.04801584      31.51678692
O7      76.39521882      38.40105988      34.31736132
H7      76.37254210      40.23598413      34.27956679
O7      78.90855481      35.26789365      31.15395947
Si7      76.36120374      31.37505745      37.30501862
O7      76.40844690      33.95264414      39.04923600
H7      76.37632155      35.52111698      38.09870367
O7      78.90855481      29.64973732      37.87949542
Si7      76.35931402      60.65447694      44.85069579
O7      76.40466745      57.55154634      44.62203890
H7      76.37821128      56.66337497      46.22641654
O7      78.90666509      61.86579151      43.49198257
Si7      76.35931402      58.48318141      37.04801584
O7      76.39521882      55.68260701      38.40105988
H7      76.37254210      55.72040154      40.23598413
O7      78.90855481      58.84600886      35.26789365
Si7      76.36120374      52.69494971      31.37505745
O7      76.40844690      50.95073232      33.95264414
H7      76.37632155      51.90126466      35.52111698
O7      78.90855481      52.12047291      29.64973732
Si7      76.35931402      29.34549139      45.14927254
O7      76.40466745      32.44842199      45.37792943
H7      76.37821128      33.33659336      43.77355179
O7      78.90666509      28.13417682      46.50798576
Si7      76.35931402      31.51678692      52.95195249
O7      76.39521882      34.31736132      51.59890845
H7      76.37254210      34.27956679      49.76398420
O7      78.90855481      31.15395947      54.73207468
Si7      76.36120374      37.30501862      58.62491088
O7      76.40844690      39.04923600      56.04732419
H7      76.37632155      38.09870367      54.47885135
O7      78.90855481      37.87949542      60.35023100
Al8      73.56818825      49.93216984      62.99773757
O8      76.32907839      49.84902188      65.25029133
H8      76.31396058      48.67928130      66.63546072
O8      76.34986538      50.83734875      60.76597079
Al8      73.57007798      58.26586288      58.11657450
O8      76.33285785      59.32033016      60.11023576
H8      76.31585031      59.00285614      61.89413740
O8      76.35175511      57.93894023      55.73173990
Al8      73.56818825      63.04498073      49.72996913
O8      76.32907839      64.95549403      50.92994533
H8      76.31585031      65.56965508      52.63447847
O8      76.34797566      61.57477366      47.82323528
Al8      73.56818825      40.06779849      27.00412049
O8      76.32907839      40.15094645      24.74967699
H8      76.31396058      41.32068703      23.36450761
O8      76.34986538      39.16261958      29.23399754
Al8      73.57007798      31.73410545      31.88339383
O8      76.33285785      30.67963816      29.88973257
H8      76.31585031      30.99711219      28.10583093
O8      76.35175511      32.06102810      34.26822843
Al8      73.56818825      26.95498760      40.26999920
O8      76.32907839      25.04447430      39.07002299
H8      76.31585031      24.43031325      37.36548986
O8      76.34797566      28.42519467      42.17673305
Al8      73.56818825      62.99584784      40.06779849
O8      76.32907839      65.25029133      40.15094645
H8      76.31396058      66.63546072      41.32068703
O8      76.34986538      60.76597079      39.16261958
Al8      73.57007798      58.11657450      31.73410545
O8      76.33285785      60.11023576      30.67963816
H8      76.31585031      61.89413740      30.99711219
O8      76.35175511      55.73173990      32.06102810
Al8      73.56818825      49.72996913      26.95498760
O8      76.32907839      50.92994533      25.04447430
H8      76.31585031      52.63447847      24.43031325
O8      76.34797566      47.82323528      28.42519467
Al8      73.56818825      27.00412049      49.93216984
O8      76.32907839      24.74967699      49.84902188
H8      76.31396058      23.36450761      48.67928130
O8      76.34986538      29.23399754      50.83734875
Al8      73.57007798      31.88339383      58.26586288
O8      76.33285785      29.88973257      59.32033016
H8      76.31585031      28.10583093      59.00285614
O8      76.35175511      34.26822843      57.93894023
Al8      73.56818825      40.26999920      63.04498073
O8      76.32907839      39.07002299      64.95549403
H8      76.31585031      37.36548986      65.56965508
O8      76.34797566      42.17673305      61.57477366
Al9      70.94902758      45.10013966      63.64213424
O9      73.77416842      46.50609604      61.79020246
O9      71.61988042      48.15204765      65.42225643
H9      70.32163845      48.87203338      66.45971617
O9      72.88221760      42.75498930      65.54886809
H9      74.21258492      43.21041335      66.69026279
Al9      70.95280703      54.41460066      61.09667290
O9      73.77794787      54.69994933      58.79309652
O9      72.89166623      53.34123611      63.91425483
H9      74.21825410      54.30877598      64.67581454
O9      71.62177015      57.94649913      61.10423180
H9      70.32541790      59.09356301      61.63713462
Al9      70.95091731      61.19115921      54.23318693
O9      73.77605815      60.28598031      52.08834756
O9      72.88977650      61.67303942      57.20950587
H9      74.21825410      62.89191290      57.37958124
O9      71.62177015      64.25818502      52.48330036
H9      70.32352817      65.51485302      52.37180651
Al9      70.94902758      44.89982867      26.35783409
O9      73.77416842      43.49387229      28.20976587
O9      71.61988042      41.84792067      24.57771190
H9      70.32163845      41.12793495      23.54025215
O9      72.88221760      47.24497902      24.45110024
H9      74.21258492      46.78955498      23.30970554
Al9      70.95280703      35.58536767      28.90329543
O9      73.77794787      35.30001900      31.20687181
O9      72.89166623      36.65873222      26.08571350
H9      74.21825410      35.69119235      25.32415379
O9      71.62177015      32.05346919      28.89573653
H9      70.32541790      30.90640532      28.36283370
Al9      70.95091731      28.80880911      35.76678140
O9      73.77605815      29.71398802      37.91162076
O9      72.88977650      28.32692890      32.79046246
H9      74.21825410      27.10805543      32.62038709
O9      71.62177015      25.74178331      37.51666797
H9      70.32352817      24.48511531      37.62816182
Al9      70.94902758      63.64213424      44.89982867
O9      72.88221760      65.54886809      47.24497902
H9      74.21258492      66.69026279      46.78955498
O9      73.77416842      61.79020246      43.49387229
O9      71.61988042      65.42225643      41.84792067
H9      70.32163845      66.45971617      41.12793495
Al9      70.95280703      61.09667290      35.58536767
O9      73.77794787      58.79309652      35.30001900
O9      72.89166623      63.91425483      36.65873222
H9      74.21825410      64.67581454      35.69119235
O9      71.62177015      61.10423180      32.05346919
H9      70.32541790      61.63713462      30.90640532
Al9      70.95091731      54.23318693      28.80880911
O9      73.77605815      52.08834756      29.71398802
O9      72.88977650      57.20950587      28.32692890
H9      74.21825410      57.37958124      27.10805543
O9      71.62177015      52.48330036      25.74178331
H9      70.32352817      52.37180651      24.48511531
Al9      70.94902758      26.35783409      45.10013966
O9      72.88221760      24.45110024      42.75498930
H9      74.21258492      23.30970554      43.21041335
O9      73.77416842      28.20976587      46.50609604
O9      71.61988042      24.57771190      48.15204765
H9      70.32163845      23.54025215      48.87203338
Al9      70.95280703      28.90329543      54.41460066
O9      73.77794787      31.20687181      54.69994933
O9      72.89166623      26.08571350      53.34123611
H9      74.21825410      25.32415379      54.30877598
O9      71.62177015      28.89573653      57.94649913
H9      70.32541790      28.36283370      59.09356301
Al9      70.95091731      35.76678140      61.19115921
O9      73.77605815      37.91162076      60.28598031
O9      72.88977650      32.79046246      61.67303942
H9      74.21825410      32.62038709      62.89191290
O9      71.62177015      37.51666797      64.25818502
H9      70.32352817      37.62816182      65.51485302
Si10      68.18057853      49.18950740      60.04031589
O10      68.16924017      48.60936142      56.98273872
H10      68.17490935      46.82923924      56.54432221
O10      70.75438577      50.79388504      60.81888313
Si10      68.18057853      56.13614133      55.93205089
O10      68.16924017      54.09901636      53.58123135
H10      68.17112990      52.33779144      54.09523691
O10      70.75438577      57.91626351      55.79788032
Si10      68.18246826      60.10456658      48.89471010
O10      68.18246826      57.16604217      47.87425789
H10      68.17679908      55.89614609      49.19706631
O10      70.75438577      61.58800174      47.89126542
Si10      68.18057853      40.81046093      29.95965244
O10      68.16924017      41.39060691      33.01722961
H10      68.17490935      43.17072909      33.45564612
O10      70.75438577      39.20608329      29.18108520
Si10      68.18057853      33.86382700      34.06791744
O10      68.16924017      35.90095197      36.41873698
H10      68.17112990      37.66217689      35.90473142
O10      70.75438577      32.08370482      34.20208801
Si10      68.18246826      29.89540174      41.10525823
O10      68.18246826      32.83392616      42.12571044
H10      68.17679908      34.10382224      40.80290202
O10      70.75438577      28.41196659      42.10870291
Si10      68.18057853      60.04031589      40.81046093
O10      68.16924017      56.98273872      41.39060691
H10      68.17490935      56.54432221      43.17072909
O10      70.75438577      60.81888313      39.20608329
Si10      68.18057853      55.93205089      33.86382700
O10      68.16924017      53.57934163      35.90095197
H10      68.17112990      54.09523691      37.66217689
O10      70.75438577      55.79788032      32.08370482
Si10      68.18246826      48.89471010      29.89540174
O10      68.18246826      47.87425789      32.83392616
H10      68.17679908      49.19706631      34.10382224
O10      70.75438577      47.89126542      28.41196659
Si10      68.17868881      29.95965244      49.18950740
O10      68.16924017      33.01722961      48.60936142
H10      68.17490935      33.45564612      46.82923924
O10      70.75438577      29.18108520      50.79388504
Si10      68.18057853      34.06791744      56.13614133
O10      68.16924017      36.41873698      54.09901636
H10      68.17112990      35.90473142      52.33779144
O10      70.75438577      34.20208801      57.91626351
Si10      68.18246826      41.10525823      60.10456658
O10      70.75438577      42.10870291      61.58800174
O10      68.18246826      42.12571044      57.16604217
H10      68.17679908      40.80290202      55.89614609
Al11      65.40835003      45.09636021      63.63835479
O11      68.18057853      46.55144947      61.69004696
O11      68.17868881      44.42550737      65.76240717
H11      68.17868881      42.92884413      66.78474910
Al11      65.41401921      54.41460066      61.10990098
O11      68.18624771      54.89648087      63.27930679
H11      68.19002716      54.10846499      64.91014060
O11      68.18246826      54.68672124      58.68538212
Al11      65.41023976      61.18926949      54.24074584
O11      68.18246826      62.68971218      55.88102827
H11      68.18246826      62.82199302      57.68949636
O11      68.18246826      60.22739879      52.00330988
Al11      65.40835003      44.90360812      26.36161354
O11      68.18057853      43.44851886      28.30992137
O11      68.17868881      45.57446096      24.23756116
H11      68.17868881      47.07112420      23.21521923
Al11      65.41401921      35.58536767      28.89006735
O11      68.18624771      35.10348746      26.72066154
H11      68.19002716      35.89150334      25.08982773
O11      68.18246826      35.31324708      31.31458621
Al11      65.41023976      28.81069884      35.75922249
O11      68.18246826      27.31025615      34.11894005
H11      68.18246826      27.17797531      32.31047197
O11      68.18246826      29.77256953      37.99665845
Al11      65.40835003      63.63835479      44.90360812
O11      68.17868881      65.76240717      45.57446096
H11      68.17868881      66.78474910      47.07112420
O11      68.18057853      61.69004696      43.44851886
Al11      65.41401921      61.10990098      35.58536767
O11      68.18624771      63.27930679      35.10348746
H11      68.19002716      64.91014060      35.89150334
O11      68.18246826      58.68538212      35.31324708
Al11      65.41023976      54.24074584      28.81069884
O11      68.18246826      55.88102827      27.31025615
H11      68.18246826      57.68949636      27.17797531
O11      68.18246826      52.00330988      29.77256953
Al11      65.40835003      26.36161354      45.09636021
O11      68.17868881      24.23756116      44.42550737
H11      68.17868881      23.21521923      42.92884413
O11      68.18057853      28.30992137      46.55144947
Al11      65.41401921      28.89006735      54.41460066
O11      68.18624771      26.72066154      54.89648087
H11      68.19002716      25.08982773      54.10846499
O11      68.18246826      31.31458621      54.68672124
Al11      65.41023976      35.75922249      61.18926949
O11      68.18246826      34.11894005      62.68971218
H11      68.18246826      32.31047197      62.82199302
O11      68.18246826      37.99665845      60.22739879
Al12      62.79485854      49.92461094      62.99584784
O12      65.61055075      50.79577477      60.82833176
O12      64.73560746      48.14259902      65.42414616
H12      66.03573917      48.86258475      66.45593672
O12      63.47516001      53.32989775      63.92559319
H12      62.17880776      54.29743762      64.74195496
Al12      62.79485854      58.26208343      58.12980259
O12      65.61055075      57.92193269      55.80921867
O12      64.74505610      57.94649913      61.12123934
H12      66.04329807      59.09356301      61.64658326
O12      63.47327029      61.66925997      57.22273396
H12      62.17502831      62.91458961      57.43627303
Al12      62.79107909      63.02230401      49.73374858
O12      65.61055075      61.58422229      47.89315515
O12      64.73560746      64.25062612      52.48896954
H12      66.04140835      65.49784548      52.35479897
O12      63.47138056      65.53375028      47.25442766
H12      62.16746940      66.71293950      46.82168033
Al12      62.79485854      40.07535739      27.00412049
O12      65.61055075      39.20419356      29.17163657
O12      64.73560746      41.85736931      24.57582217
H12      66.03573917      41.13738358      23.54403161
O12      63.47516001      36.67007058      26.07437514
H12      62.17880776      35.70253070      25.25801337
Al12      62.79485854      31.73788490      31.87016574
O12      65.61055075      32.07803564      34.19074965
O12      64.74505610      32.05346919      28.87872899
H12      66.04329807      30.90640532      28.35338507
O12      63.47327029      28.33070836      32.77723437
H12      62.17502831      27.08537872      32.56369530
Al12      62.79107909      26.97766432      40.26621975
O12      65.61055075      28.41574604      42.10681318
O12      64.73560746      25.74934221      37.51099879
H12      66.04140835      24.50212285      37.64516935
O12      63.47138056      24.46621805      42.74554067
H12      62.16746940      23.28702883      43.17828800
Al12      62.79485854      62.99584784      40.07535739
O12      65.61055075      60.82833176      39.20419356
O12      64.73560746      65.42414616      41.85736931
H12      66.03573917      66.45593672      41.13738358
O12      63.47516001      63.92559319      36.67007058
H12      62.17880776      64.74195496      35.70253070
Al12      62.79485854      58.12980259      31.73788490
O12      65.61055075      55.80921867      32.07803564
O12      64.74505610      61.12123934      32.05346919
H12      66.04329807      61.64658326      30.90640532
O12      63.47327029      57.22273396      28.33070836
H12      62.17502831      57.43627303      27.08537872
Al12      62.79107909      49.73374858      26.97766432
O12      65.61055075      47.89315515      28.41574604
O12      64.73560746      52.48707982      25.74934221
H12      66.04140835      52.35479897      24.50212285
O12      63.47138056      47.25442766      24.46621805
H12      62.16746940      46.82168033      23.28702883
Al12      62.79485854      27.00412049      49.92461094
O12      65.61055075      29.17163657      50.79577477
O12      64.73560746      24.57582217      48.14259902
H12      66.03573917      23.54403161      48.86258475
O12      63.47516001      26.07437514      53.32989775
H12      62.17880776      25.25801337      54.29743762
Al12      62.79485854      31.87016574      58.26208343
O12      65.61055075      34.19074965      57.92193269
O12      64.74505610      28.87872899      57.94649913
H12      66.04329807      28.35338507      59.09356301
O12      63.47327029      32.77723437      61.66925997
H12      62.17502831      32.56369530      62.91458961
Al12      62.79107909      40.26621975      63.02230401
O12      65.61055075      42.10681318      61.58422229
O12      64.73560746      37.51288851      64.25062612
H12      66.04140835      37.64516935      65.49784548
O12      63.47138056      42.74554067      65.53375028
H12      62.16746940      43.17828800      66.71293950
Si13      60.03018894      52.94061413      58.46428414
O13      57.45827143      54.69428015      58.79687598
O13      60.02640949      50.82223093      60.73951463
O13      60.03585812      51.57812146      55.65993030
H13      60.02829922      49.74130748      55.70906318
O13      62.59643728      54.69805960      58.80065543
Si13      60.02263004      58.57955745      52.69494971
O13      57.45638171      60.26708305      52.09401674
O13      60.02451977      57.90492516      55.72796044
O13      60.01885059      55.98874267      50.96396041
H13      60.02829922      54.42782874      51.93150028
O13      62.59076810      60.26708305      52.09590647
Si13      60.02263004      45.15305200      60.59967488
O13      57.45449198      46.50609604      61.77508465
O13      62.59265782      46.50420631      61.77319492
O13      60.01885059      45.38170888      57.49296482
H13      60.02074031      43.77166206      56.61046263
O13      60.02263004      42.18051251      61.51619214
Si13      60.02263004      44.84691633      29.40029345
O13      57.45449198      43.49387229      28.22488368
O13      60.01885059      44.61825945      32.50700351
H13      60.02074031      46.22830627      33.38950570
O13      62.59265782      43.49576202      28.22677341
O13      60.02263004      47.81945582      28.48377619
Si13      60.03018894      37.05935420      31.53568418
O13      57.45827143      35.30568818      31.20309235
O13      60.02640949      39.17773739      29.26045370
O13      60.03585812      38.42184687      34.34003803
H13      60.02829922      40.25866085      34.29090515
O13      62.59643728      35.30190873      31.19931290
Si13      60.02263004      31.42041088      37.30501862
O13      57.45638171      29.73288528      37.90595159
O13      60.02451977      32.09504317      34.27200789
O13      60.01885059      34.01122565      39.03600792
H13      60.02829922      35.57213959      38.06846805
O13      62.59076810      29.73288528      37.90406186
Si13      60.02263004      60.59967488      44.84691633
O13      60.02263004      61.51619214      47.81945582
O13      57.45449198      61.77508465      43.49387229
O13      60.01885059      57.49296482      44.61825945
H13      60.02074031      56.61046263      46.22830627
O13      62.59265782      61.77319492      43.49576202
Si13      60.03018894      58.46428414      37.05935420
O13      57.45827143      58.79687598      35.30568818
O13      60.02640949      60.73951463      39.17773739
O13      60.03585812      55.65993030      38.42184687
H13      60.02829922      55.70906318      40.25866085
O13      62.59643728      58.80065543      35.30190873
Si13      60.02263004      52.69494971      31.42041088
O13      57.45638171      52.09401674      29.73288528
O13      60.02451977      55.72796044      32.09504317
O13      60.01885059      50.96396041      34.01122565
H13      60.02829922      51.93150028      35.57213959
O13      62.59076810      52.09590647      29.73288528
Si13      60.02263004      29.40029345      45.15305200
O13      60.02263004      28.48377619      42.18051251
O13      57.45449198      28.22488368      46.50609604
O13      60.01885059      32.50700351      45.38170888
H13      60.02074031      33.38950570      43.77166206
O13      62.59265782      28.22677341      46.50420631
Si13      60.03018894      31.53568418      52.94061413
O13      57.45827143      31.20309235      54.69428015
O13      60.02640949      29.26045370      50.82223093
O13      60.03585812      34.34003803      51.57812146
H13      60.02829922      34.29090515      49.74130748
O13      62.59643728      31.19931290      54.69805960
Si13      60.02263004      37.30501862      58.57955745
O13      57.45638171      37.90595159      60.26708305
O13      60.02451977      34.27200789      57.90492516
O13      60.01885059      39.03600792      55.98874267
H13      60.02829922      38.06846805      54.42782874
O13      62.59076810      37.90406186      60.26708305
Al14      57.25229126      49.92461094      62.99395812
O14      55.30776289      48.14637848      65.42225643
H14      53.99629283      48.87014365      66.43892919
O14      56.57576924      53.33178748      63.91992401
H14      57.87401122      54.30121708      64.72683715
O14      54.43659906      50.79577477      60.82455231
Al14      57.25229126      58.25830398      58.12602313
O14      55.30209371      57.94460941      61.11368044
H14      53.99440310      59.08600410      61.62957572
O14      56.57576924      61.66737025      57.22084423
H14      57.87590095      62.90892043      57.43060385
O14      54.43659906      57.91815324      55.80732895
Al14      57.25418099      63.02230401      49.73185885
O14      55.31154234      64.25062612      52.48707982
H14      54.00196201      65.49406603      52.35479897
O14      56.57576924      65.53186056      47.25253793
H14      57.87590095      66.71482923      46.82168033
O14      54.43659906      61.58233256      47.89315515
Al14      57.25229126      40.07535739      27.00601021
O14      55.30776289      41.85358985      24.57771190
H14      53.99629283      41.12982468      23.56103914
O14      56.57576924      36.66818085      26.08004432
H14      57.87401122      35.69875125      25.27313118
O14      54.43659906      39.20419356      29.17541602
Al14      57.25229126      31.74166435      31.87394519
O14      55.30209371      32.05535892      28.88628789
H14      53.99440310      30.91396423      28.37039261
O14      56.57576924      28.33259808      32.78101382
H14      57.87590095      27.09104790      32.56936448
O14      54.43659906      32.08181509      34.19263938
Al14      57.25418099      26.97766432      40.26810948
O14      55.31154234      25.74934221      37.51288851
H14      54.00196201      24.50590230      37.64516935
O14      56.57576924      24.46810777      42.74743040
H14      57.87590095      23.28513910      43.17828800
O14      54.43659906      28.41763577      42.10681318
Al14      57.25229126      62.99395812      40.07535739
O14      55.30776289      65.42225643      41.85358985
H14      53.99629283      66.43892919      41.12982468
O14      56.57576924      63.91992401      36.66818085
H14      57.87401122      64.72683715      35.69875125
O14      54.43659906      60.82455231      39.20419356
Al14      57.25229126      58.12602313      31.74166435
O14      55.30209371      61.11368044      32.05535892
H14      53.99440310      61.62957572      30.91396423
O14      56.57576924      57.21895450      28.33448781
H14      57.87590095      57.43060385      27.09104790
O14      54.43659906      55.80732895      32.08181509
Al14      57.25418099      49.73185885      26.97766432
O14      55.31154234      52.48707982      25.74934221
H14      54.00196201      52.35479897      24.50590230
O14      56.57576924      47.25253793      24.46810777
H14      57.87590095      46.82168033      23.28513910
O14      54.43659906      47.89315515      28.41763577
Al14      57.25229126      27.00601021      49.92461094
O14      55.30776289      24.57771190      48.14637848
H14      53.99629283      23.56103914      48.87014365
O14      56.57576924      26.08004432      53.33178748
H14      57.87401122      25.27313118      54.30121708
O14      54.43659906      29.17541602      50.79577477
Al14      57.25229126      31.87394519      58.25830398
O14      55.30209371      28.88628789      57.94460941
H14      53.99440310      28.37039261      59.08600410
O14      56.57576924      32.78101382      61.66737025
H14      57.87590095      32.56936448      62.90892043
O14      54.43659906      34.19263938      57.91815324
Al14      57.25418099      40.26810948      63.02230401
O14      56.57576924      42.74743040      65.53186056
H14      57.87590095      43.17828800      66.71482923
O14      54.43659906      42.10681318      61.58233256
O14      55.31154234      37.51288851      64.25062612
H14      54.00196201      37.64516935      65.49406603
Al15      54.63691005      45.09824993      63.63835479
O15      51.86468155      46.55144947      61.69004696
O15      51.86657127      44.42739709      65.76429689
H15      51.86657127      42.93262358      66.78852855
Al15      54.63313059      54.41271093      61.10423180
O15      51.86279182      54.89648087      63.27552734
H15      51.85901237      54.11224445      64.90825087
O15      51.86468155      54.68483152      58.68160267
Al15      54.63691005      61.18926949      54.23885611
O15      51.86657127      62.69349163      55.88102827
H15      51.86657127      62.82388275      57.68949636
O15      51.86657127      60.22550907      52.00330988
Al15      54.63691005      44.90171840      26.36161354
O15      51.86468155      43.44851886      28.30992137
O15      51.86657127      45.57257124      24.23567144
H15      51.86657127      47.06734475      23.21332950
Al15      54.63313059      35.58725740      28.89573653
O15      51.86279182      35.10348746      26.72444099
H15      51.85901237      35.88772388      25.09171746
O15      51.86468155      35.31513681      31.31836566
Al15      54.63691005      28.81069884      35.76111222
O15      51.86657127      27.30836642      34.11894005
H15      51.86657127      27.17608558      32.31047197
O15      51.86657127      29.77445926      37.99665845
Al15      54.63691005      63.63835479      44.90171840
O15      51.86657127      65.76429689      45.57257124
H15      51.86657127      66.78852855      47.06734475
O15      51.86468155      61.69004696      43.44851886
Al15      54.63313059      61.10423180      35.58725740
O15      51.86279182      63.27552734      35.10348746
H15      51.85901237      64.90825087      35.88772388
O15      51.86468155      58.68160267      35.31513681
Al15      54.63691005      54.23885611      28.81069884
O15      51.86657127      55.88102827      27.30836642
H15      51.86657127      57.68949636      27.17608558
O15      51.86657127      52.00330988      29.77445926
Al15      54.63691005      26.36161354      45.09824993
O15      51.86657127      24.23567144      44.42739709
H15      51.86657127      23.21332950      42.93262358
O15      51.86468155      28.30992137      46.55144947
Al15      54.63313059      28.89573653      54.41271093
O15      51.86279182      26.72444099      54.89648087
H15      51.85901237      25.09171746      54.11224445
O15      51.86468155      31.31836566      54.68483152
Al15      54.63691005      35.76111222      61.18926949
O15      51.86657127      34.11894005      62.69349163
H15      51.86657127      32.31047197      62.82388275
O15      51.86657127      37.99665845      60.22550907
Si16      51.86468155      49.18950740      60.03842616
O16      51.86657127      48.60936142      56.98273872
H16      51.86846100      46.82923924      56.54243248
O16      49.29276403      50.79577477      60.82077286
Si16      51.86468155      56.13614133      55.93016116
O16      51.86279182      54.10090609      53.57745190
H16      51.87035073      52.33968116      54.09145746
O16      49.29276403      57.92004297      55.79977004
Si16      51.86279182      60.10456658      48.89471010
O16      51.85523291      57.16604217      47.87425789
H16      51.86657127      55.89614609      49.19706631
O16      49.29465376      61.58989147      47.89126542
Si16      51.86468155      40.81046093      29.96154217
O16      51.86657127      41.39060691      33.01722961
H16      51.86846100      43.17072909      33.45753584
O16      49.29276403      39.20419356      29.17919547
Si16      51.86468155      33.86382700      34.06980717
O16      51.86279182      35.89906224      36.42251643
H16      51.87035073      37.66028716      35.90851087
O16      49.29276403      32.07992536      34.20019829
Si16      51.86279182      29.89540174      41.10525823
O16      51.85523291      32.83203644      42.12571044
H16      51.86657127      34.10382224      40.80290202
O16      49.29465376      28.41007686      42.10870291
Si16      51.86468155      60.03842616      40.81046093
O16      51.86657127      56.98273872      41.39060691
H16      51.86846100      56.54243248      43.17072909
O16      49.29276403      60.82077286      39.20419356
Si16      51.86468155      55.93016116      33.86382700
O16      51.86279182      53.57745190      35.89906224
H16      51.87035073      54.09145746      37.66028716
O16      49.29276403      55.79977004      32.07992536
Si16      51.86279182      48.89471010      29.89540174
O16      51.85523291      47.87425789      32.83203644
H16      51.86657127      49.19706631      34.10382224
O16      49.29465376      47.89126542      28.41007686
Si16      51.86468155      29.96154217      49.18950740
O16      51.86657127      33.01722961      48.60936142
H16      51.86846100      33.45753584      46.82923924
O16      49.29276403      29.17919547      50.79577477
Si16      51.86468155      34.06980717      56.13614133
O16      51.86279182      36.42251643      54.10090609
H16      51.87035073      35.90851087      52.33968116
O16      49.29276403      34.20019829      57.92004297
Si16      51.86279182      41.10525823      60.10456658
O16      51.85523291      42.12571044      57.16604217
H16      51.86657127      40.80290202      55.89614609
O16      49.29465376      42.10870291      61.58989147
Al17      49.09623250      45.10202939      63.64402397
O17      48.42537966      48.15204765      65.42225643
H17      49.71039355      48.87014365      66.47672371
O17      46.27487111      46.50609604      61.79020246
O17      47.16115275      42.75687903      65.54886809
H17      45.83078543      43.21419280      66.69026279
Al17      49.09434277      54.41460066      61.09667290
O17      47.15548357      53.34123611      63.91614456
H17      45.83078543      54.30877598      64.68148371
O17      48.42537966      57.94649913      61.10423180
H17      49.71417300      59.09923219      61.64469353
O17      46.27298139      54.69805960      58.79120680
Al17      49.09812222      61.19304894      54.23318693
O17      47.15737330      61.67492915      57.20950587
H17      45.83078543      62.89569235      57.38525042
O17      48.42726938      64.25818502      52.48330036
H17      49.72173191      65.52052220      52.37180651
O17      46.27487111      60.28787004      52.08834756
Al17      49.09623250      44.89793894      26.35594436
O17      48.42537966      41.84792067      24.57771190
H17      49.71039355      41.12982468      23.52324462
O17      46.27487111      43.49387229      28.20976587
O17      47.16115275      47.24308930      24.45110024
H17      45.83078543      46.78577553      23.30970554
Al17      49.09434277      35.58347795      28.90329543
O17      47.15548357      36.65873222      26.08382377
H17      45.83078543      35.69119235      25.31848461
O17      48.42537966      32.05346919      28.89573653
H17      49.71417300      30.90073614      28.35527480
O17      46.27298139      35.30190873      31.20876153
Al17      49.09812222      28.80691939      35.76678140
O17      47.15737330      28.32503918      32.79046246
H17      45.83078543      27.10427598      32.61471791
O17      48.42726938      25.74178331      37.51666797
H17      49.72173191      24.47944613      37.62816182
O17      46.27487111      29.71209829      37.91162076
Al17      49.09623250      63.64402397      44.89793894
O17      47.16115275      65.54886809      47.24308930
H17      45.83078543      66.69026279      46.78577553
O17      48.42537966      65.42225643      41.84792067
H17      49.71039355      66.47672371      41.12982468
O17      46.27487111      61.79020246      43.49387229
Al17      49.09434277      61.09667290      35.58536767
O17      47.15548357      63.91614456      36.65873222
H17      45.83078543      64.68148371      35.69119235
O17      48.42537966      61.10423180      32.05346919
H17      49.71417300      61.64469353      30.90073614
O17      46.27298139      58.79120680      35.30190873
Al17      49.09812222      54.23318693      28.80691939
O17      47.15737330      57.20950587      28.32503918
H17      45.83078543      57.38525042      27.10427598
O17      48.42726938      52.48330036      25.74178331
H17      49.72173191      52.37180651      24.47944613
O17      46.27487111      52.08834756      29.71209829
Al17      49.09623250      26.35594436      45.10202939
O17      47.16115275      24.45110024      42.75687903
H17      45.83078543      23.30970554      43.21419280
O17      48.42537966      24.57771190      48.15204765
H17      49.71039355      23.52324462      48.87014365
O17      46.27487111      28.20976587      46.50609604
Al17      49.09434277      28.90329543      54.41460066
O17      47.15548357      26.08382377      53.34123611
H17      45.83078543      25.31848461      54.30877598
O17      48.42537966      28.89573653      57.94649913
H17      49.71417300      28.35527480      59.09923219
O17      46.27298139      31.20876153      54.69805960
Al17      49.09812222      35.76678140      61.19304894
O17      47.15737330      32.79046246      61.67492915
H17      45.83078543      32.61471791      62.89569235
O17      48.42726938      37.51666797      64.25818502
H17      49.72173191      37.62816182      65.52052220
O17      46.27487111      37.91162076      60.28787004
Al18      46.47896155      49.93216984      62.99773757
O18      43.71807141      49.85091161      65.24840161
H18      43.72752004      48.68117102      66.63357100
O18      43.70106387      50.83545902      60.76408107
Al18      46.47518210      58.26586288      58.11657450
O18      43.71429196      59.31655071      60.11023576
H18      43.72374059      58.99718696      61.89413740
O18      43.69917415      57.93516078      55.72985017
Al18      46.48085128      63.04687045      49.72996913
O18      43.71807141      64.95171458      50.93183506
H18      43.72752004      65.56776536      52.63447847
O18      43.70295360      61.57099420      47.82323528
Al18      46.47896155      40.06779849      27.00223076
O18      43.71807141      40.14905672      24.75156672
H18      43.72752004      41.31879731      23.36639733
O18      43.70106387      39.16450931      29.23588726
Al18      46.47518210      31.73410545      31.88339383
O18      43.71429196      30.68341762      29.88973257
H18      43.72374059      31.00278136      28.10583093
O18      43.69917415      32.06480755      34.27011816
Al18      46.48085128      26.95309788      40.26999920
O18      43.71807141      25.04825375      39.06813327
H18      43.72752004      24.43220297      37.36548986
O18      43.70295360      28.42897413      42.17673305
Al18      46.47896155      62.99773757      40.06779849
O18      43.71807141      65.24840161      40.14905672
H18      43.72752004      66.63357100      41.31879731
O18      43.70106387      60.76408107      39.16450931
Al18      46.47518210      58.11657450      31.73410545
O18      43.71429196      60.11023576      30.68341762
H18      43.72374059      61.89413740      31.00278136
O18      43.69917415      55.72985017      32.06480755
Al18      46.48085128      49.72996913      26.95309788
O18      43.71807141      50.93183506      25.04825375
H18      43.72752004      52.63447847      24.43220297
O18      43.70295360      47.82323528      28.42897413
Al18      46.47896155      27.00223076      49.93216984
O18      43.71807141      24.75156672      49.85091161
H18      43.72752004      23.36639733      48.68117102
O18      43.70106387      29.23588726      50.83545902
Al18      46.47518210      31.88339383      58.26586288
O18      43.71429196      29.88973257      59.31655071
H18      43.72374059      28.10583093      58.99718696
O18      43.69917415      34.27011816      57.93516078
Al18      46.48085128      40.26999920      63.04687045
O18      43.71807141      39.06813327      64.95171458
H18      43.72752004      37.36548986      65.56776536
O18      43.70295360      42.17673305      61.57099420
Si19      43.69161524      52.95006276      58.48129168
O19      41.14237445      54.72829522      58.84222941
O19      43.65760017      51.59701872      55.67882756
H19      43.68405634      49.76398420      55.71662209
Si19      43.69161524      58.62302115      52.69305998
O19      41.14426417      60.34456183      52.11858319
O19      43.65193099      56.04543446      50.94884260
H19      43.68216661      54.47696162      51.89937493
Si19      43.69350497      45.14927254      60.65069749
O19      41.14426417      46.50609604      61.86012233
O19      43.65382072      45.37981915      57.54776688
H19      43.68027688      43.77544151      56.65959552
Si19      43.69350497      44.85069579      29.34927084
O19      41.14426417      43.49387229      28.13984600
O19      43.65382072      44.62014918      32.45220145
H19      43.68027688      46.22452682      33.34037281
Si19      43.69161524      37.04990557      31.51867665
O19      41.14237445      35.27167311      31.15773892
O19      43.65760017      38.40294961      34.32114077
H19      43.68405634      40.23598413      34.28334624
Si19      43.69161524      31.37694717      37.30690834
O19      41.14426417      29.65540650      37.88138514
O19      43.65193099      33.95453387      39.05112573
H19      43.68216661      35.52300670      38.10059340
Si19      43.69350497      60.65069749      44.85069579
O19      41.14426417      61.86012233      43.49387229
O19      43.65382072      57.54776688      44.62203890
H19      43.68027688      56.65959552      46.22452682
Si19      43.69161524      58.48129168      37.04990557
O19      41.14237445      58.84222941      35.27167311
O19      43.65760017      55.67882756      38.40294961
H19      43.68405634      55.71473236      40.23598413
Si19      43.69161524      52.69305998      31.37694717
O19      41.14426417      52.11858319      29.65540650
O19      43.65193099      50.94884260      33.95453387
H19      43.68216661      51.89937493      35.52300670
Si19      43.69350497      29.34927084      45.14927254
O19      41.14426417      28.13984600      46.50609604
O19      43.65382072      32.45220145      45.37981915
H19      43.68027688      33.34037281      43.77544151
Si19      43.69161524      31.52056637      52.95006276
O19      41.14237445      31.15773892      54.72829522
O19      43.65760017      34.32114077      51.59701872
H19      43.68405634      34.28334624      49.76209447
Si19      43.69161524      37.30690834      58.62302115
O19      41.14426417      37.88138514      60.34456183
O19      43.65193099      39.05112573      56.04543446
H19      43.68216661      38.10059340      54.47696162
Al20      40.94017373      49.93783902      63.04498073
O20      38.98808645      48.16338601      65.50162493
H20      37.61425542      48.91171763      66.41625247
O20      40.23530582      53.35824365      63.97661580
H20      41.48819436      54.35034996      64.82888237
O20      38.17550413      50.81467203      60.86045711
Al20      40.94017373      58.29420878      58.15436903
O20      38.98430700      57.98807311      61.16659277
H20      37.61425542      59.09734246      61.58422229
O20      40.23341609      61.72217231      57.25674903
H20      41.48819436      63.00529647      57.49863400
O20      38.17550413      57.96161695      55.82055703
Al20      40.94206346      63.09411361      49.74697666
O20      38.98997618      64.33188435      52.51164626
H20      37.61425542      65.49595576      52.31889417
O20      40.23719554      65.61689824      47.25631738
H20      41.49386354      66.84899980      46.82734951
O20      38.17739386      61.64658326      47.89315515
Al20      40.94017373      40.06212931      26.95498760
O20      38.98808645      41.83658232      24.49834339
H20      37.61425542      41.08825070      23.58371586
O20      40.23530582      36.64172468      26.02335253
H20      41.48819436      35.64961837      25.17108596
O20      38.17550413      39.18529630      29.13951122
Al20      40.93828400      31.70575955      31.84559930
O20      38.98430700      32.01189522      28.83337556
H20      37.61425542      30.90262587      28.41574604
O20      40.23341609      28.27779602      32.74321930
H20      41.48819436      26.99467185      32.50133433
O20      38.17550413      32.03835138      34.17941130
Al20      40.94206346      26.90585472      40.25299167
O20      38.98997618      25.66808398      37.48832207
H20      37.61425542      24.50401257      37.68107415
O20      40.23719554      24.38307009      42.74365095
H20      41.49386354      23.15096853      43.17261882
O20      38.17739386      28.35338507      42.10681318
Al20      40.94017373      63.04498073      40.06212931
O20      38.98808645      65.50162493      41.83658232
H20      37.61425542      66.41625247      41.08825070
O20      40.23530582      63.97661580      36.64172468
H20      41.48819436      64.82888237      35.64961837
O20      38.17550413      60.86045711      39.18529630
Al20      40.93828400      58.15436903      31.70575955
O20      38.98430700      61.16659277      32.01189522
H20      37.61425542      61.58422229      30.90262587
O20      40.23341609      57.25674903      28.27779602
H20      41.48819436      57.49863400      26.99467185
O20      38.17550413      55.82055703      32.03835138
Al20      40.94206346      49.74697666      26.90585472
O20      38.98997618      52.51164626      25.66808398
H20      37.61425542      52.31889417      24.50401257
O20      40.23719554      47.25631738      24.38307009
H20      41.49386354      46.82734951      23.15096853
O20      38.17739386      47.89315515      28.35338507
Al20      40.94017373      26.95498760      49.93783902
O20      38.98808645      24.49834339      48.16338601
H20      37.61425542      23.58371586      48.91171763
O20      40.23530582      26.02335253      53.35824365
H20      41.48819436      25.17108596      54.35034996
O20      38.17550413      29.13951122      50.81467203
Al20      40.93828400      31.84559930      58.29420878
O20      38.98430700      28.83337556      57.98807311
H20      37.61425542      28.41574604      59.09734246
O20      40.23341609      32.74321930      61.72217231
H20      41.48819436      32.50133433      63.00529647
O20      38.17550413      34.17941130      57.96161695
Al20      40.94206346      40.25299167      63.09411361
O20      40.23719554      42.74365095      65.61689824
H20      41.49197381      43.17261882      66.84899980
O20      38.17739386      42.10681318      61.64658326
O20      38.98997618      37.48832207      64.33188435
H20      37.61425542      37.68107415      65.49595576
Al21      38.31723361      45.10202939      63.72717193
O21      35.61492498      46.55711865      61.78831273
O21      35.46752633      44.43495600      65.83232704
H21      35.48075441      42.97797701      66.90947104
Al21      38.31534388      54.45050546      61.16092359
O21      35.46752633      54.93049594      63.31710132
H21      35.47697496      54.20673076      64.98006047
O21      35.61492498      54.73774386      58.74963282
Al21      38.31723361      61.26485854      54.27098146
O21      35.46941605      62.76152178      55.90181526
H21      35.47697496      62.96372250      57.70272444
O21      35.61492498      60.31999538      52.03921468
Al21      38.31723361      44.89793894      26.27279640
O21      35.61492498      43.44284968      28.21165560
O21      35.46752633      45.56501233      24.16764129
H21      35.48075441      47.02199132      23.09049729
Al21      38.31534388      35.54946287      28.83904474
O21      35.46752633      35.06947239      26.68286701
H21      35.47697496      35.79323757      25.01990786
O21      35.61492498      35.26222447      31.25033551
Al21      38.31723361      28.73510979      35.72898687
O21      35.46941605      27.23844655      34.09815306
H21      35.47697496      27.03624583      32.29724389
O21      35.61492498      29.67997295      37.96075365
Al21      38.31723361      63.72717193      44.89793894
O21      35.46752633      65.83232704      45.56501233
H21      35.48075441      66.90947104      47.02199132
O21      35.61492498      61.78831273      43.44284968
Al21      38.31534388      61.16092359      35.54946287
O21      35.46752633      63.31710132      35.06947239
H21      35.47697496      64.98006047      35.79323757
O21      35.61492498      58.74963282      35.26222447
Al21      38.31723361      54.27098146      28.73510979
O21      35.46941605      55.90181526      27.23844655
H21      35.47697496      57.70272444      27.03624583
O21      35.61492498      52.03921468      29.67997295
Al21      38.31723361      26.27279640      45.10202939
O21      35.46752633      24.16764129      44.43495600
H21      35.48075441      23.09049729      42.97797701
O21      35.61492498      28.21165560      46.55711865
Al21      38.31534388      28.83904474      54.45050546
O21      35.46752633      26.68286701      54.93049594
H21      35.47697496      25.01990786      54.20673076
O21      35.61492498      31.25033551      54.73774386
Al21      38.31723361      35.72898687      61.26485854
O21      35.46941605      34.09815306      62.76152178
H21      35.47697496      32.29724389      62.96372250
O21      35.61492498      37.96075365      60.31999538
Si22      35.63949142      49.15738205      60.08566932
O22      35.64516060      48.53944155      57.03565105
H22      35.70941130      46.75742964      56.61046263
O22      33.09969926      50.78254668      60.84911875
Si22      35.63949142      56.14181051      55.97740432
O22      35.63760170      54.08200882      53.63981287
H22      35.70563184      52.32645308      54.16326706
O22      33.10158898      57.93327105      55.83000566
Si22      35.64138115      60.14614056      48.93628407
O22      35.64705033      57.19627779      47.94228803
H22      35.71130102      55.93394061      49.27265536
O22      33.10158898      61.62012709      47.91583187
Si22      35.63949142      40.84258628      29.91429901
O22      35.64516060      41.46052678      32.96431728
H22      35.70941130      43.24253869      33.38950570
O22      33.09969926      39.21742165      29.15084958
Si22      35.63949142      33.85815782      34.02256401
O22      35.63760170      35.91795950      36.36015546
H22      35.70563184      37.67351525      35.83670127
O22      33.10158898      32.06669728      34.16996266
Si22      35.64138115      29.85382777      41.06368425
O22      35.64705033      32.80369054      42.05768030
H22      35.71130102      34.06602772      40.72731297
O22      33.10158898      28.37984124      42.08413646
Si22      35.63949142      60.08566932      40.84258628
O22      35.64516060      57.03565105      41.46052678
H22      35.70941130      56.61046263      43.24253869
O22      33.09969926      60.84911875      39.21742165
Si22      35.63949142      55.97740432      33.85815782
O22      35.63760170      53.63981287      35.91795950
H22      35.70563184      54.16326706      37.67351525
O22      33.10158898      55.83000566      32.06669728
Si22      35.64138115      48.93628407      29.85382777
O22      35.64705033      47.94228803      32.80369054
H22      35.71130102      49.27265536      34.06602772
O22      33.10158898      47.91583187      28.37984124
Si22      35.63949142      29.91429901      49.15738205
O22      35.64516060      32.96431728      48.53944155
H22      35.70941130      33.38950570      46.75742964
O22      33.09969926      29.15084958      50.78254668
Si22      35.63949142      34.02256401      56.14181051
O22      35.63760170      36.36015546      54.08200882
H22      35.70563184      35.83670127      52.32645308
O22      33.10158898      34.16996266      57.93327105
Si22      35.64138115      41.06368425      60.14614056
O22      35.64705033      42.05768030      57.19627779
H22      35.71130102      40.72731297      55.93394061
O22      33.10158898      42.08413646      61.62012709
Al23      32.75954852      45.07935267      63.64780342
O23      31.83547235      48.16149629      65.41091807
H23      33.01277185      49.02699094      66.48995180
O23      30.21219745      46.44751452      61.60878873
H23      28.91395547      45.13226501      61.21194620
O23      30.57691463      42.68884888      65.37123382
H23      29.07836166      43.20474417      66.25562573
Al23      32.76143825      54.39192394      61.10423180
O23      30.58825299      53.18816828      63.80465071
H23      29.09158975      54.07822937      64.32054599
O23      31.83547235      57.94460941      61.08722427
H23      33.01655130      59.23340275      61.58422229
O23      30.21597690      54.55255068      58.65136705
H23      28.91395547      53.22407308      58.97828970
Al23      32.76332797      61.18926949      54.25208419
O23      30.58825299      61.49162570      57.18493943
H23      29.09725893      62.52530599      57.18493943
O23      31.82980318      64.25629529      52.46629283
H23      33.01088212      65.61878797      52.24141540
O23      30.21597690      60.09889741      52.04866331
H23      28.91773493      59.10490136      52.99541619
Al23      32.75954852      44.92061566      26.35216491
O23      31.83358263      41.83847204      24.58905026
H23      33.01277185      40.97297739      23.51001653
O23      30.21219745      43.55245381      28.39117960
H23      28.91395547      44.86770332      28.78802213
O23      30.57691463      47.31111945      24.62873451
H23      29.07836166      46.79522416      23.74434260
Al23      32.76143825      35.60804439      28.89573653
O23      30.58825299      36.81180005      26.19531762
H23      29.09158975      35.92173896      25.67942234
O23      31.83547235      32.05535892      28.91274406
H23      33.01655130      30.76656557      28.41574604
O23      30.21597690      35.44741765      31.34860128
H23      28.91395547      36.77589525      31.02167863
Al23      32.76332797      28.81069884      35.74788414
O23      30.58825299      28.50834263      32.81502890
H23      29.09536920      27.47466234      32.81502890
O23      31.82980318      25.74367304      37.53367550
H23      33.01088212      24.38118036      37.75855293
O23      30.21597690      29.90107092      37.95130502
H23      28.91773493      30.89506696      37.00455213
Al23      32.75954852      63.64780342      44.92061566
O23      30.57691463      65.37123382      47.31111945
H23      29.07836166      66.25562573      46.79522416
O23      31.83358263      65.41091807      41.83847204
H23      33.01277185      66.48995180      40.97297739
O23      30.21219745      61.60878873      43.55245381
H23      28.91395547      61.21194620      44.86770332
Al23      32.76143825      61.10423180      35.60804439
O23      30.58636326      63.80465071      36.81180005
H23      29.09158975      64.32054599      35.92173896
O23      31.83547235      61.08722427      32.05535892
H23      33.01655130      61.58422229      30.76656557
O23      30.21597690      58.65136705      35.44741765
H23      28.91395547      58.97828970      36.77589525
Al23      32.76332797      54.25208419      28.81069884
O23      30.58825299      57.18493943      28.50834263
H23      29.09536920      57.18493943      27.47466234
O23      31.82980318      52.46629283      25.74367304
H23      33.01088212      52.24141540      24.38118036
O23      30.21597690      52.04866331      29.90107092
H23      28.91773493      52.99541619      30.89506696
Al23      32.75954852      26.35216491      45.07935267
O23      30.57691463      24.62873451      42.68884888
H23      29.07836166      23.74434260      43.20474417
O23      31.83358263      24.58905026      48.16149629
H23      33.01277185      23.51001653      49.02699094
O23      30.21219745      28.39117960      46.44751452
H23      28.91395547      28.78802213      45.13226501
Al23      32.76143825      28.89573653      54.39192394
O23      30.58636326      26.19531762      53.18816828
H23      29.09158975      25.67942234      54.07822937
O23      31.83547235      28.91274406      57.94460941
H23      33.01655130      28.41574604      59.23340275
O23      30.21597690      31.34860128      54.55255068
H23      28.91395547      31.02167863      53.22407308
Al23      32.76332797      35.74788414      61.18926949
O23      30.58825299      32.81502890      61.49162570
H23      29.09536920      32.81502890      62.52530599
O23      31.82980318      37.53367550      64.25629529
H23      33.01088212      37.75855293      65.61878797
O23      30.21597690      37.95130502      60.09889741
H23      28.91773493      37.00455213      59.10490136
Al24      30.10448305      49.81689654      62.59333614
O24      27.09225931      49.52398896      65.37690300
H24      27.75933270      48.18606273      66.44270864
H24      25.62016251      48.88148201      64.50006999
O24      27.55146280      50.77498778      60.42959951
H24      27.24154769      49.49186361      59.16915206
Al24      30.10637278      57.96539640      57.82366692
O24      27.09792849      59.11435000      60.38046663
H24      27.76878133      58.50018894      61.97350591
H24      25.61638306      58.12413341      59.95716793
O24      27.55335253      57.70839362      55.47284739
H24      27.25288605      55.96606596      55.02309253
Al24      30.10070360      62.63302039      49.62414445
O24      27.09036958      64.89691251      51.26442689
H24      27.76878133      65.17281256      52.94628331
H24      25.61260361      63.82543769      51.40615637
O24      27.55524226      61.22706401      47.71174142
H24      27.24910659      59.49607471      48.19929081
Al24      30.10448305      40.18307179      27.40663219
O24      27.09225931      40.47597937      24.62306533
H24      27.75933270      41.81390560      23.55725969
H24      25.62016251      41.11848632      25.49989834
O24      27.55146280      39.22498055      29.57036882
H24      27.24154769      40.50810472      30.83081627
Al24      30.10637278      32.03457193      32.17630140
O24      27.09792849      30.88561833      29.61950170
H24      27.76878133      31.49977938      28.02646242
H24      25.61638306      31.87583492      30.04091067
O24      27.55335253      32.29157471      34.52712094
H24      27.25288605      34.03390237      34.97687580
Al24      30.10070360      27.36694794      40.37582388
O24      27.09036958      25.10305581      38.73554144
H24      27.76878133      24.82715577      37.05368502
H24      25.61260361      26.17453063      38.59381196
O24      27.55524226      28.77290431      42.28822691
H24      27.24910659      30.50389362      41.80067752
Al24      30.10448305      62.59333614      40.18307179
O24      27.09225931      65.37690300      40.47597937
H24      27.75933270      66.44270864      41.81390560
H24      25.62016251      64.50006999      41.11848632
O24      27.55146280      60.42959951      39.22498055
H24      27.24154769      59.16915206      40.50810472
Al24      30.10637278      57.82366692      32.03457193
O24      27.09792849      60.38046663      30.88561833
H24      27.76878133      61.97350591      31.49977938
H24      25.61638306      59.95716793      31.87583492
O24      27.55335253      55.47284739      32.29157471
H24      27.25288605      55.02309253      34.03390237
Al24      30.10070360      49.62414445      27.36694794
O24      27.09036958      51.26442689      25.10305581
H24      27.76878133      52.94628331      24.82715577
H24      25.61260361      51.40615637      26.17453063
O24      27.55524226      47.71174142      28.77290431
H24      27.24910659      48.19929081      30.50389362
Al24      30.10448305      27.40663219      49.81689654
O24      27.09225931      24.62306533      49.52398896
H24      27.75933270      23.55725969      48.18606273
H24      25.62016251      25.49989834      48.88148201
O24      27.55146280      29.57036882      50.77498778
H24      27.24154769      30.83081627      49.49186361
Al24      30.10637278      32.17630140      57.96539640
O24      27.09792849      29.61950170      59.11435000
H24      27.76878133      28.02646242      58.50018894
H24      25.61638306      30.04280040      58.12413341
O24      27.55335253      34.52712094      57.70839362
H24      27.25288605      34.97687580      55.96606596
Al24      30.10070360      40.37582388      62.63302039
O24      27.09036958      38.73554144      64.89691251
H24      27.76878133      37.05368502      65.17281256
H24      25.61260361      38.59381196      63.82543769
O24      27.55524226      42.28822691      61.22706401
H24      27.24910659      41.80067752      59.49607471
OOH      60.02263004      49.82634517      65.22383517
HOH      60.02074031      48.64904567      66.60144565
OOH      60.02263004      59.29198427      60.10834604
HOH      60.02074031      58.96695134      61.89035795
OOH      60.02263004      64.90258169      50.93183506
HOH      60.02074031      65.50351466      52.64203737
OOH      60.02263004      40.17362316      24.77613316
HOH      60.02074031      41.35092265      23.39852268
OOH      60.02263004      30.70798406      29.89162229
HOH      60.02074031      31.03301699      28.10961038
OOH      60.02263004      25.09738664      39.06813327
HOH      60.02074031      24.49645367      37.35793095
OOH      60.02263004      65.22383517      40.17362316
HOH      60.02074031      66.60144565      41.35092265
OOH      60.02263004      60.10834604      30.70798406
HOH      60.02074031      61.89035795      31.03301699
OOH      60.02263004      50.93183506      25.09738664
HOH      60.02074031      52.64203737      24.49645367
OOH      60.02263004      24.77613316      49.82634517
HOH      60.02074031      23.39852268      48.64904567
OOH      60.02263004      29.89162229      59.29198427
HOH      60.02074031      28.10961038      58.96695134
OOH      60.02263004      39.06813327      64.90258169
HOH      60.02074031      37.35793095      65.50351466
 %endblock positions_abs"""


def test_onetep_input():
    """Read onetep input file."""
    # For tests we use a StringIO object.
    # in-memory text-file like object.
    fd = StringIO(onetep_input)
    atoms = read(fd, format='onetep-in')
    fd.seek(0)
    original_cell = atoms.get_cell()
    original_tags = atoms.get_tags()
    original_species = atoms.info['onetep_species']
    original_keywords = get_onetep_keywords(fd)
    fd.seek(0)
    original_keywords = dict(sorted(original_keywords.items()))
    # We test for the correct number of tags and species
    assert len(set(original_species)) == 71
    assert len(set(original_tags)) == 24
    assert len(atoms) == 1_416
    assert original_cell.cellpar() == approx(
        [120.0, 90.0, 90.0, 90.0, 90.0, 90.0])
    fdbis = StringIO()
    # We pass the pseudopotentials manually as well,
    #  To test if it is working correctly.
    write(fdbis, atoms, format='onetep-in',
          keywords=original_keywords,
          pseudopotentials=original_keywords['species_pot'])
    fdbis.seek(0)
    cycled_atoms = read(fdbis, format='onetep-in')
    fdbis.seek(0)
    cycled_keywords = get_onetep_keywords(fdbis)
    # The dictionary should be similar, cycled dictionary
    # Might have additional keywords (the defaults), which
    # is normal
    for key in original_keywords:
        keyl = key.lower()
        assert keyl in cycled_keywords
        assert cycled_keywords[keyl] == original_keywords[key]
    # Is everything well written ?
    assert cycled_atoms.positions == approx(atoms.positions)
    assert cycled_atoms.info['onetep_species'] == original_species
    assert np.all(cycled_atoms.get_tags() == original_tags)
    assert cycled_atoms.get_cell().cellpar() == approx(original_cell.cellpar())


@pytest.fixture()
def test_onetep_recursive_include_input(datadir):
    testfile_path = datadir / "onetep_include.dat"
    with pytest.raises(ValueError):
        read(testfile_path, format='onetep-in')


@pytest.fixture()
def test_onetep_nested_include_input(datadir):
    testfile_path = datadir / "onetep_include_nested.dat"
    with pytest.raises(ValueError):
        read(testfile_path, format='onetep-in')


test_output = """
|Linear-Scaling Ab Initio Total Energy Program|
--------------------------------------------------------------------------------
---------------------------------- INPUT FILE ----------------------------------
--------------------------------------------------------------------------------

cutoff_energy : 25
edft : F
forces_output_detail : verbose
output_detail : verbose
paw : F
do_properties = T
read_denskern : F
read_hamiltonian : F
read_tightbox_ngwfs : F
task : singlepoint
write_forces : T
xc_functional : pbe
zero_total_force: : T
%BLOCK LATTICE_CART
ang
14.00000000       0.00000000       0.00000000
0.00000000      15.52647800       0.00000000
0.00000000       0.00000000      14.59630900
%ENDBLOCK LATTICE_CART
%BLOCK POSITIONS_ABS
ang
O     7.000000     7.763239     7.596309
HKO     7.000000     8.526478     7.000000
H1     7.000000     7.000000     7.000000
%ENDBLOCK POSITIONS_ABS
%BLOCK SPECIES
H1 H 1 -1 9.0
HKO H 1 -1 9.0
O O 8 -1 9.0
%ENDBLOCK SPECIES
%BLOCK SPECIES_POT
H1 H_NCP19_PBE.usp
HKO H_NCP19_PBE.usp
O O_NCP19_PBE.usp
%ENDBLOCK SPECIES_POT

...

           ........................................................
           |         *** NGWF optimisation converged ***          |
           | RMS NGWF gradient =    0.00000049392392              |
           | Criteria satisfied:                                  |
           | -> RMS NGWF gradient lower than set threshold.       |
           ========================================================

================================================================================
           ---------------- ENERGY COMPONENTS (Eh) ----------------
           | Kinetic                    :       12.15138690746626 |
           | Pseudopotential (local)    :      -45.29591161192612 |
           | Pseudo (non-coul chg cor)  :        0.00000000000000
           | Pseudopotential (non-local):       -1.33142854358881 |
           | Hartree                    :       17.94293053457817 |
           | Exchange-correlation       :       -4.25961784082707 |
           | Ewald                      :        3.64621627669446 |
           | Total                      :      -17.14642427760311 |
           --------------------------------------------------------
           ------ LOCAL ENERGY COMPONENTS FROM MATRIX TRACES ------
           | Pseudopotential (local)    :      -45.29591161192528 |
           | Hartree                    :       17.94293053457780 |
           --------------------------------------------------------
           Integrated density           :        7.99999999999942
           Integrated density tr(KS)    :        8.00000000000000
================================================================================

...

    Mulliken Atomic Populations
    ---------------------------
Species  Ion    Total   Charge (e)    Spin (hbar)
=================================================
  O      1       7.011     -1.011      0.00
  H      2       0.494      0.506      0.00
  H      3       0.494      0.506     -0.00
=================================================

...

************************* Forces *************************
*                                                        *
* Element  Atom         Cartesian components (Eh/a)      *
* ------------------------------------------------------ *
*                       x            y            z      *
*                                                        *
*    O       1      0.00000017   0.00000001   0.00168584 *
*    H       2      0.00000005   0.00468666  -0.00084292 *
*    H       3     -0.00000021  -0.00468667  -0.00084292 *
*                                                        *
**********************************************************
* TOTAL:            0.00000000   0.00000000  -0.00000000 *
**********************************************************
"""


def test_onetep_output():
    fd = StringIO(test_output)
    atoms = read(fd, format='onetep-out')
    rpositions = np.loadtxt(test_output.splitlines()[27:30],
                            usecols=(1, 2, 3))
    rcell = np.loadtxt(test_output.splitlines()[21:24])
    assert len(atoms) == 3
    assert atoms.get_chemical_symbols() == ['O', 'H', 'H']
    assert list(atoms.get_tags()) == [0, 0, 1]
    assert atoms.info['onetep_species'] == ['O', 'HKO', 'H1']
    assert atoms.get_cell() == approx(rcell)
    assert atoms.positions == approx(rpositions)
    assert atoms.get_total_energy() == approx(-17.14642427760311 * 27.2116529)
    assert atoms.get_magnetic_moments() == approx([0, 0, 0])
    assert atoms.get_charges() == approx([-1.011, 0.506, 0.506])
    fdi = StringIO()
    write(fdi, atoms, format='onetep-in', keywords={'species_pot': []})
    fdi.seek(0)
    cycled_atoms = read(fdi, format='onetep-in')
    assert cycled_atoms.positions == approx(atoms.positions)
    assert cycled_atoms.get_cell().array \
        == approx(atoms.get_cell().array)
    assert cycled_atoms.get_chemical_symbols() == ['O', 'H', 'H']
    assert list(cycled_atoms.get_tags()) == [0, 0, 1]
    assert cycled_atoms.info['onetep_species'] == ['O', 'HKO', 'H1']


test_geom = """
|Linear-Scaling Ab Initio Total Energy Program|
--------------------------------------------------------------------------------
---------------------------------- INPUT FILE ----------------------------------
--------------------------------------------------------------------------------

cutoff_energy : 25
edft : F
forces_output_detail : verbose
output_detail : verbose
paw : F
do_properties = T
pseudo_path : /home/td5g20/PseudoPotentials/CASTEP
read_denskern : F
read_hamiltonian : F
read_tightbox_ngwfs : F
task : geometryoptimization
write_forces : T
xc_functional : pbe
zero_total_force: : T
spin_polarized : T
%BLOCK LATTICE_CART
ang
14.00000000       0.00000000       0.00000000
0.00000000      15.52647800       0.00000000
0.00000000       0.00000000      14.59630900
%ENDBLOCK LATTICE_CART
%BLOCK POSITIONS_ABS
ang
O     7.000000     7.763239     7.596309
H     7.000000     8.526478     7.000000
H     7.000000     7.000000     7.000000
%ENDBLOCK POSITIONS_ABS
%BLOCK SPECIES
H H 1 -1 9.0
O O 8 -1 9.0
%ENDBLOCK SPECIES
%BLOCK SPECIES_POT
H H_NCP19_PBE.usp
O O_NCP19_PBE.usp
%ENDBLOCK SPECIES_POT

--------------------------------------------------------------------------------
-------------------------------- END INPUT FILE --------------------------------
--------------------------------------------------------------------------------

================================================================================
<<<<<<<<<<<<<<<<<<<< Starting ONETEP Geometry Optimisation >>>>>>>>>>>>>>>>>>>>>
================================================================================

                           -------------------------------
                                     Cell Contents
                           -------------------------------

            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
            x  Element    Atom      Absolute co-ordinates of atoms     x
            x            Number        x           y           z       x
            x----------------------------------------------------------x
            x    O         1       13.228083   14.670396   14.354944   x
            x    H         2       13.228083   16.112708   13.228083   x
            x    H         3       13.228083   13.228083   13.228083   x
            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

           ........................................................
           |         *** NGWF optimisation converged ***          |
           | RMS NGWF gradient =    0.00000049365477              |
           | Criteria satisfied:                                  |
           | -> RMS NGWF gradient lower than set threshold.       |
           ========================================================

================================================================================
           ---------------- ENERGY COMPONENTS (Eh) ----------------
           | Kinetic                    :       12.15138700581343 |
           | Pseudopotential (local)    :      -45.29591196181978 |
           | Pseudo (non-coul chg cor)  :        0.00000000000000
           | Pseudopotential (non-local):       -1.33142848578819 |
           | Hartree                    :       17.94293076819679 |
           | Exchange-correlation       :       -4.25961788057561 |
           | Ewald                      :        3.64621627669446 |
           | Total                      :      -17.14642427747891 |
           --------------------------------------------------------
           ------ LOCAL ENERGY COMPONENTS FROM MATRIX TRACES ------
           | Pseudopotential (local)    :      -45.29591196181885 |
           | Hartree                    :       17.94293076819639 |
           --------------------------------------------------------
           Integrated density           :        7.99999999999935
           Integrated spin density      :        0.00000000000000
           Integrated |spin density|    :        0.00000000000000
           Local density approx. <S^2>  :        0.00000000000000
           Integrated density tr(KS)    :        8.00000000000000
           Integrated spin tr(KS)       :        0.00000000000000
================================================================================

********************* Unconstrained **********************
************************* Forces *************************
*                                                        *
* Element  Atom         Cartesian components (Eh/a)      *
* ------------------------------------------------------ *
*                       x            y            z      *
*                                                        *
*    O       1      0.00000017   0.00000001   0.00168584 *
*    H       2      0.00000005   0.00468665  -0.00084292 *
*    H       3     -0.00000021  -0.00468666  -0.00084292 *
*                                                        *
**********************************************************
* TOTAL:           -0.00000000   0.00000000   0.00000000 *
**********************************************************

                           -------------------------------
                                     Cell Contents
                           -------------------------------

            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
            x  Element    Atom      Absolute co-ordinates of atoms     x
            x            Number        x           y           z       x
            x----------------------------------------------------------x
            x    O         1       13.228083   14.670396   14.354944   x
            x    H         2       13.228083   16.112708   13.228083   x
            x    H         3       13.228083   13.228083   13.228083   x
            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx


================================================================================
 Starting BFGS iteration    1 ...
================================================================================

--------------------------------------------------------------------------------
 BFGS: starting iteration   1 with trial guess (lambda=    1.000000)
--------------------------------------------------------------------------------

                           -------------------------------
                                     Cell Contents
                           -------------------------------

            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
            x  Element    Atom      Absolute co-ordinates of atoms     x
            x            Number        x           y           z       x
            x----------------------------------------------------------x
            x    O         1       13.228083   14.670396   14.357610   x
            x    H         2       13.228083   16.120121   13.226750   x
            x    H         3       13.228083   13.220671   13.226750   x
            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

           ........................................................
           |         *** NGWF optimisation converged ***          |
           | RMS NGWF gradient =    0.00000171727103              |
           | Criteria satisfied:                                  |
           | -> RMS NGWF gradient lower than set threshold.       |
           ========================================================

================================================================================
           ---------------- ENERGY COMPONENTS (Eh) ----------------
           | Kinetic                    :       12.13481223219661 |
           | Pseudopotential (local)    :      -45.23291246538913 |
           | Pseudo (non-coul chg cor)  :        0.00000000000000
           | Pseudopotential (non-local):       -1.32646093297728 |
           | Hartree                    :       17.91882908342904 |
           | Exchange-correlation       :       -4.25560090215423 |
           | Ewald                      :        3.61486999206686 |
           | Total                      :      -17.14646299282811 |
           --------------------------------------------------------
           ------ LOCAL ENERGY COMPONENTS FROM MATRIX TRACES ------
           | Pseudopotential (local)    :      -45.23291246538828 |
           | Hartree                    :       17.91882908342841 |
           --------------------------------------------------------
           Integrated density           :        7.99999999999959
           Integrated spin density      :        0.00000000000000
           Integrated |spin density|    :        0.00000000000000
           Local density approx. <S^2>  :        0.00000000000000
           Integrated density tr(KS)    :        8.00000000000000
           Integrated spin tr(KS)       :        0.00000000000000
================================================================================


********************* Unconstrained **********************
************************* Forces *************************
*                                                        *
* Element  Atom         Cartesian components (Eh/a)      *
* ------------------------------------------------------ *
*                       x            y            z      *
*                                                        *
*    O       1      0.00000016  -0.00000002  -0.00307940 *
*    H       2      0.00000004   0.00097257   0.00153970 *
*    H       3     -0.00000020  -0.00097256   0.00153971 *
*                                                        *
**********************************************************
* TOTAL:            0.00000000   0.00000000   0.00000000 *
**********************************************************


================================================================================
 Starting BFGS iteration    2 ...
================================================================================


--------------------------------------------------------------------------------
 BFGS: starting iteration   2 with trial guess (lambda=    1.000000)
--------------------------------------------------------------------------------

                           -------------------------------
                                     Cell Contents
                           -------------------------------

            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
            x  Element    Atom      Absolute co-ordinates of atoms     x
            x            Number        x           y           z       x
            x----------------------------------------------------------x
            x    O         1       13.228084   14.670396   14.353542   x
            x    H         2       13.228083   16.124318   13.228784   x
            x    H         3       13.228082   13.216473   13.228784   x
            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

           ........................................................
           |         *** NGWF optimisation converged ***          |
           | RMS NGWF gradient =    0.00000192244606              |
           | Criteria satisfied:                                  |
           | -> RMS NGWF gradient lower than set threshold.       |
           ========================================================

================================================================================
           ---------------- ENERGY COMPONENTS (Eh) ----------------
           | Kinetic                    :       12.13593475185653 |
           | Pseudopotential (local)    :      -45.23397489927260 |
           | Pseudo (non-coul chg cor)  :        0.00000000000000
           | Pseudopotential (non-local):       -1.32747190453635 |
           | Hartree                    :       17.91926583391236 |
           | Exchange-correlation       :       -4.25563115178118 |
           | Ewald                      :        3.61539396585219 |
           | Total                      :      -17.14648340396906 |
           --------------------------------------------------------
           ------ LOCAL ENERGY COMPONENTS FROM MATRIX TRACES ------
           | Pseudopotential (local)    :      -45.23397489927187 |
           | Hartree                    :       17.91926583391191 |
           --------------------------------------------------------
           Integrated density           :        7.99999999999975
           Integrated spin density      :        0.00000000000000
           Integrated |spin density|    :        0.00000000000000
           Local density approx. <S^2>  :        0.00000000000000
           Integrated density tr(KS)    :        8.00000000000000
           Integrated spin tr(KS)       :        0.00000000000000
================================================================================

********************* Unconstrained **********************
************************* Forces *************************
*                                                        *
* Element  Atom         Cartesian components (Eh/a)      *
* ------------------------------------------------------ *
*                       x            y            z      *
*                                                        *
*    O       1      0.00000009   0.00000001  -0.00166320 *
*    H       2      0.00000007   0.00050115   0.00083161 *
*    H       3     -0.00000016  -0.00050116   0.00083160 *
*                                                        *
**********************************************************
* TOTAL:            0.00000000   0.00000000  -0.00000000 *
**********************************************************

--------------------------------------------------------------------------------
 BFGS: improving iteration   2 with line minimization (lambda=    2.139448)
--------------------------------------------------------------------------------

                           -------------------------------
                                     Cell Contents
                           -------------------------------

            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
            x  Element    Atom      Absolute co-ordinates of atoms     x
            x            Number        x           y           z       x
            x----------------------------------------------------------x
            x    O         1       13.228084   14.670396   14.348907   x
            x    H         2       13.228083   16.129100   13.231101   x
            x    H         3       13.228082   13.211691   13.231101   x
            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

           ........................................................
           |         *** NGWF optimisation converged ***          |
           | RMS NGWF gradient =    0.00000062544500              |
           | Criteria satisfied:                                  |
           | -> RMS NGWF gradient lower than set threshold.       |
           ========================================================

================================================================================
           ---------------- ENERGY COMPONENTS (Eh) ----------------
           | Kinetic                    :       12.13744677158684 |
           | Pseudopotential (local)    :      -45.23541018986488 |
           | Pseudo (non-coul chg cor)  :        0.00000000000000
           | Pseudopotential (non-local):       -1.32868109575617 |
           | Hartree                    :       17.92001657331161 |
           | Exchange-correlation       :       -4.25573633839171 |
           | Ewald                      :        3.61587019993617 |
           | Total                      :      -17.14649407917814 |
           --------------------------------------------------------
           ------ LOCAL ENERGY COMPONENTS FROM MATRIX TRACES ------
           | Pseudopotential (local)    :      -45.23541018986398 |
           | Hartree                    :       17.92001657331134 |
           --------------------------------------------------------
           Integrated density           :        7.99999999999966
           Integrated spin density      :       -0.00000000000000
           Integrated |spin density|    :        0.00000000000000
           Local density approx. <S^2>  :        0.00000000000000
           Integrated density tr(KS)    :        8.00000000000000
           Integrated spin tr(KS)       :       -0.00000000000000
================================================================================

********************* Unconstrained **********************
************************* Forces *************************
*                                                        *
* Element  Atom         Cartesian components (Eh/a)      *
* ------------------------------------------------------ *
*                       x            y            z      *
*                                                        *
*    O       1      0.00000005   0.00000004  -0.00041890 *
*    H       2      0.00000008   0.00005458   0.00020947 *
*    H       3     -0.00000013  -0.00005462   0.00020943 *
*                                                        *
**********************************************************
* TOTAL:            0.00000000  -0.00000000  -0.00000000 *
**********************************************************


================================================================================
 Starting BFGS iteration    3 ...
================================================================================


--------------------------------------------------------------------------------
 BFGS: starting iteration   3 with trial guess (lambda=    1.000000)
--------------------------------------------------------------------------------

                           -------------------------------
                                     Cell Contents
                           -------------------------------

            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
            x  Element    Atom      Absolute co-ordinates of atoms     x
            x            Number        x           y           z       x
            x----------------------------------------------------------x
            x    O         1       13.228084   14.670396   14.347758   x
            x    H         2       13.228083   16.130115   13.231676   x
            x    H         3       13.228081   13.210676   13.231676   x
            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

           ........................................................
           |         *** NGWF optimisation converged ***          |
           | RMS NGWF gradient =    0.00000195824432              |
           | Criteria satisfied:                                  |
           | -> RMS NGWF gradient lower than set threshold.       |
           ========================================================

================================================================================
           ---------------- ENERGY COMPONENTS (Eh) ----------------
           | Kinetic                    :       12.13806089541814 |
           | Pseudopotential (local)    :      -45.23657364189620 |
           | Pseudo (non-coul chg cor)  :        0.00000000000000
           | Pseudopotential (non-local):       -1.32907290394940 |
           | Hartree                    :       17.92040456637980 |
           | Exchange-correlation       :       -4.25580224655208 |
           | Ewald                      :        3.61648868205110 |
           | Total                      :      -17.14649464854864 |
           --------------------------------------------------------
           ------ LOCAL ENERGY COMPONENTS FROM MATRIX TRACES ------
           | Pseudopotential (local)    :      -45.23657364189524 |
           | Hartree                    :       17.92040456637923 |
           --------------------------------------------------------
           Integrated density           :        7.99999999999938
           Integrated spin density      :        0.00000000000000
           Integrated |spin density|    :        0.00000000000000
           Local density approx. <S^2>  :        0.00000000000000
           Integrated density tr(KS)    :        8.00000000000000
           Integrated spin tr(KS)       :        0.00000000000000
================================================================================

********************* Unconstrained **********************
************************* Forces *************************
*                                                        *
* Element  Atom         Cartesian components (Eh/a)      *
* ------------------------------------------------------ *
*                       x            y            z      *
*                                                        *
*    O       1      0.00000002  -0.00000005   0.00005977 *
*    H       2      0.00000009  -0.00001746  -0.00002990 *
*    H       3     -0.00000011   0.00001751  -0.00002987 *
*                                                        *
**********************************************************
* TOTAL:           -0.00000000   0.00000000   0.00000000 *
**********************************************************

================================================================================
 BFGS : Final Configuration:
================================================================================

                           -------------------------------
                                     Cell Contents
                           -------------------------------

            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
            x  Element    Atom      Absolute co-ordinates of atoms     x
            x            Number        x           y           z       x
            x----------------------------------------------------------x
            x    O         1       13.228084   14.670396   14.347758   x
            x    H         2       13.228083   16.130115   13.231676   x
            x    H         3       13.228081   13.210676   13.231676   x
            xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

           ........................................................
           |         *** NGWF optimisation converged ***          |
           | RMS NGWF gradient =    0.00000195826989              |
           | Criteria satisfied:                                  |
           | -> RMS NGWF gradient lower than set threshold.       |
           ========================================================

================================================================================
           ---------------- ENERGY COMPONENTS (Eh) ----------------
           | Kinetic                    :       12.13806088831265 |
           | Pseudopotential (local)    :      -45.23657354211792 |
           | Pseudo (non-coul chg cor)  :        0.00000000000000
           | Pseudopotential (non-local):       -1.32907295131071 |
           | Hartree                    :       17.92040450997095 |
           | Exchange-correlation       :       -4.25580223545482 |
           | Ewald                      :        3.61648868205110 |
           | Total                      :      -17.14649464854874 |
           --------------------------------------------------------
           ------ LOCAL ENERGY COMPONENTS FROM MATRIX TRACES ------
           | Pseudopotential (local)    :      -45.23657354211692 |
           | Hartree                    :       17.92040450997042 |
           --------------------------------------------------------
           Integrated density           :        7.99999999999944
           Integrated spin density      :        0.00000000000000
           Integrated |spin density|    :        0.00000000000000
           Local density approx. <S^2>  :        0.00000000000000
           Integrated density tr(KS)    :        8.00000000000000
           Integrated spin tr(KS)       :       -0.00000000000000
================================================================================

    Mulliken Atomic Populations
    ---------------------------
Species  Ion    Total   Charge (e)    Spin (hbar)
=================================================
  O      1       7.009     -1.009     -0.00
  H      2       0.495      0.505      0.00
  H      3       0.495      0.505      0.00
=================================================

********************* Unconstrained **********************
************************* Forces *************************
*                                                        *
* Element  Atom         Cartesian components (Eh/a)      *
* ------------------------------------------------------ *
*                       x            y            z      *
*                                                        *
*    O       1      0.00000002  -0.00000005   0.00005976 *
*    H       2      0.00000009  -0.00001747  -0.00002990 *
*    H       3     -0.00000011   0.00001751  -0.00002986 *
*                                                        *
**********************************************************
* TOTAL:            0.00000000   0.00000000   0.00000000 *
**********************************************************
"""


def test_geom_output():
    positions_to_test = [14.354944, 14.357610, 14.353542, 14.347758]
    forces_to_test = [0.00168584, -0.00307940, -0.00166320, 0.00005977]
    energy_to_test = [-17.14642427747891, -17.14646299282811,
                      -17.14648340396906, -17.14649464854864]
    fd = StringIO(test_geom)
    atoms = read(fd, format='onetep-out', index="::")
    # Last configuration should have the properties:
    last_charges = atoms[-1].get_charges()
    last_magnetic_moments = atoms[-1].get_magnetic_moments()
    assert last_charges == approx([-1.009, 0.505, 0.505])
    assert last_magnetic_moments == approx([0, 0, 0])
    # We test the z positions of the oxygen atom
    # and we test if the forces correspond to the correct
    # positions
    for i, x in enumerate(atoms):
        tmp = x.positions[0][-1] * angtobohr
        assert tmp == approx(positions_to_test[i])
        assert x.get_forces()[
            0][-1] == approx(forces_to_test[i] * eV2au * angtobohr)
        assert x.get_total_energy() == approx(energy_to_test[i] * eV2au)


# We test an output that has different ONETEP
# outputs in it. (multi-calculation output)
weird_output = '\n'.join([test_output,
                         test_geom,
                         test_output])


def test_weird_output():
    energy_to_test = [-17.14642427760311, -17.14642427747891,
                      -17.14646299282811, -17.14648340396906,
                      -17.14649464854864, -17.14642427760311]
    fd = StringIO(weird_output)
    atoms_list = read(fd, format='onetep-out', index="::")
    # Should be 1 + 4 + 1
    assert len(atoms_list) == 6
    # The before last configuration should be the same as the
    # last configuration of the previous test
    last_charges = atoms_list[-2].get_charges()
    last_magnetic_moments = atoms_list[-2].get_magnetic_moments()
    assert last_charges == approx([-1.009, 0.505, 0.505])
    assert last_magnetic_moments == approx([0, 0, 0])
    # The first and last energy should be equal
    first_energy = atoms_list[0].get_total_energy()
    last_energy = atoms_list[-1].get_total_energy()
    # The most important: we test all the
    # energy to make sure configurations
    # didn't get mixed up...
    assert first_energy == approx(last_energy)
    for ref_energy, test_atoms in zip(energy_to_test, atoms_list):
        test_energy = test_atoms.get_potential_energy()
        assert test_energy == approx(ref_energy * eV2au)
