<?php

/**
* Does everything relevant to text drawing
*
* The routine for dealing text is to load a font, set an image to draw on,
* and draw a string.
*
* @version 0.3
* @author  Matt McClanahan <cardinal@dodds.net>
* @package Imlib
* @access public
*/
class ImlibText extends ImlibColor
{
   /**
   * Resource id# of the currently loaded font
   *
   * @access private
   */
   var $fnt;

   /**
   * Resource id# of the image to draw on
   *
   * @access private
   */
   var $im;

   /**
   * ImlibText constructor
   *
   * @access public
   */
   function ImlibText()
   {
      $this->fnt = 0;
      $this->im = 0;
   }

   /**
   * Draw a text string onto the current image
   *
   * @param int X coordinate of the upper left corner of the string
   * @param int Y coordinate of the upper left corner of the string
   * @param string Text string to draw
   * @param int Direction of the text
   * @return bool False if no font is loaded
   * @access public
   */
   function draw($x,$y,$str,$dir=IMLIB_TEXT_TO_RIGHT)
   {
      if (!is_resource($this->fnt) || !is_resource($this->im))
         return false;

      // If we don't have a color set yet, we'll default to white
      if (!$this->get_color($r,$g,$b,$a))
         list($r,$g,$b,$a) = Array(255,255,255,255);

      return imlib_text_draw($this->im,$this->fnt,$x,$y,$str,$dir,
                             $r,$g,$b,$a);
   }

   /**
   * Free the currently loaded font
   *
   * @return bool False if no font is loaded
   * @access public
   */
   function free()
   {
      if (!is_resource($this->fnt))
         return false;

      imlib_free_font($this->fnt);
      $this->fnt = 0;
   }

   /**
   * Get the current image resource id#
   *
   * @return integer Image resource id#
   * @access public
   */
   function get_image()
   {
      return $this->im;
   }

   /**
   * Get the width and height of a given string if it were drawn
   *
   * @param string String to measure
   * @param int &$w Width the string would be
   * @param int &$h Height the string would be
   * @param int Direction to measure the string
   * @return bool False if no font is loaded
   * @access public
   */
   function get_size($str,&$w,&$h,$dir=IMLIB_TEXT_TO_RIGHT)
   {
      if (!is_resource($this->fnt))
         return false;

      return imlib_get_text_size($this->fnt,$str,&$fw,&$fh,$dir);
   }

   /**
   * Load a font
   *
   * If the font is in the font path, it can be referred to by font name,
   * and the path does not need to be specified.  Otherwise, it must be
   * referred to by filename.
   *
   * @param string Path, or current directory if passed blank
   * @param string Font name
   * @param int Font size
   * @return bool False if no font is loaded
   * @access public
   */
   function load($path,$font,$size)
   {
      $fontdata = '';

      if (is_resource($this->fnt))
         return false;

      if ($path)
         $fontdata = $path;
      $fontdata .= $font . '/' . $size;

      return $this->fnt = imlib_load_font($fontdata);
   }

   /**
   * Set the image resource id# to draw on
   *
   * @param int Image resource id#
   * @access public
   */
   function set_image($im)
   {
      $this->im = $im;
   }
};

?>
