<?php

declare(strict_types=1);

namespace Doctrine\DBAL\Tests\Platforms;

use Doctrine\DBAL\Exception\InvalidColumnDeclaration;
use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\DBAL\Platforms\DB2Platform;
use Doctrine\DBAL\Schema\Column;
use Doctrine\DBAL\Schema\ColumnDiff;
use Doctrine\DBAL\Schema\ForeignKeyConstraint;
use Doctrine\DBAL\Schema\Index;
use Doctrine\DBAL\Schema\PrimaryKeyConstraint;
use Doctrine\DBAL\Schema\Table;
use Doctrine\DBAL\Schema\TableDiff;
use Doctrine\DBAL\Types\Types;
use PHPUnit\Framework\Attributes\DataProvider;

/** @extends AbstractPlatformTestCase<DB2Platform> */
class DB2PlatformTest extends AbstractPlatformTestCase
{
    public function createPlatform(): AbstractPlatform
    {
        return new DB2Platform();
    }

    protected function getGenerateForeignKeySql(): string
    {
        return 'ALTER TABLE test ADD FOREIGN KEY (fk_name_id) REFERENCES other_table (id)';
    }

    public function getGenerateIndexSql(): string
    {
        return 'CREATE INDEX my_idx ON mytable (user_name, last_login)';
    }

    public function getGenerateTableSql(): string
    {
        return 'CREATE TABLE test (id INTEGER GENERATED BY DEFAULT AS IDENTITY NOT NULL, '
            . 'test VARCHAR(255) DEFAULT NULL, PRIMARY KEY (id))';
    }

    /**
     * {@inheritDoc}
     */
    public function getGenerateTableWithMultiColumnUniqueIndexSql(): array
    {
        return [
            'CREATE TABLE test (foo VARCHAR(255) DEFAULT NULL, bar VARCHAR(255) DEFAULT NULL)',
            'CREATE UNIQUE INDEX UNIQ_D87F7E0C8C73652176FF8CAA ON test (foo, bar)',
        ];
    }

    public function getGenerateUniqueIndexSql(): string
    {
        return 'CREATE UNIQUE INDEX index_name ON test (test, test2)';
    }

    /**
     * {@inheritDoc}
     */
    protected function getQuotedColumnInForeignKeySQL(): array
    {
        return [
            'CREATE TABLE "quoted" ("create" VARCHAR(255) NOT NULL, foo VARCHAR(255) NOT NULL, '
                . '"bar" VARCHAR(255) NOT NULL)',
            'ALTER TABLE "quoted" ADD CONSTRAINT FK_WITH_RESERVED_KEYWORD FOREIGN KEY ("create", foo, "bar")'
                . ' REFERENCES "foreign" ("create", bar, "foo-bar")',
            'ALTER TABLE "quoted" ADD CONSTRAINT FK_WITH_NON_RESERVED_KEYWORD FOREIGN KEY ("create", foo, "bar")'
                . ' REFERENCES foo ("create", bar, "foo-bar")',
            'ALTER TABLE "quoted" ADD CONSTRAINT FK_WITH_INTENDED_QUOTATION FOREIGN KEY ("create", foo, "bar")'
                . ' REFERENCES "foo-bar" ("create", bar, "foo-bar")',
            'CREATE INDEX IDX_22660D028FD6E0FB8C7365216D704F76 ON "quoted" ("create", foo, "bar")',
        ];
    }

    /**
     * {@inheritDoc}
     */
    protected function getQuotedColumnInIndexSQL(): array
    {
        return [
            'CREATE TABLE "quoted" ("create" VARCHAR(255) NOT NULL)',
            'CREATE INDEX IDX_22660D028FD6E0FB ON "quoted" ("create")',
        ];
    }

    /**
     * {@inheritDoc}
     */
    protected function getQuotedNameInIndexSQL(): array
    {
        return [
            'CREATE TABLE test (column1 VARCHAR(255) NOT NULL)',
            'CREATE INDEX "key" ON test (column1)',
        ];
    }

    /**
     * {@inheritDoc}
     */
    protected function getQuotedColumnInPrimaryKeySQL(): array
    {
        return ['CREATE TABLE "quoted" ("create" VARCHAR(255) NOT NULL, PRIMARY KEY ("create"))'];
    }

    protected function getBitAndComparisonExpressionSql(string $value1, string $value2): string
    {
        return 'BITAND(' . $value1 . ', ' . $value2 . ')';
    }

    protected function getBitOrComparisonExpressionSql(string $value1, string $value2): string
    {
        return 'BITOR(' . $value1 . ', ' . $value2 . ')';
    }

    public function testGeneratesCreateTableSQLWithCommonIndexes(): void
    {
        $table = Table::editor()
            ->setUnquotedName('test')
            ->setColumns(
                Column::editor()
                    ->setUnquotedName('id')
                    ->setTypeName(Types::INTEGER)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('name')
                    ->setTypeName(Types::STRING)
                    ->setLength(50)
                    ->create(),
            )
            ->setPrimaryKeyConstraint(
                PrimaryKeyConstraint::editor()
                    ->setUnquotedColumnNames('id')
                    ->create(),
            )
            ->create();

        $table->addIndex(['name']);
        $table->addIndex(['id', 'name'], 'composite_idx');

        self::assertEquals(
            [
                'CREATE TABLE test (id INTEGER NOT NULL, name VARCHAR(50) NOT NULL, PRIMARY KEY (id))',
                'CREATE INDEX IDX_D87F7E0C5E237E06 ON test (name)',
                'CREATE INDEX composite_idx ON test (id, name)',
            ],
            $this->platform->getCreateTableSQL($table),
        );
    }

    public function testGeneratesCreateTableSQLWithForeignKeyConstraints(): void
    {
        $table = Table::editor()
            ->setUnquotedName('test')
            ->setColumns(
                Column::editor()
                    ->setUnquotedName('id')
                    ->setTypeName(Types::INTEGER)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('fk_1')
                    ->setTypeName(Types::INTEGER)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('fk_2')
                    ->setTypeName(Types::INTEGER)
                    ->create(),
            )
            ->setPrimaryKeyConstraint(
                PrimaryKeyConstraint::editor()
                    ->setUnquotedColumnNames('id')
                    ->create(),
            )
            ->setForeignKeyConstraints(
                ForeignKeyConstraint::editor()
                    ->setUnquotedReferencingColumnNames('fk_1', 'fk_2')
                    ->setUnquotedReferencedTableName('foreign_table')
                    ->setUnquotedReferencedColumnNames('pk_1', 'pk_2')
                    ->create(),
                ForeignKeyConstraint::editor()
                    ->setUnquotedName('named_fk')
                    ->setUnquotedReferencingColumnNames('fk_1', 'fk_2')
                    ->setUnquotedReferencedTableName('foreign_table2')
                    ->setUnquotedReferencedColumnNames('pk_1', 'pk_2')
                    ->create(),
            )
            ->create();

        self::assertEquals(
            [
                'CREATE TABLE test (id INTEGER NOT NULL, fk_1 INTEGER NOT NULL, fk_2 INTEGER NOT NULL'
                    . ', PRIMARY KEY (id))',
                'ALTER TABLE test ADD FOREIGN KEY (fk_1, fk_2)'
                    . ' REFERENCES foreign_table (pk_1, pk_2)',
                'ALTER TABLE test ADD CONSTRAINT named_fk FOREIGN KEY (fk_1, fk_2)'
                    . ' REFERENCES foreign_table2 (pk_1, pk_2)',
                'CREATE INDEX IDX_D87F7E0C177612A38E7F4319 ON test (fk_1, fk_2)',
            ],
            $this->platform->getCreateTableSQL($table),
        );
    }

    public function testGeneratesCreateTableSQLWithCheckConstraints(): void
    {
        $table = Table::editor()
            ->setUnquotedName('test')
            ->setColumns(
                Column::editor()
                    ->setUnquotedName('id')
                    ->setTypeName(Types::INTEGER)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('check_max')
                    ->setTypeName(Types::INTEGER)
                    ->setMaximumValue(10)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('check_min')
                    ->setTypeName(Types::INTEGER)
                    ->setMinimumValue(10)
                    ->create(),
            )
            ->setPrimaryKeyConstraint(
                PrimaryKeyConstraint::editor()
                    ->setUnquotedColumnNames('id')
                    ->create(),
            )
            ->create();

        self::assertEquals(
            [
                'CREATE TABLE test (id INTEGER NOT NULL, check_max INTEGER NOT NULL, '
                    . 'check_min INTEGER NOT NULL, PRIMARY KEY (id), CHECK (check_max <= 10), CHECK (check_min >= 10))',
            ],
            $this->platform->getCreateTableSQL($table),
        );
    }

    public function testGeneratesColumnTypesDeclarationSQL(): void
    {
        $fullColumnDef = [
            'length' => 10,
            'fixed' => true,
            'unsigned' => true,
            'autoincrement' => true,
        ];

        self::assertEquals('SMALLINT', $this->platform->getSmallIntTypeDeclarationSQL([]));
        self::assertEquals('SMALLINT', $this->platform->getSmallIntTypeDeclarationSQL(['unsigned' => true]));

        self::assertEquals(
            'SMALLINT GENERATED BY DEFAULT AS IDENTITY',
            $this->platform->getSmallIntTypeDeclarationSQL($fullColumnDef),
        );

        self::assertEquals('INTEGER', $this->platform->getIntegerTypeDeclarationSQL([]));
        self::assertEquals('INTEGER', $this->platform->getIntegerTypeDeclarationSQL(['unsigned' => true]));

        self::assertEquals(
            'INTEGER GENERATED BY DEFAULT AS IDENTITY',
            $this->platform->getIntegerTypeDeclarationSQL($fullColumnDef),
        );

        self::assertEquals('BIGINT', $this->platform->getBigIntTypeDeclarationSQL([]));
        self::assertEquals('BIGINT', $this->platform->getBigIntTypeDeclarationSQL(['unsigned' => true]));

        self::assertEquals(
            'BIGINT GENERATED BY DEFAULT AS IDENTITY',
            $this->platform->getBigIntTypeDeclarationSQL($fullColumnDef),
        );

        self::assertEquals('BLOB(1M)', $this->platform->getBlobTypeDeclarationSQL($fullColumnDef));
        self::assertEquals('SMALLINT', $this->platform->getBooleanTypeDeclarationSQL($fullColumnDef));
        self::assertEquals('CLOB(1M)', $this->platform->getClobTypeDeclarationSQL($fullColumnDef));
        self::assertEquals('DATE', $this->platform->getDateTypeDeclarationSQL($fullColumnDef));

        self::assertEquals(
            'TIMESTAMP(0) WITH DEFAULT',
            $this->platform->getDateTimeTypeDeclarationSQL(['version' => true]),
        );

        self::assertEquals('TIMESTAMP(0)', $this->platform->getDateTimeTypeDeclarationSQL($fullColumnDef));
        self::assertEquals('TIME', $this->platform->getTimeTypeDeclarationSQL($fullColumnDef));
    }

    public function testGeneratesDDLSnippets(): void
    {
        self::assertEquals('DECLARE GLOBAL TEMPORARY TABLE', $this->platform->getCreateTemporaryTableSnippetSQL());
        self::assertEquals('TRUNCATE foobar IMMEDIATE', $this->platform->getTruncateTableSQL('foobar'));

        $viewSql = 'SELECT * FROM footable';

        self::assertEquals(
            'CREATE VIEW fooview AS ' . $viewSql,
            $this->platform->getCreateViewSQL('fooview', $viewSql),
        );

        self::assertEquals('DROP VIEW fooview', $this->platform->getDropViewSQL('fooview'));
    }

    public function testGeneratesCreateUnnamedPrimaryKeySQL(): void
    {
        self::assertEquals(
            'ALTER TABLE foo ADD PRIMARY KEY (a, b)',
            $this->platform->getCreatePrimaryKeySQL(
                new Index('any_pk_name', ['a', 'b'], true, true),
                'foo',
            ),
        );
    }

    public function testGeneratesSQLSnippets(): void
    {
        self::assertEquals('CURRENT DATE', $this->platform->getCurrentDateSQL());
        self::assertEquals('CURRENT TIME', $this->platform->getCurrentTimeSQL());
        self::assertEquals('CURRENT TIMESTAMP', $this->platform->getCurrentTimestampSQL());
        self::assertEquals("'1987/05/02' + 4 DAY", $this->platform->getDateAddDaysExpression("'1987/05/02'", '4'));
        self::assertEquals("'1987/05/02' + 12 HOUR", $this->platform->getDateAddHourExpression("'1987/05/02'", '12'));

        self::assertEquals(
            "'1987/05/02' + 2 MINUTE",
            $this->platform->getDateAddMinutesExpression("'1987/05/02'", '2'),
        );

        self::assertEquals(
            "'1987/05/02' + 102 MONTH",
            $this->platform->getDateAddMonthExpression("'1987/05/02'", '102'),
        );

        self::assertEquals(
            "'1987/05/02' + (5 * 3) MONTH",
            $this->platform->getDateAddQuartersExpression("'1987/05/02'", '5'),
        );

        self::assertEquals(
            "'1987/05/02' + 1 SECOND",
            $this->platform->getDateAddSecondsExpression("'1987/05/02'", '1'),
        );

        self::assertEquals(
            "'1987/05/02' + (3 * 7) DAY",
            $this->platform->getDateAddWeeksExpression("'1987/05/02'", '3'),
        );

        self::assertEquals("'1987/05/02' + 10 YEAR", $this->platform->getDateAddYearsExpression("'1987/05/02'", '10'));

        self::assertEquals(
            "DAYS('1987/05/02') - DAYS('1987/04/01')",
            $this->platform->getDateDiffExpression("'1987/05/02'", "'1987/04/01'"),
        );

        self::assertEquals("'1987/05/02' - 4 DAY", $this->platform->getDateSubDaysExpression("'1987/05/02'", '4'));
        self::assertEquals("'1987/05/02' - 12 HOUR", $this->platform->getDateSubHourExpression("'1987/05/02'", '12'));

        self::assertEquals(
            "'1987/05/02' - 2 MINUTE",
            $this->platform->getDateSubMinutesExpression("'1987/05/02'", '2'),
        );

        self::assertEquals(
            "'1987/05/02' - 102 MONTH",
            $this->platform->getDateSubMonthExpression("'1987/05/02'", '102'),
        );

        self::assertEquals(
            "'1987/05/02' - (5 * 3) MONTH",
            $this->platform->getDateSubQuartersExpression("'1987/05/02'", '5'),
        );

        self::assertEquals(
            "'1987/05/02' - 1 SECOND",
            $this->platform->getDateSubSecondsExpression("'1987/05/02'", '1'),
        );

        self::assertEquals(
            "'1987/05/02' - (3 * 7) DAY",
            $this->platform->getDateSubWeeksExpression("'1987/05/02'", '3'),
        );

        self::assertEquals("'1987/05/02' - 10 YEAR", $this->platform->getDateSubYearsExpression("'1987/05/02'", '10'));

        self::assertEquals(
            'LOCATE(substring_column, string_column)',
            $this->platform->getLocateExpression('string_column', 'substring_column'),
        );

        self::assertEquals(
            'LOCATE(substring_column, string_column, 1)',
            $this->platform->getLocateExpression('string_column', 'substring_column', '1'),
        );

        self::assertEquals('SUBSTR(column, 5)', $this->platform->getSubstringExpression('column', '5'));
        self::assertEquals('SUBSTR(column, 5, 2)', $this->platform->getSubstringExpression('column', '5', '2'));
    }

    public function testSupportsIdentityColumns(): void
    {
        self::assertTrue($this->platform->supportsIdentityColumns());
    }

    public function testDoesNotSupportSavePoints(): void
    {
        self::assertFalse($this->platform->supportsSavepoints());
    }

    public function testDoesNotSupportReleasePoints(): void
    {
        self::assertFalse($this->platform->supportsReleaseSavepoints());
    }

    public function testGetVariableLengthStringTypeDeclarationSQLNoLength(): void
    {
        $this->expectException(InvalidColumnDeclaration::class);

        parent::testGetVariableLengthStringTypeDeclarationSQLNoLength();
    }

    public function getExpectedFixedLengthBinaryTypeDeclarationSQLNoLength(): string
    {
        return 'CHAR FOR BIT DATA';
    }

    public function getExpectedFixedLengthBinaryTypeDeclarationSQLWithLength(): string
    {
        return 'CHAR(16) FOR BIT DATA';
    }

    public function getExpectedVariableLengthBinaryTypeDeclarationSQLNoLength(): string
    {
        return 'CHAR(16) FOR BIT DATA';
    }

    public function testGetVariableLengthBinaryTypeDeclarationSQLNoLength(): void
    {
        $this->expectException(InvalidColumnDeclaration::class);

        parent::testGetVariableLengthBinaryTypeDeclarationSQLNoLength();
    }

    public function getExpectedVariableLengthBinaryTypeDeclarationSQLWithLength(): string
    {
        return 'VARCHAR(16) FOR BIT DATA';
    }

    /**
     * {@inheritDoc}
     */
    protected function getAlterTableRenameIndexSQL(): array
    {
        return ['RENAME INDEX idx_foo TO idx_bar'];
    }

    /**
     * {@inheritDoc}
     */
    protected function getQuotedAlterTableRenameIndexSQL(): array
    {
        return [
            'RENAME INDEX "create" TO "select"',
            'RENAME INDEX "foo" TO "bar"',
        ];
    }

    /**
     * {@inheritDoc}
     */
    protected function getAlterTableRenameIndexInSchemaSQL(): array
    {
        return ['RENAME INDEX myschema.idx_foo TO idx_bar'];
    }

    /**
     * {@inheritDoc}
     */
    protected function getQuotedAlterTableRenameIndexInSchemaSQL(): array
    {
        return [
            'RENAME INDEX "schema"."create" TO "select"',
            'RENAME INDEX "schema"."foo" TO "bar"',
        ];
    }

    public function testReturnsGuidTypeDeclarationSQL(): void
    {
        self::assertSame('CHAR(36)', $this->platform->getGuidTypeDeclarationSQL([]));
    }

    /**
     * {@inheritDoc}
     */
    protected function getCommentOnColumnSQL(): array
    {
        return [
            'COMMENT ON COLUMN foo.bar IS \'comment\'',
            'COMMENT ON COLUMN "Foo"."BAR" IS \'comment\'',
            'COMMENT ON COLUMN "select"."from" IS \'comment\'',
        ];
    }

    #[DataProvider('getGeneratesAlterColumnSQL')]
    public function testGeneratesAlterColumnSQL(
        Column $oldColumn,
        Column $newColumn,
        ?string $expectedSQLClause,
        bool $shouldReorg = true,
    ): void {
        $table = Table::editor()
            ->setUnquotedName('foo')
            ->setColumns(
                Column::editor()
                    ->setUnquotedName('id')
                    ->setTypeName(Types::INTEGER)
                    ->create(),
            )
            ->create();

        $tableDiff = new TableDiff($table, changedColumns: [
            $oldColumn->getObjectName()->toString() => new ColumnDiff($oldColumn, $newColumn),
        ]);

        $expectedSQL = [];

        if ($expectedSQLClause !== null) {
            $expectedSQL[] = 'ALTER TABLE foo ALTER COLUMN bar ' . $expectedSQLClause;
        }

        if ($shouldReorg) {
            $expectedSQL[] = "CALL SYSPROC.ADMIN_CMD ('REORG TABLE foo')";
        }

        self::assertSame($expectedSQL, $this->platform->getAlterTableSQL($tableDiff));
    }

    /** @return iterable<array{Column, Column, ?string, 3?: bool}> */
    public static function getGeneratesAlterColumnSQL(): iterable
    {
        return [
            [
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::DECIMAL)
                    ->setColumnDefinition('MONEY NULL')
                    ->create(),
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::DECIMAL)
                    ->setColumnDefinition('MONEY NOT NULL')
                    ->create(),
                'MONEY NOT NULL',
            ],
            [
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::INTEGER)
                    ->create(),
                'SET DATA TYPE INTEGER',
            ],
            [
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->setLength(50)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->setLength(100)
                    ->create(),
                'SET DATA TYPE VARCHAR(100)',
            ],
            [
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::DECIMAL)
                    ->setPrecision(8)
                    ->setScale(2)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::DECIMAL)
                    ->setPrecision(10)
                    ->setScale(2)
                    ->create(),
                'SET DATA TYPE NUMERIC(10, 2)',
            ],
            [
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::DECIMAL)
                    ->setPrecision(5)
                    ->setScale(3)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::DECIMAL)
                    ->setPrecision(5)
                    ->setScale(4)
                    ->create(),
                'SET DATA TYPE NUMERIC(5, 4)',
            ],
            [
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->setLength(10)
                    ->setFixed(true)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->setLength(20)
                    ->setFixed(true)
                    ->create(),
                'SET DATA TYPE CHAR(20)',
            ],
            [
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->setNotNull(false)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->create(),
                'SET NOT NULL',
            ],
            [
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->setNotNull(false)
                    ->create(),
                'DROP NOT NULL',
            ],
            [
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->setDefaultValue('foo')
                    ->create(),
                "SET DEFAULT 'foo'",
            ],
            [
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::INTEGER)
                    ->create(),
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::INTEGER)
                    ->setAutoincrement(true)
                    ->setDefaultValue(666)
                    ->create(),
                null,
                false,
            ],
            [
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->setDefaultValue('foo')
                    ->create(),
                Column::editor()
                    ->setUnquotedName('bar')
                    ->setTypeName(Types::STRING)
                    ->create(),
                'DROP DEFAULT',
            ],
        ];
    }

    protected function getQuotesReservedKeywordInUniqueConstraintDeclarationSQL(): string
    {
        return 'CONSTRAINT "select" UNIQUE (foo)';
    }

    protected function getQuotesReservedKeywordInIndexDeclarationSQL(): string
    {
        return ''; // not supported by this platform
    }

    protected function getQuotesReservedKeywordInTruncateTableSQL(): string
    {
        return 'TRUNCATE "select" IMMEDIATE';
    }

    protected function supportsInlineIndexDeclaration(): bool
    {
        return false;
    }

    protected function supportsCommentOnStatement(): bool
    {
        return true;
    }

    /**
     * {@inheritDoc}
     */
    protected function getAlterStringToFixedStringSQL(): array
    {
        return [
            'ALTER TABLE mytable ALTER COLUMN name SET DATA TYPE CHAR(2)',
            'CALL SYSPROC.ADMIN_CMD (\'REORG TABLE mytable\')',
        ];
    }

    /**
     * {@inheritDoc}
     */
    protected function getGeneratesAlterTableRenameIndexUsedByForeignKeySQL(): array
    {
        return ['RENAME INDEX idx_foo TO idx_foo_renamed'];
    }
}
