/*
    This file is part of Msc-generator.
    Copyright (C) 2008-2021 Zoltan Turanyi
    Distributed under GNU Affero General Public License.

    Msc-generator is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Msc-generator is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with Msc-generator.  If not, see <http://www.gnu.org/licenses/>.
*/
/** @file style.cpp The implementation of styles (MscStyle) and contexts (ContextBase).
 * @ingroup libmscgen_files */


#include "msc.h"

using namespace msc;

/** Returns true if value `v` is valid for side type `t` */
bool IsValidSideValue(ESideType t, ESide v)
{
    return ((v==ESide::LEFT || v==ESide::RIGHT) && (t==ESideType::ANY || t==ESideType::LEFT_RIGHT)) ||
                                                     (v==ESide::END && t==ESideType::ANY);
}


/** Create an empty style that contains all possible attributes.*/
MscStyle::MscStyle(EStyleType tt, EColorMeaning cm) : 
    SimpleStyleWithArrow<MscArrowHeads>(tt, cm)
{
    f_vline=f_vfill=true;
    f_solid=f_vspacing=f_indicator=true;
    f_makeroom=f_note=f_lost=f_lsym=true;
    f_alt = false;
    f_side = ESideType::ANY;
    f_tag = true;
    Empty();
}

/** Create an empty style that contains only some of the attributes.
 * @param [in] tt The type of style instance.
 * @param [in] cm How an unqualified "color" attribute shall be interpreted.
 * @param [in] a What type of arrows we accept.
 * @param [in] alt True if the style shall contain aline and atext attributes.
 * @param [in] t True if the style shall contain text attributes.
 * @param [in] l True if the style shall contain line attributes.
 * @param [in] f True if the style shall contain fill attributes.
 * @param [in] s True if the style shall contain shadow attributes.
 * @param [in] vl True if the style shall contain vline attributes.
 * @param [in] so True if the style shall contain the `solid` attribute.
 * @param [in] nu True if the style shall contain the `number` attribute.
 * @param [in] co True if the style shall contain the `compress`/`vspacing` attribute.
 * @param [in] si True if the style shall contain the `side` attribute.
 * @param [in] i True if the style shall contain the `indicator` attribute.
 * @param [in] vf True if the style shall contain vfill attributes.
 * @param [in] mr True if the style shall contain the `makeroom` attribute.
 * @param [in] n True if the style shall contain note attributes.
 * @param [in] lo True if the style shall contain lost.* attributes.
 * @param [in] lsym True if the style shall contain loss symbol attributes (x.*)
 * @param [in] shp True if the style shall contain entity related attributes (shape, shape.size)
 * @param [in] tag True if the style shall contain box tag related attributes (tag.*)
 */
MscStyle::MscStyle(EStyleType tt, EColorMeaning cm, EArcArrowType a, 
                   bool alt, bool t, bool l, bool f, bool s, bool vl,
                   bool so, bool nu, bool co, ESideType si, bool i, bool vf, bool mr, bool n, 
                   bool lo, bool lsym, bool shp, bool tag) :
    SimpleStyleWithArrow<MscArrowHeads>(tt, cm, a, t, l, f, s, nu, shp),
    lost_arrow(EArcArrowType::ARROW), 
    f_vline(vl), f_vfill(vf), 
    f_solid(so), f_vspacing(co), f_side(si),
    f_indicator(i), f_makeroom(mr), f_note(n), f_alt(alt),
    f_lost(lo), f_lsym(lsym), f_tag(tag)
{
    Empty();
}

/** Make a style complete by setting the default values - but leave text & lost attributes empty.
 * Default attributes are half solid, right side, no numbering, no compress, indicator yes,
 * makeroom yes and the default of other attribute classes (specified there).
 * We skip text styles, since we have a global text attribute per chart and 
 * we set the default there. We also skip loss attributes as they do not have to be
 * fully specified.
 */
void MscStyle::MakeCompleteButText() 
{
    SimpleStyleWithArrow<MscArrowHeads>::MakeCompleteButText();
    if (f_vline) vline.MakeComplete();
    else vline.Empty();
    if (f_vfill) vfill.MakeComplete();
    else vfill.Empty();
    //no change to aline and atext - those may remain incomplete
    solid.is_set=f_solid;
    solid.value = 128;
    side.is_set = f_side != ESideType::NO;  
    side.value = ESide::RIGHT;
    vspacing.is_set = f_vspacing;
    vspacing.value = 0;
    indicator.is_set = f_indicator;
    indicator.value = true;
    makeroom.is_set = f_makeroom;
    makeroom.value = true;
    if (f_note) note.MakeComplete();
    else note.Empty();
    //no change to lost fields - those may remain incomplete, 
    lsym_line.MakeComplete(); 
    lsym_line.color.value = ColorType(255, 0, 0);
    lsym_line.width.value = 2;
    lsym_size.is_set = true;
    lsym_size.value = EArrowSize::NORMAL;
    tag_line.MakeComplete();
    tag_fill.MakeComplete();
    //tag_text unchanged
}

/** Make the style empty be unsetting all attributes it contains.*/
void MscStyle::Empty()
{
    SimpleStyleWithArrow<MscArrowHeads>::Empty();
    vline.Empty();
    vfill.Empty();
    aline.Empty();
    atext.Empty();
    solid.is_set = false;
    side.is_set = false;
    vspacing.is_set = false;
    indicator.is_set = false;
    makeroom.is_set = false;
    note.Empty();
    lost_line.Empty();
    lost_arrow.Empty();
    lost_text.Empty();
    lsym_line.Empty();
    lsym_size.is_set = false;
    tag_line.Empty();
    tag_fill.Empty();
    tag_text.Empty();
}

/** Merge another style to us by copying the value of those attributes which are contained by both and are set in `toadd`.*/
Style & MscStyle::operator +=(const Style &t)
{
    SimpleStyleWithArrow<MscArrowHeads>::operator+=(t);
    const MscStyle *p = static_cast<const MscStyle*>(&t);
    if (p==nullptr) return *this;

    if (p->f_vline && f_vline) vline += p->vline;
    if (p->f_vfill && f_vfill) vfill += p->vfill;
    if (p->f_alt && f_alt) { 
        aline += p->aline; 
        atext += p->atext; 
    }
    if (p->f_solid && f_solid && p->solid.is_set) solid = p->solid;
    if (p->f_side != ESideType::NO && p->side.is_set && IsValidSideValue(f_side, p->side.value)) side = p->side;
    if (p->f_vspacing && f_vspacing && p->vspacing.is_set) vspacing = p->vspacing;
    if (p->f_indicator && f_indicator && p->indicator.is_set) indicator = p->indicator;
    if (p->f_makeroom && f_makeroom && p->makeroom.is_set) makeroom = p->makeroom;
    if (p->f_note && f_note) note += p->note;
    if (p->f_lost && f_lost) {
        lost_line += p->lost_line;
        lost_arrow += p->lost_arrow;
        lost_text += p->lost_text;
    }
    if (p->f_lsym && f_lsym) {
        lsym_line += p->lsym_line;
        if (p->lsym_size.is_set) lsym_size = p->lsym_size;
    }
    if (p->f_tag && f_tag) {
        tag_line += p->tag_line;
        tag_fill += p->tag_fill;
        tag_text += p->tag_text;
    }
    return *this;
}

/** Possible values for the 'side' attribute.*/
template<> const char EnumEncapsulator<ESide>::names[][ENUM_STRING_LEN] =
    {"invalid", "left", "right", "end", ""};
/** Possible values for the 'side' attribute.*/
template<> const char * const EnumEncapsulator<ESide>::descriptions[] =
    {nullptr, "Read from left/place on left side.", "Read from right/place on right side.", 
    "Read from bottom/place to document end.", ""};


/** Apply an attribute to us.
 * Generate an error if the we recognize the attribute, but bad value.
 * Do not recognize attribute names that correspond to attributes we do not contain.*/
bool MscStyle::AddAttribute(const Attribute &a, Chart *c)
{
    MscChart * chart = dynamic_cast<MscChart*>(c);
    _ASSERT(chart);
    if (a.type == EAttrType::STYLE) {
        const char *newname = a.name == "emphasis"?"box":"emptybox";
        if (a.name == "emphasis" || a.name == "emptyemphasis") {
            chart->Error.Warning(a, false, "Stylename '" + a.name + "' is deprecated, using " + newname + " instead.");
            operator +=(chart->MyCurrentContext().styles[newname].read());
            return true;
        }
        if (chart->SkipContent()) return true; //if we parse a procedure silently accept any style
        if (chart->MyCurrentContext().styles.find(a.name) == chart->MyCurrentContext().styles.end()) {
            a.InvalidStyleError(chart->Error);
            return true;
        }
        operator +=(chart->MyCurrentContext().styles[a.name].read());
        return true;
    }
    if (a.Is("color") && color_meaning==EColorMeaning::LINE_VLINE_TEXT) {
        unsigned num = 0;
        if (f_line)
            num += line.AddAttribute(a, chart, type);
        if (f_text)
            num += text.AddAttribute(a, chart, type);
        if (f_vline)
            num += vline.AddAttribute(a, chart, type);
        return num;
    }
    if (f_text) {
        if (a.StartsWith("text") || a.Is("ident"))
            return text.AddAttribute(a, chart, type);
        if (a.Is("textcolor") || a.Is("textcolour")) {
            chart->Error.WarnMscgenAttr(a, false, "text.color");
            return text.AddAttribute(Attribute("text.color", a), chart, type);
        }
        if (a.Is("textbgcolor") || a.Is("textbgcolour")) {
            chart->Error.WarnMscgenAttr(a, false, "text.bgcolor");
            return text.AddAttribute(Attribute("text.bgcolor", a), chart, type);
        }
    }
    if (f_line) {
        if (a.StartsWith("line"))
            return line.AddAttribute(a, chart, type);
        if (a.Is("linecolor") || a.Is("linecolour")) {
            chart->Error.WarnMscgenAttr(a, false, "line.color");
            return line.AddAttribute(Attribute("line.color", a), chart, type);
        }
    }
    if (f_vline && a.StartsWith("vline"))
        return vline.AddAttribute(a, chart, type);
    if (f_vfill && a.StartsWith("vfill"))
        return vfill.AddAttribute(a, chart, type);
    if (f_alt) {
        if (a.StartsWith("aline"))
            return aline.AddAttribute(a, chart, type);
        if (a.StartsWith("atext"))
            return atext.AddAttribute(a, chart, type);
        if (a.Is("arclinecolor") || a.Is("arclinecolour")) {
            chart->Error.WarnMscgenAttr(a, false, "aline.color");
            line.AddAttribute(Attribute("aline.color", a), chart, type);
        }
        if (a.Is("arctextcolor") || a.Is("arctextcolour")) {
            chart->Error.WarnMscgenAttr(a, false, "atext.color");
            text.AddAttribute(Attribute("atext.color", a), chart, type);
        }
        if (a.Is("arctextbgcolor") || a.Is("arctextbgcolour")) {
            chart->Error.WarnMscgenAttr(a, false, "atext.bgcolor");
            text.AddAttribute(Attribute("atext.bgcolor", a), chart, type);
        }
    }
    if (f_note && a.StartsWith("note"))
        return note.AddAttribute(a, chart, type);
    if (a.Is("solid") && f_solid) {
        if (a.type == EAttrType::CLEAR) {
            if (a.EnsureNotClear(chart->Error, type))
                solid.is_set = false;
            return true;
        }
        if (a.type == EAttrType::NUMBER && a.number>=0 && a.number <= 255) {
            solid.is_set = true;
            solid.value = a.number<=1 ? (unsigned char)(a.number*255) : (unsigned char)a.number;
            return true;
        }
        a.InvalidValueError("0..1' or '0..255", chart->Error);
    }
    if (a.Is("side") && f_side != ESideType::NO) {
        if (a.type == EAttrType::CLEAR) {
            if (a.EnsureNotClear(chart->Error, type))
                side.is_set = false;
            return true;
        }
        ESide tmp_side;
        if (a.type == EAttrType::STRING && Convert(a.value, tmp_side)) 
            if (IsValidSideValue(f_side, tmp_side)) {
                side.is_set = true;
                side.value = tmp_side;
                return true;
            }
        a.InvalidValueError(f_side == ESideType::LEFT_RIGHT ? "left/right" : "left/right/end", chart->Error);
        return true;
    }
    if (a.Is("compress") && f_vspacing) {
        if (a.type == EAttrType::CLEAR) {
            if (a.EnsureNotClear(chart->Error, type))
                vspacing.is_set = false;
            return true;
        }
        if (!a.CheckType(EAttrType::BOOL, chart->Error)) return true;
        vspacing.is_set = true;
        vspacing.value = a.yes ? DBL_MIN : 0;
        return true;
    }
    if (a.Is("vspacing") && f_vspacing) {
        if (a.type == EAttrType::CLEAR) {
            if (a.EnsureNotClear(chart->Error, type))
                vspacing.is_set = false;
        } else if (a.type == EAttrType::STRING && a.value == "compress") {
            vspacing.is_set = true;
            vspacing.value = DBL_MIN;
        } else if (a.CheckType(EAttrType::NUMBER, chart->Error)) {
            vspacing.is_set = true;
            vspacing.value = a.number;
        }
        return true;
    }
    if (a.Is("indicator") && f_indicator) {
        if (a.type == EAttrType::CLEAR) {
            if (a.EnsureNotClear(chart->Error, type))
                indicator.is_set = false;
            return true;
        }
        if (!a.CheckType(EAttrType::BOOL, chart->Error)) return true;
        indicator.is_set = true;
        indicator.value = a.yes;
        return true;
    }
    if (a.Is("makeroom") && f_makeroom) {
        if (a.type == EAttrType::CLEAR) {
            if (a.EnsureNotClear(chart->Error, type))
                makeroom.is_set = false;
            return true;
        }
        if (!a.CheckType(EAttrType::BOOL, chart->Error)) return true;
        makeroom.is_set = true;
        makeroom.value = a.yes;
        return true;
    }
    if (f_lost) {
        if (a.StartsWith("lost.line"))
            return lost_line.AddAttribute(a, chart, type);
        if (a.StartsWith("lost.arrow"))
            return lost_arrow.AddAttribute(a, chart, type);
        if (a.StartsWith("lost.text"))
            return lost_text.AddAttribute(a, chart, type);
    }
    if (f_lsym) {
        if (a.StartsWith("x.line"))
            return lsym_line.AddAttribute(a, chart, type);
        if (a.Is("x.size")) {
            if (a.type == EAttrType::STRING &&
                Convert(a.value, lsym_size.value)) {
                lsym_size.is_set = true;
                return true;
            }
            a.InvalidValueError(CandidatesFor(lsym_size.value), chart->Error);
            return true;
        }
    }
    if (f_tag && a.StartsWith("tag.line"))
        return tag_line.AddAttribute(a, chart, type);
    if (f_tag && a.StartsWith("tag.fill"))
        return tag_fill.AddAttribute(a, chart, type);
    if (f_tag && a.StartsWith("tag.text"))
        return tag_text.AddAttribute(a, chart, type);
    return SimpleStyleWithArrow<MscArrowHeads>::AddAttribute(a, chart);
}

bool MscStyle::DoIAcceptUnqualifiedColorAttr() const
{
    if (color_meaning==EColorMeaning::LINE_VLINE_TEXT && f_vline)
        return true;
    return SimpleStyleWithArrow<MscArrowHeads>::DoIAcceptUnqualifiedColorAttr();
}


/** Add the attribute names we take to `csh`.*/
void MscStyle::AttributeNames(Csh &csh) const
{
    if (DoIAcceptUnqualifiedColorAttr())
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"color",
          "Set the color of the element.", 
          EHintType::ATTR_NAME));
    if (f_alt) {
        LineAttr::AttributeNames(csh, "aline.");
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"aline.*",
            "Use these attributes to influence the style of arrows starting from this entity.",
            EHintType::ATTR_NAME));
        StringFormat::AttributeNames(csh, "atext.");
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"atext.*",
            "Use these attributes to influence the style of arrows starting from this entity.",
            EHintType::ATTR_NAME));
    }
    if (f_solid)
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"solid",
        "Specifies, how much the pipe covers its content. Use a value between [0..255] or [0..1], where 0 is fully transparent.",
        EHintType::ATTR_NAME));
    if (f_side != ESideType::NO)
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"side", 
        "Specify which side the element ends up.", 
        EHintType::ATTR_NAME));
    if (f_indicator) csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"indicator", 
        "Enable or disable indicators for this element (in case of a collapsed group entity or box).",
        EHintType::ATTR_NAME));
    if (f_makeroom) csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"makeroom", 
        "Specify if this vertical makes more horizontal space to avoid overlap.",
        EHintType::ATTR_NAME));
    if (f_vspacing) {
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"compress", 
            "Turning this on will push this element upwards until it bumps into the one above it in order to compress the chart vertically.",
            EHintType::ATTR_NAME));
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"vspacing", 
            "Specify the vertical spacing above this element.",
            EHintType::ATTR_NAME));
    }
    if (f_vline) {
        LineAttr::AttributeNames(csh, "vline.");
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"vline.*",
            "Set how the entity lines look like for this entity (or around this element).",
            EHintType::ATTR_NAME));
    }
    if (f_vfill) {
        FillAttr::AttributeNames(csh, "vfill.");
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"vfill.*",
            "Set how to fill the activated entity lines for this entity (or around this element).",
            EHintType::ATTR_NAME));
    }
    if (f_note) NoteAttr::AttributeNames(csh);
    if (f_lost) {
        LineAttr::AttributeNames(csh, "lost.line.");
        MscArrowHeads::AttributeNames(csh, "lost.arrow.");
        StringFormat::AttributeNames(csh, "lost.text.");
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"lost.line.*",
            "Options for how the line of the lost part of a lost message appears.",
            EHintType::ATTR_NAME));
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"lost.arrow.*",
            "Options for how the arrowhead of the lost part of a lost message appears.",
            EHintType::ATTR_NAME));
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"lost.text.*",
            "Options for how the label at the lost part of a lost message appears.",
            EHintType::ATTR_NAME));
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"lost.*",
            "Options for how the lost part of the lost message appears.",
            EHintType::ATTR_NAME));
    }
    if (f_lsym) {
        LineAttr::AttributeNames(csh, "x.line.");
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"x.size", 
            "Specify the size of the loss symbol.", 
            EHintType::ATTR_NAME));
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"x.*",
            "Options for how the loss symbol appears.",
            EHintType::ATTR_NAME));
    }
    if (f_tag) {
        LineAttr::AttributeNames(csh, "tag.line.");
        FillAttr::AttributeNames(csh, "tag.fill.");
        StringFormat::AttributeNames(csh, "tag.text.");
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"tag.line.*",
            "Set the line style of the tag of the box.",
            EHintType::ATTR_NAME));
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"tag.fill.*",
            "Set the fill style of the tag of the box.",
            EHintType::ATTR_NAME));
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"tag.text.*",
            "Set the text format of the tag label.",
            EHintType::ATTR_NAME));
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRNAME)+"tag.*",
            "Options to control how box tags appear.",
            EHintType::ATTR_NAME));
    }
    SimpleStyleWithArrow<MscArrowHeads>::AttributeNames(csh);
}

/** Callback for drawing a symbol before side values in the hints popup list box.
 * @ingroup hintpopup_callbacks*/
bool CshHintGraphicCallbackForSide(Canvas *canvas, CshHintGraphicParam p, CshHintStore &)
{
    if (!canvas) return false;
    const ESide t = (ESide)(int)p;
    XY bounds(HINT_GRAPHIC_SIZE_X, HINT_GRAPHIC_SIZE_Y);
    if (t==ESide::END) bounds.SwapXY();
    std::vector<double> xPos(2); 
    xPos[0] = 0;
    xPos[1] = bounds.x*0.7;
    LineAttr eLine(ELineType::SOLID, ColorType(0,0,0), 1, ECornerType::NONE, 0);
    MscArrowHeads aa(EArcArrowType::BIGARROW);
    aa.SetLineColor(ColorType(0,32,192)); //blue-green
    aa.SetType(EArrowEnd::END, {true, EArrowType::SOLID});
    aa.SetType(EArrowEnd::START, {true, EArrowType::NONE});
    aa.SetSize(EArrowSize::SPOT);
    aa.CreateArrowHeads();
    ShadowAttr shadow;
    FillAttr fill(aa.GetLine().color.value.Lighter(0.7), t==ESide::RIGHT ? EGradientType::DOWN : EGradientType::UP);
    std::vector<double> active(2,0.);
    canvas->Transform_Rotate(bounds/2, t==ESide::LEFT ? 0 : t==ESide::RIGHT ? M_PI : M_PI/2);
    canvas->Clip(XY(bounds.x*0.1,1), XY(bounds.x-1, bounds.y-1));
    aa.BigCalculateAndDraw(xPos, active, bounds.y*0.3, bounds.y*0.7, 
                           true, false, &aa.GetLine(), nullptr, fill, shadow, *canvas);
    canvas->UnClip();
    canvas->UnTransform();
    return true;
}

/** Add a list of possible attribute value names to `csh` for attribute `attr`.*/
bool MscStyle::AttributeValues(std::string_view attr, Csh &csh) const
{
    if (CaseInsensitiveBeginsWith(attr, "vline") && f_vline)
        return vline.AttributeValues(attr, csh);
    if (CaseInsensitiveBeginsWith(attr, "vfill") && f_vfill)
        return vfill.AttributeValues(attr, csh);
    if (f_alt) {
        if (CaseInsensitiveBeginsWith(attr, "afill"))
            return fill.AttributeValues(attr, csh);
        if (CaseInsensitiveBeginsWith(attr, "atext"))
            StringFormat::AttributeValues(attr, csh);
    }
    if (CaseInsensitiveEqual(attr, "solid") && f_solid) {
        csh.AddToHints(CshHint(csh.HintPrefixNonSelectable() + "<number: \b0.0..1.0\b>", 
            "Specify the opacity as a floating point number. 0 is fully transparent, 1 is fully opaque.",
            EHintType::ATTR_VALUE, false));
        csh.AddToHints(CshHint(csh.HintPrefixNonSelectable() + "<number: \b0..255\b>", 
            "Specify the opacity as an integer. 0 is fully transparent, 255 is fully opaque.",
            EHintType::ATTR_VALUE, false));
        return true;
    }
    if ((CaseInsensitiveEqual(attr, "compress") && f_vspacing) ||
        (CaseInsensitiveEqual(attr, "indicator") && f_indicator) ||
        (CaseInsensitiveEqual(attr, "makeroom") && f_makeroom)) {
        csh.AddYesNoToHints();
        return true;
    }
    if (CaseInsensitiveEqual(attr, "vspacing") && f_vspacing) {
        csh.AddToHints(CshHint(csh.HintPrefixNonSelectable() + "<number>", 
            "Specify extra spading above this element in pixels. 0 means no extra space.",
            EHintType::ATTR_VALUE, false));
        csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRVALUE)+"compress", 
            "Specifying 'compress' will auto-adjust vertical spacing to be as little as possible by moving the element up until it bumps into the ones above.",
            EHintType::ATTR_VALUE, true));
        return true;
    }
    if (CaseInsensitiveEndsWith(attr, "side")) {
        for (auto s = ESide::LEFT; s<=ESide::END; s = ESide(int(s)+1))
            if (IsValidSideValue(f_side, s))
                csh.AddToHints(CshHint(csh.HintPrefix(COLOR_ATTRVALUE)+EnumEncapsulator<ESide>::names[unsigned(s)], 
                                       EnumEncapsulator<ESide>::descriptions[unsigned(s)],
                                       EHintType::ATTR_VALUE, true, CshHintGraphicCallbackForSide, 
                                       CshHintGraphicParam(s)));
        return true;
    }
    if (f_note && CaseInsensitiveBeginsWith(attr, "note"))
        return note.AttributeValues(attr, csh);
    if (f_lost) {
        if (CaseInsensitiveEqual(attr, "lost.line"))
            return line.AttributeValues(attr, csh);
        if (CaseInsensitiveEqual(attr, "lost.arrow"))
            return arrow.AttributeValues(attr, csh, EArcArrowType::ARROW);
        if (CaseInsensitiveEqual(attr, "lost.text"))
            StringFormat::AttributeValues(attr, csh);
    }
    if (f_lsym) {
        if (CaseInsensitiveEqual(attr, "x.line"))
            return line.AttributeValues(attr, csh);
        if (CaseInsensitiveEqual(attr, "x.size")) {
            arrow.AttributeValues(attr, csh, EArcArrowType::ANY);
            return true;
        }
    }
    if (f_tag) {
        if (CaseInsensitiveBeginsWith(attr, "tag_text"))
            return tag_text.AttributeValues(attr, csh);
        if (CaseInsensitiveBeginsWith(attr, "tag.line"))
            return tag_line.AttributeValues(attr, csh);
        if (CaseInsensitiveBeginsWith(attr, "tag.fill"))
            return tag_fill.AttributeValues(attr, csh);
    }
    return SimpleStyleWithArrow<MscArrowHeads>::AttributeValues(attr, csh);
}

//////////////////////////////////////////////////////////////////////

void msc::MscContext::ApplyContextContent(const MscContext & o)
{
    ContextBase::ApplyContextContent(o);
    if (o.is_full) {
        hscale = o.hscale;
        numbering = o.numbering;
        vspacing = o.vspacing;
        indicator = o.indicator;
        slant_angle = o.slant_angle;
        slant_depth = o.slant_depth;
        auto_heading = o.auto_heading;
        defCommentLine = o.defCommentLine;
        defCommentFill = o.defCommentFill;
        defBackground = o.defBackground;
    } else {
        if (o.hscale.is_set) hscale = o.hscale;
        if (o.numbering.is_set) numbering = o.numbering;
        if (o.vspacing.is_set) vspacing = o.vspacing;
        if (o.indicator.is_set) indicator = o.indicator;
        if (o.slant_angle.is_set) slant_angle = o.slant_angle;
        if (o.slant_depth.is_set) slant_depth = o.slant_depth;
        if (o.auto_heading.is_set) auto_heading = o.auto_heading;
        defCommentLine += o.defCommentLine;
        defCommentFill += o.defCommentFill;
        defBackground += o.defBackground;
    }
}

void msc::MscContext::ApplyContextContent(MscContext && o)
{
    if (o.is_full) {
        hscale = o.hscale;
        numbering = o.numbering;
        vspacing = o.vspacing;
        indicator = o.indicator;
        slant_angle = o.slant_angle;
        slant_depth = o.slant_depth;
        auto_heading = o.auto_heading;
        defCommentLine = o.defCommentLine;
        defCommentFill = o.defCommentFill;
        defBackground = o.defBackground;
    } else {
        if (o.hscale.is_set) hscale = o.hscale;
        if (o.numbering.is_set) numbering = o.numbering;
        if (o.vspacing.is_set) vspacing = o.vspacing;
        if (o.indicator.is_set) indicator = o.indicator;
        if (o.slant_angle.is_set) slant_angle = o.slant_angle;
        if (o.slant_depth.is_set) slant_depth = o.slant_depth;
        if (o.auto_heading.is_set) auto_heading = o.auto_heading;
        defCommentLine += o.defCommentLine;
        defCommentFill += o.defCommentFill;
        defBackground += o.defBackground;
    }
    ContextBase::ApplyContextContent(std::move(o));
}

void MscContext::Empty()
{
    ContextBase::Empty();
    Procedures.clear();
    hscale.is_set = false;
    numbering.is_set = false;
    vspacing.is_set = false;
    indicator.is_set = false;
    slant_angle.is_set = false;
    slant_depth.is_set = false;
    auto_heading.is_set = false;
    defCommentLine.Empty();
    defCommentFill.Empty();
    defBackground.Empty();

    //Now add default styles, but all empty
    
    styles["arrow"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::LINE_ARROW_TEXT, EArcArrowType::ARROW,
                 false, true, true, false, false, false, 
                 false, true, true, ESideType::LEFT_RIGHT, true, false, false, false, 
                 true, true, false, false); 
                 //no alt fill, shadow, vline solid vfill, makeroom, note, shape, tag
    styles["arrow_self"] = styles["arrow"];
    styles["->"] = styles["arrow"]; 
    styles["->"].write().type = EStyleType::DEF_ADD;
    styles[">"]  = styles["->"];
    styles[">>"] = styles["->"];
    styles["=>"] = styles["->"];
    styles["=>>"] = styles["->"];
    styles["==>"] = styles["->"];

    styles["blockarrow"] =
        MscStyle(EStyleType::DEFAULT, EColorMeaning::FILL, EArcArrowType::BIGARROW,
                 false, true, true, true, true, false,
                 false, true, true, ESideType::NO, false, false, false, false, 
                 false, false, false, false);
                 //no alt vline solid side indicator vfill makeroom note loss lsym shape, tag
    styles["box_collapsed_arrow"] = styles["blockarrow"];
    styles["block->"] = styles["blockarrow"];
    styles["block->"].write().type = EStyleType::DEF_ADD;
    styles["block>"] = styles["block->"];
    styles["block>>"] = styles["block->"];
    styles["block=>"] = styles["block->"];
    styles["block=>>"] = styles["block->"];
    styles["block==>"] = styles["block->"];

    styles["vertical"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::FILL, EArcArrowType::BIGARROW,
                 false, true, true, true, true, false,
                 false, true, true, ESideType::ANY, false, false, true, false, 
                 false, false, false, false);
                 //no alt vline solid indicator vfill note loss lsym shape tag
    styles["vertical_range"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::LINE_ARROW_TEXT, EArcArrowType::ARROW,
                 false, true, true, true, true, false,
                 false, true, true, ESideType::ANY, false, false, true, false, 
                 false, false, false, false);
                 //no alt vline solid indicator vfill note loss lsym shape tag
    styles["vertical_pointer"] =
        MscStyle(EStyleType::DEFAULT, EColorMeaning::LINE_ARROW_TEXT, EArcArrowType::ARROW,
                 false, true, true, true, true, false,
                 false, true, true, ESideType::ANY, false, false, true, false, 
                 true, true, false, false);
                 //no alt vline solid indicator vfill note shape tag
    styles["vertical_bracket"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::LINE_ARROW_TEXT, EArcArrowType::NONE,
                 false, true, true, true, true, false,
                 false, true, true, ESideType::ANY, false, false, true, false, 
                 false, false, false, false);
                 //no alt arrow vline solid indicator vfill note loss lsym shape tag
    styles["vertical_brace"] = styles["vertical_bracket"];

    styles["vertical->"] =
        MscStyle(EStyleType::DEF_ADD, EColorMeaning::NOHOW, EArcArrowType::ANY,
                false, true, true, true, true, false,
                false, true, true, ESideType::ANY, false, false, true, false,
                true, true, false, false);
                //no alt vline solid indicator vfill note  shape tag
    styles["vertical>"] = styles["vertical->"];
    styles["vertical>>"] = styles["vertical->"];
    styles["vertical=>"] = styles["vertical->"];
    styles["vertical=>>"] = styles["vertical->"];
    styles["vertical==>"] = styles["vertical->"];
    styles["vertical--"] = styles["vertical->"];
    styles["vertical++"] = styles["vertical->"];
    styles["vertical.."] = styles["vertical->"];
    styles["vertical=="] = styles["vertical->"];

    styles["divider"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::LINE_ARROW_TEXT, EArcArrowType::NONE,
                 false, true, true, false, false, true,
                 false, true, true, ESideType::NO, false, false, false, false, 
                 false, false, false, false);
                 //no alt arrow, fill, shadow solid side indicator vfill makeroom note loss lsym shape tag
    styles["---"] = styles["divider"];
    styles["---"].write().type = EStyleType::DEF_ADD;
    styles["..."] = styles["---"];

    styles["box"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::FILL, EArcArrowType::NONE,
                 false, true, true, true, true, false,
                 false, true, true, ESideType::NO, true, false, false, false, 
                 false, false, false, true);
                 //no alt arrow, vline solid side vfill makeroom note loss lsym shape
    styles["box_collapsed"] = styles["box"];
    styles["emptybox"] = styles["box"];
    styles["--"] = styles["box"];
    styles["--"].write().type = EStyleType::DEF_ADD;
    styles["++"] = styles["--"];
    styles[".."] = styles["--"];
    styles["=="] = styles["--"];

    styles["pipe"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::FILL, EArcArrowType::NONE,
                 false, true, true, true, true, false,
                 true, true, true, ESideType::LEFT_RIGHT, false, false, false, false, 
                 false, false, false, false);
                 //no alt arrow, vline indicator vfill makeroom note loss lsym shape tag
    styles["pipe--"] = styles["pipe"];
    styles["pipe--"].write().type = EStyleType::DEF_ADD;
    styles["pipe++"] = styles["pipe--"];
    styles["pipe.."] = styles["pipe--"];
    styles["pipe=="] = styles["pipe--"];

    styles["entity"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::LINE_VLINE_TEXT, EArcArrowType::ARROW,
                 true, true, true, true, true, true,
                 false, false, false, ESideType::NO, true, true, false, false, 
                 false, false, true, false);
                 //no solid numbering compress side makeroom note loss lsym tag
    styles["entitygroup_collapsed"] = styles["entity"];
    styles["entitygroup"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::LINE_VLINE_TEXT, EArcArrowType::NONE,  //no point in using a different color meaning, as attributes are always added to "entity"
                 false, true, true, true, true, false,
                 false, false, false, ESideType::NO, true, false, false, false, 
                 false, false, false, false);
                 //only line, fill, text and shadow
    styles["entitygroup_large"] = styles["entitygroup"];

    styles["indicator"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::LINE_ARROW_TEXT, EArcArrowType::NONE,
                 false, false, true, true, true, false,
                 false, false, false, ESideType::NO, false, false, false, false, 
                 false, false, false, false);
                 //fill line shadow only
    styles["symbol"] =
        MscStyle(EStyleType::DEFAULT, EColorMeaning::NOHOW, EArcArrowType::NONE,
                 false, true, true, true, true, false,
                 false, false, false, ESideType::NO, false, false, true, false, 
                 false, false, true, false);
                 //only line fill shadow text makeroom shape
    styles["text"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::TEXT, EArcArrowType::NONE,
                 false, true, true, true, true, false,
                 false, false, false, ESideType::NO, false, false, true, false, 
                 false, false, false, false);
                 //only line fill shadow text makeroom; color is for text


    styles["note"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::NOHOW, EArcArrowType::ANY,
                 false, true, true, true, true, false, 
                 false, true, false, ESideType::NO, false, false, false, true, 
                 false, false, false, false);
                 //no alt vline side solid indicator compress vfill makreoom loss lsym shape tag
    
    styles["comment"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::LINE_ARROW_TEXT, EArcArrowType::NONE,
                 false, true, false, false, false, false,
                 false, true, false, ESideType::ANY, false, false, false, false, 
                 false, false, false, false);
                 //only text numbering and side
    styles["endnote"] = styles["comment"];
    //styles["footnote"] = styles["comment"];

    styles["title"] = 
        MscStyle(EStyleType::DEFAULT, EColorMeaning::TEXT, EArcArrowType::NONE,
                 false, true, true, true, true, true,
                 false, false, false, ESideType::NO, false, false, false, false, 
                 false, false, false, false);
                 //line, fill, shadow, vline text  
    styles["subtitle"] = styles["title"];
}

void MscContext::Plain()
{
    Empty();
    ContextBase::Plain();
    hscale.is_set = true;
    hscale.value = 1.0;
    numbering.is_set = true;
    numbering.value=false;
    vspacing.is_set = true;
    vspacing.value = 0;
    indicator.is_set = true;
    indicator.value = true;
    slant_angle.is_set = true;
    slant_angle.value = 0;
    slant_depth.is_set = true;
    slant_depth.value = 0;
    auto_heading.is_set = true;
    auto_heading.value = false;
    defCommentLine.MakeComplete();
    defCommentFill.MakeComplete();
    defBackground.MakeComplete();
    defCommentLine.width.value = 3;
    defCommentFill.color.value.a = 0; //fully transparent
    defBackground.color = ColorType::white();

    MscStyle *p;
    const ColorType faint(255, 255, 255, 224, ColorType::OVERLAY);

    p = &styles["arrow"].write();
    p->MakeCompleteButText();
    p->vspacing.is_set = false;
    p->numbering.is_set = false;
    p->line.radius.value = 10;
    p->line.corner.value = ECornerType::ROUND;
    p->lost_line.color = {true, faint};
    p->lost_arrow.write().SetLineColor(faint);
    //p->lost_text.SetColor(faint);  do not change label in the lost part

    styles["arrow_self"].write().line.radius.is_set = true;
    styles["arrow_self"].write().line.radius.value = -1;
    styles["arrow_self"].write().side.is_set = true;
    styles["arrow_self"].write().side.value = ESide::RIGHT;

    styles["->"].write().line.type.is_set = true;
    styles["->"].write().line.type.value = ELineType::SOLID;
    styles[">"].write().line.type.is_set = true;
    styles[">"].write().line.type.value = ELineType::DOTTED;
    styles[">>"].write().line.type.is_set = true;
    styles[">>"].write().line.type.value = ELineType::DASHED;
    styles["=>"].write().line.type.is_set = true;
    styles["=>"].write().line.type.value = ELineType::DOUBLE;
    styles["=>>"].write().line.type.is_set = true;
    styles["=>>"].write().line.type.value = ELineType::DOUBLE;
    styles["=>>"].write().arrow.write().SetType(EArrowEnd::END, {true, EArrowType::SHARP});
    styles["=>>"].write().arrow.write().SetType(EArrowEnd::MIDDLE, {true, EArrowType::SHARP});
    styles["==>"].write().line.type.is_set = true;
    styles["==>"].write().line.type.value = ELineType::DOUBLE;

    p = &styles["blockarrow"].write();
    p->MakeCompleteButText();
    p->vspacing.is_set = false;
    p->numbering.is_set = false;
    p->line.radius.value = -1;
    styles["box_collapsed_arrow"] = *p;
    styles["box_collapsed_arrow"].write().arrow.write().SetType(EArrowEnd::MIDDLE, {true, EArrowType::DOT});

    styles["block->"].write().line.type.is_set = true;
    styles["block->"].write().line.type.value = ELineType::SOLID;
    styles["block>"].write().line.type.is_set = true;
    styles["block>"].write().line.type.value = ELineType::DOTTED;
    styles["block>>"].write().line.type.is_set = true;
    styles["block>>"].write().line.type.value = ELineType::DASHED;
    styles["block=>"].write().line.type.is_set = true;
    styles["block=>"].write().line.type.value = ELineType::DOUBLE;
    styles["block=>>"].write().line.type.is_set = true;
    styles["block=>>"].write().line.type.value = ELineType::DOUBLE;
    styles["block=>>"].write().arrow.write().SetType(EArrowEnd::END, {true, EArrowType::SHARP});
    styles["block=>>"].write().arrow.write().SetType(EArrowEnd::MIDDLE, {true, EArrowType::SHARP});
    styles["block==>"].write().line.type.is_set = true;
    styles["block==>"].write().line.type.value = ELineType::DOUBLE;

    p = &styles["vertical"].write();
    p->MakeCompleteButText();
    p->vspacing.is_set = false;
    p->numbering.is_set = false;
    p->makeroom.value = true;
    p->side.is_set = false;
    p->line.radius.value = -1;
    p->text.UnsetWordWrap();
    //We need this otherwise setting the global text style to wrap
    //would cause warnings for all horizontally set verticals which
    //have no text.width attribute. Better if the user sets this explicitly.

    p = &styles["vertical_range"].write();
    p->MakeCompleteButText();
    p->vspacing.is_set = false;
    p->numbering.is_set = false;
    p->makeroom.value = true;
    p->side.is_set = false;
    p->line.radius.value = 8;
    p->text.UnsetWordWrap();

    p = &styles["vertical_bracket"].write();
    p->MakeCompleteButText();
    p->vspacing.is_set = false;
    p->numbering.is_set = false;
    p->makeroom.value = true;
    p->side.is_set = false;
    p->line.radius.value = 8;
    p->line.corner.value = ECornerType::NONE;
    p->text.UnsetWordWrap();
    
    p = &styles["vertical_pointer"].write();
    p->MakeCompleteButText();
    p->vspacing.is_set = false;
    p->numbering.is_set = false;
    p->makeroom.value = true;
    p->side.is_set = false;
    p->line.radius.value = 8;
    p->line.corner.value = ECornerType::NONE;
    p->text.UnsetWordWrap();
    p->lost_line.color = {true, faint};
    p->lost_arrow.write().SetLineColor(faint);
    //p->lost_text.SetColor(faint); do not change label in lost part

    p = &styles["vertical_brace"].write();
    p->MakeCompleteButText();
    p->vspacing.is_set = false;
    p->numbering.is_set = false;
    p->makeroom.value = true;
    p->side.is_set = false;
    p->line.radius.value = 8;
    p->line.corner.value = ECornerType::ROUND;
    p->text.UnsetWordWrap();

    styles["vertical->"].write().line.type.is_set = true;
    styles["vertical->"].write().line.type.value = ELineType::SOLID;
    styles["vertical>"].write().line.type.is_set = true; 
    styles["vertical>"].write().line.type.value = ELineType::DOTTED;
    styles["vertical>>"].write().line.type.is_set = true;
    styles["vertical>>"].write().line.type.value = ELineType::DASHED;
    styles["vertical=>"].write().line.type.is_set = true;
    styles["vertical=>"].write().line.type.value = ELineType::DOUBLE;
    styles["vertical=>>"].write().line.type.is_set = true;
    styles["vertical=>>"].write().line.type.value = ELineType::DOUBLE;
    styles["vertical=>>"].write().arrow.write().SetType(EArrowEnd::END, {true, EArrowType::SHARP});
    styles["vertical=>>"].write().arrow.write().SetType(EArrowEnd::MIDDLE, {true, EArrowType::SHARP});
    styles["vertical==>"].write().line.type.is_set = true;
    styles["vertical==>"].write().line.type.value = ELineType::DOUBLE;

    styles["vertical--"].write().arrow.write().SetType(EArrowEnd::START, {true, EArrowType::NONE});
    styles["vertical--"].write().arrow.write().SetType(EArrowEnd::MIDDLE, {true, EArrowType::NONE});
    styles["vertical--"].write().arrow.write().SetType(EArrowEnd::END, {true, EArrowType::NONE});
    styles["vertical--"].write().line.type.is_set = true;
    styles["vertical--"].write().line.type.value = ELineType::SOLID;
    styles["vertical++"] = styles["vertical--"];
    styles["vertical++"].write().line.type.value = ELineType::DASHED;
    styles["vertical.."] = styles["vertical--"];
    styles["vertical.."].write().line.type.value = ELineType::DOTTED;
    styles["vertical=="] = styles["vertical--"];
    styles["vertical=="].write().line.type.value = ELineType::DOUBLE;

    p = &styles["divider"].write();
    p->MakeCompleteButText();
    p->vspacing.is_set = false;
    p->numbering.is_set = false;
    p->vline.Empty();
    p->line.type.value = ELineType::NONE;

    p = &styles["---"].write();
    p->line.type.is_set = true;
    p->line.type.value = ELineType::DOTTED;
    p = &styles["..."].write();
    p->vline.type.is_set = true;
    p->vline.type.value = ELineType::DOTTED;
    p->text.Apply("\\mu(10)\\md(10)");

    p = &styles["emptybox"].write();
    p->MakeCompleteButText();
    p->vspacing.is_set = false;
    p->numbering.is_set = false;
    p->tag_line.corner.value = ECornerType::BEVEL;
    p->tag_line.radius.value = 10;
    p->tag_text.Apply("\\B\\-");
    styles["box_collapsed"] = styles["emptybox"];
    styles["box"] = styles["emptybox"];
    styles["box"].write().text.Apply("\\pl");
    styles["box"].write().line.type.is_set = false;

    styles["--"].write().line.type.is_set = true;
    styles["--"].write().line.type.value = ELineType::SOLID;
    styles["++"].write().line.type.is_set = true;
    styles["++"].write().line.type.value = ELineType::DASHED;
    styles[".."].write().line.type.is_set = true;
    styles[".."].write().line.type.value = ELineType::DOTTED;
    styles["=="].write().line.type.is_set = true;
    styles["=="].write().line.type.value = ELineType::DOUBLE;

    p = &styles["pipe"].write();
    p->MakeCompleteButText();
    p->vspacing.is_set = false;
    p->numbering.is_set = false;
    p->line.radius.value = 5;

    styles["pipe--"].write().line.type.is_set = true;
    styles["pipe--"].write().line.type.value = ELineType::SOLID;
    styles["pipe++"].write().line.type.is_set = true;
    styles["pipe++"].write().line.type.value = ELineType::DASHED;
    styles["pipe.."].write().line.type.is_set = true;
    styles["pipe.."].write().line.type.value = ELineType::DOTTED;
    styles["pipe=="].write().line.type.is_set = true;
    styles["pipe=="].write().line.type.value = ELineType::DOUBLE;

    styles["entity"].write().MakeCompleteButText();
    styles["entity"].write().arrow.Empty(); //we have to ensure this is empty. aline and atext is left empty by MakeCompleteButText()
    styles["entitygroup_collapsed"].write().MakeCompleteButText();
    styles["entitygroup_collapsed"].write().arrow.Empty(); 
    styles["entitygroup"].write().MakeCompleteButText();
    styles["entitygroup"].write().line.type.value = ELineType::DASHED;
    styles["entitygroup_large"].write().MakeCompleteButText();
    styles["entitygroup_large"].write().line.type.value = ELineType::NONE;
    styles["entitygroup_large"].write().fill.color.value = ColorType(192,192,192);

    styles["indicator"].write().MakeCompleteButText();
    styles["indicator"].write().line.width.value = 2;

    p = &styles["symbol"].write();
    p->MakeCompleteButText();
    p->makeroom.value = true;
    styles["text"] = styles["symbol"];
    styles["text"].write().line.type.value = ELineType::NONE;
    styles["text"].write().fill.color.value = ColorType(0, 0, 0, 0); //fully transparent
    styles["text"].write().text += "\\mn(10)\\ms(6)"; //small font to 6, normal to 10

    styles["note"].write().MakeCompleteButText();
    styles["note"].write().numbering.is_set = false;
    styles["note"].write().text += "\\mn(10)\\ms(6)"; //small font to 6, normal to 10

    styles["comment"].write().MakeCompleteButText();
    styles["comment"].write().numbering.is_set = false;
    styles["comment"].write().text += "\\mn(10)\\ms(6)\\pl"; //small font to 6, normal to 10, left para

    p = &styles["endnote"].write();
    p->MakeCompleteButText();
    p->numbering.is_set = false;
    p->text += "\\mn(10)\\ms(6)\\pl"; //small font to 6, normal to 10, left para
    p->side.value = ESide::END;

    //styles["footnote"].write().MakeCompleteButText();
    //styles["footnote"].write().numbering.is_set = false;
    //styles["footnote"].write().text += "\\mn(10)\\ms(6)\\pl"; //small font to 6, normal to 10, left para
    //styles["footnote"].write().side.value = ESide::END;

    p = &styles["title"].write();
    p->MakeCompleteButText();
    p->vline.type.value = ELineType::NONE;
    p->line.type.value = ELineType::NONE;
    p->fill.color.value = ColorType(0,0,0,0); //no fill
    p->text += "\\mn(28)\\ms(18)\\B";
    styles["subtitle"] = styles["title"];
    styles["subtitle"].write().text += "\\mn(22)\\ms(14)\\B";

    const OptAttr<ColorType> weak(true, ColorType(255, 255, 255, 128, ColorType::OVERLAY));
    styles["weak"].write().line.color = weak;
    styles["weak"].write().arrow.write().SetLineColor(weak.value);
    styles["weak"].write().text.SetColor(weak.value);
    styles["weak"].write().text.Apply("\\i");
    OptAttr<double> w2(true, 2.0);
    styles["strong"].write().line.width = w2;
    LineAttr l = styles["strong"].read().arrow.read().GetLine();
    l.width = w2;
    styles["strong"].write().arrow.write().SetLine(l);
    styles["strong"].write().text.Apply("\\b");
}

void MscContext::MscgenCompat()
{
    Empty();
    is_full = false;

    MscStyle *p = &styles["arrow"].write();
    p->lost_line.color.is_set = true;
    p->lost_line.color.value = ColorType(0, 0, 0, 0); //none
    p->lost_arrow.write().SetLineColor(ColorType(0, 0, 0, 0));
    p->lsym_line.width.is_set = true;
    p->lsym_line.width.value = 1;
    p->lsym_line.color.is_set = true;
    p->lsym_line.color.value = ColorType(0, 0, 0); //black

    styles["->"].write().line.type.is_set = true;
    styles["->"].write().line.type.value = ELineType::SOLID;
    styles["->"].write().arrow.write().SetType(EArrowEnd::END, {true, EArrowType::HALF});
    styles["->"].write().arrow.write().SetType(EArrowEnd::MIDDLE, {true, EArrowType::HALF});
    styles[">"].write().line.type.is_set = true;
    styles[">"].write().line.type.value = ELineType::DOTTED;
    styles[">"].write().arrow.write().SetType(EArrowEnd::END, {true, EArrowType::SOLID});
    styles[">"].write().arrow.write().SetType(EArrowEnd::MIDDLE, {true, EArrowType::SOLID});
    styles[">>"].write().line.type.is_set = true;
    styles[">>"].write().line.type.value = ELineType::DOTTED;
    styles[">>"].write().arrow.write().SetType(EArrowEnd::END, {true, EArrowType::SOLID});
    styles[">>"].write().arrow.write().SetType(EArrowEnd::MIDDLE, {true, EArrowType::SOLID});
    styles["=>"].write().line.type.is_set = true;
    styles["=>"].write().line.type.value = ELineType::SOLID;
    styles["=>"].write().arrow.write().SetType(EArrowEnd::END, {true, EArrowType::SOLID});
    styles["=>"].write().arrow.write().SetType(EArrowEnd::MIDDLE, {true, EArrowType::SOLID});
    styles["=>>"].write().line.type.is_set = true;
    styles["=>>"].write().line.type.value = ELineType::SOLID;
    styles["=>>"].write().arrow.write().SetType(EArrowEnd::END, {true, EArrowType::LINE});
    styles["=>>"].write().arrow.write().SetType(EArrowEnd::MIDDLE, {true, EArrowType::LINE});
    //==> is the same as in plain: double line, solid arrow
    styles[":>"].write().line.type.is_set = true;
    styles[":>"].write().line.type.value = ELineType::DOUBLE;
    styles[":>"].write().arrow.write().SetType(EArrowEnd::END, {true, EArrowType::SOLID});
    styles[":>"].write().arrow.write().SetType(EArrowEnd::MIDDLE, {true, EArrowType::SOLID});
}

