// Author(s): Jeroen Keiren, Wieger Wesselink
// Copyright: see the accompanying file COPYING or copy at
// https://svn.win.tue.nl/trac/MCRL2/browser/trunk/COPYING
//
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)
//
/// \file data_specification_test.cpp
/// \brief Basic regression test for data specifications.

#include <iostream>
#include <boost/range/iterator_range.hpp>
#include <boost/test/minimal.hpp>

#include "mcrl2/atermpp/convert.h"
#include "mcrl2/data/data_specification.h"
#include "mcrl2/data/basic_sort.h"
#include "mcrl2/data/find.h"
#include "mcrl2/data/data_expression.h"
#include "mcrl2/data/sort_expression.h"
#include "mcrl2/data/set.h"
#include "mcrl2/data/bag.h"
#include "mcrl2/data/list.h"
#include "mcrl2/data/structured_sort.h"
#include "mcrl2/data/utility.h"
#include "mcrl2/data/parse.h"
#include "mcrl2/core/garbage_collection.h"
#include "mcrl2/atermpp/aterm_init.h"

using namespace mcrl2;
using namespace mcrl2::data;

template <typename Container1, typename Container2>
bool equal_content(Container1 const& c1, Container2 const& c2)
{
  atermpp::set<typename Container1::value_type> s1(c1.begin(), c1.end());
  atermpp::set<typename Container2::value_type> s2(c2.begin(), c2.end());

  if (s1 != s2)
  {
    std::clog << "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++" << std::endl
                      << "Detailed comparison:" << std::endl;
    std::clog << data::pp(c1) << std::endl;
    std::clog << data::pp(c2) << std::endl;
  }
  return s1 == s2;
}

bool detailed_compare_for_equality(data_specification const& left, data_specification const& right)
{
  /*
  atermpp::set<alias> left_aliases = atermpp::convert(left.aliases());
  atermpp::set<alias> right_aliases = atermpp::convert(right.aliases());
  BOOST_CHECK(left_aliases == right_aliases); */

  atermpp::set<sort_expression> left_sorts(left.sorts().begin(), left.sorts().end());
  atermpp::set<sort_expression> right_sorts(right.sorts().begin(), right.sorts().end());
  BOOST_CHECK(left_sorts == right_sorts);

  atermpp::set<function_symbol> left_constructors(left.constructors().begin(), left.constructors().end());
  atermpp::set<function_symbol> right_constructors(right.constructors().begin(), right.constructors().end());
  BOOST_CHECK(left_constructors == right_constructors);

  atermpp::set<function_symbol> left_mappings(left.mappings().begin(), left.mappings().end());
  atermpp::set<function_symbol> right_mappings(right.mappings().begin(), right.mappings().end());
  BOOST_CHECK(left_mappings == right_mappings);

  atermpp::set<data_equation> left_equations(left.equations().begin(), left.equations().end());
  atermpp::set<data_equation> right_equations(right.equations().begin(), right.equations().end());
  BOOST_CHECK(left_equations == right_equations);

  if (/*(left_aliases != right_aliases)*/
       (left_sorts != right_sorts)
      || (left_constructors != right_constructors)
      || (left_mappings != right_mappings)
      || (left_equations != right_equations))
  {
    std::clog << "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++" << std::endl
                  << "Specification detailed comparison:" << std::endl;

  /*
    if (left_aliases != right_aliases)
    {
      std::clog << "Aliases (left)  " << data::pp(left.aliases()) << std::endl;
      std::clog << "Aliases (right)  " << data::pp(right.aliases()) << std::endl;
    }*/
    if (left_sorts != right_sorts)
    {
      std::clog << "Sorts (left)  " << data::pp(left.sorts()) << std::endl;
      std::clog << "Sorts (right) " << data::pp(right.sorts()) << std::endl;
    }
    if (left_constructors != right_constructors)
    {
      std::clog << "Constructors (left)  " << data::pp(left.constructors()) << std::endl;
      std::clog << "Constructors (right) " << data::pp(right.constructors()) << std::endl;
    }
    if (left_mappings != right_mappings)
    {
      std::clog << "Mappings (left)  " << data::pp(left.mappings()) << std::endl;
      std::clog << "Mappings (right) " << data::pp(right.mappings()) << std::endl;
    }
    if (left_equations != right_equations)
    {
      std::clog << "Equations (left)  " << data::pp(left.equations()) << std::endl;
      std::clog << "Equations (right) " << data::pp(right.equations()) << std::endl;
    }

    std::clog << "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++" << std::endl;

    return false;
  }
  return true;
}

bool compare_for_equality(data_specification const& left, data_specification const& right)
{
  return detailed_compare_for_equality(left, right);
  /*
  if (!(left == right))
  {
    BOOST_CHECK(left == right);

    std::clog << "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++" << std::endl
              << "Specification detailed comparison:" << std::endl;

    if (left.sorts() != right.sorts())
    {
      std::clog << "Sorts (left)  " << data::pp(left.sorts()) << std::endl;
      std::clog << "Sorts (right) " << data::pp(right.sorts()) << std::endl;
    }
    if (left.constructors() != right.constructors())
    {
      std::clog << "Constructors (left)  " << data::pp(left.constructors()) << std::endl;
      std::clog << "Constructors (right) " << data::pp(right.constructors()) << std::endl;
    }
    if (left.mappings() != right.mappings())
    {
      std::clog << "Mappings (left)  " << data::pp(left.mappings()) << std::endl;
      std::clog << "Mappings (right) " << data::pp(right.mappings()) << std::endl;
    }
    if (left.equations() != right.equations())
    {
      std::clog << "Equations (left)  " << data::pp(left.equations()) << std::endl;
      std::clog << "Equations (right) " << data::pp(right.equations()) << std::endl;
    }

    std::clog << "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++" << std::endl;

    return false;
  }

  return true;
  */
}

void test_sorts()
{
  std::clog << "test_sorts" << std::endl;

  basic_sort s("S");
  basic_sort s0("S0");
  alias s1(s,basic_sort("S1"));

  atermpp::vector< sort_expression > sl;
  sl.push_back(basic_sort("S1"));
  sl.push_back(s0);

  data_specification spec;
  spec.add_sort(s);
  spec.add_sort(s0);
  spec.add_alias(s1);
  data_specification spec1;
  spec1.add_alias(s1);
  spec1.add_sort(s0);
  spec1.add_sort(s);

  //BOOST_CHECK(equal_content(sl, spec.user_defined_sorts()));
  //BOOST_CHECK(equal_content(sl, spec1.user_defined_sorts()));
  BOOST_CHECK(compare_for_equality(spec, spec1));

  basic_sort s2("S2");
  sort_expression_vector s2l(atermpp::make_vector(reinterpret_cast<sort_expression&>(s2)));
  spec.add_context_sort(s2);
  spec1.add_context_sorts(s2l);
  BOOST_CHECK(compare_for_equality(spec, spec1));

  std::for_each(s2l.begin(), s2l.end(), boost::bind(&data_specification::remove_sort, &spec, _1));
  spec1.remove_sort(s2);
  compare_for_equality(spec, spec1);
}

void test_aliases()
{
  std::clog << "test_aliases" << std::endl;

  basic_sort s("S");
  basic_sort t("T");
  alias      s1(basic_sort("S1"), s);
  alias      s2(basic_sort("S2"), s);

  data_specification spec;

  // BOOST_CHECK(boost::distance(spec.aliases()) == 0);

  atermpp::set< sort_expression > sorts;
  sorts.insert(s);
  sorts.insert(t);
  std::for_each(sorts.begin(), sorts.end(), boost::bind(&data_specification::add_sort, &spec, _1));

  /* atermpp::set< sort_expression > aliases;
  aliases.insert(s1);
  aliases.insert(s2);
  spec.add_aliases(boost::make_iterator_range(aliases));

  BOOST_CHECK(boost::distance(spec.aliases(s)) == 2);
  BOOST_CHECK(boost::distance(spec.aliases(t)) == 0);
  BOOST_CHECK(spec.aliases(s) == boost::make_iterator_range(aliases)); */
}

void test_constructors()
{
  std::clog << "test_constructors" << std::endl;

  basic_sort s("S");
  basic_sort s0("S0");
  function_sort s0s(atermpp::make_vector(sort_expression(s0)),s);
  function_symbol f("f", s);
  function_symbol g("g", s0s);
  function_symbol h("h", s0);
  function_symbol_vector fgl(atermpp::make_vector(f,g));
  function_symbol_vector hl(atermpp::make_vector(h));
  function_symbol_vector fghl(atermpp::make_vector(f,g,h));

  data_specification spec;
  spec.add_sort(s);
  spec.add_sort(s0);

  spec.add_constructor(f);
  spec.add_constructor(g);
  spec.add_constructor(h);

  data_specification spec1(spec);
  std::for_each(fghl.begin(), fghl.end(), boost::bind(&data_specification::add_constructor, &spec1, _1));

  function_symbol_vector constructors(boost::copy_range< function_symbol_vector >(spec.constructors()));
  BOOST_CHECK(spec.constructors(s) == fgl);
  BOOST_CHECK(constructors.size() == 5); // f,g,h, true, false.
  BOOST_CHECK(std::find(constructors.begin(), constructors.end(), f) != constructors.end());
  BOOST_CHECK(std::find(constructors.begin(), constructors.end(), g) != constructors.end());
  BOOST_CHECK(std::find(constructors.begin(), constructors.end(), h) != constructors.end());

  BOOST_CHECK(compare_for_equality(spec, spec1));
  BOOST_CHECK(spec.constructors() == spec1.constructors());
  BOOST_CHECK(spec.constructors(s) == fgl);
  BOOST_CHECK(spec.constructors(s0) == hl);
  BOOST_CHECK(spec1.constructors(s) == fgl);
  BOOST_CHECK(spec1.constructors(s0) == hl);
  spec.add_constructor(function_symbol("i", s0));
  function_symbol i("i", s0);
  spec.remove_constructor(i);
  BOOST_CHECK(compare_for_equality(spec, spec1));

  spec.add_constructor(i);
  function_symbol_vector il(atermpp::make_vector(i));
  std::for_each(il.begin(), il.end(), boost::bind(&data_specification::add_constructor, &spec1, _1));
  BOOST_CHECK(compare_for_equality(spec, spec1));

  spec.remove_constructor(i);
  std::for_each(il.begin(), il.end(), boost::bind(&data_specification::remove_constructor, &spec1, _1));
  BOOST_CHECK(compare_for_equality(spec, spec1));
}

void test_functions()
{
  std::clog << "test_functions" << std::endl;

  basic_sort s("S");
  basic_sort s0("S0");
  function_sort s0s(atermpp::make_vector(sort_expression(s0)), s);
  function_symbol f("f", s);
  function_symbol g("g", s0s);
  function_symbol h("h", s0);

  function_symbol_vector fgl(atermpp::make_vector(f,g));
  function_symbol_vector hl(atermpp::make_vector(h));
  function_symbol_vector fghl(atermpp::make_vector(f,g,h));

  data_specification spec;
  spec.add_sort(s);
  spec.add_sort(s0);
  spec.add_mapping(f);
  spec.add_mapping(g);
  spec.add_mapping(h);

  data_specification spec1(spec);
  std::for_each(fghl.begin(), fghl.end(), boost::bind(&data_specification::add_mapping, &spec1, _1));

  BOOST_CHECK(boost::distance(spec.mappings()) == 35);

  function_symbol_vector mappings(spec.mappings());
  BOOST_CHECK(std::find(mappings.begin(), mappings.end(), f) != mappings.end());
  BOOST_CHECK(std::find(mappings.begin(), mappings.end(), g) != mappings.end());
  BOOST_CHECK(std::find(mappings.begin(), mappings.end(), h) != mappings.end());

  BOOST_CHECK(compare_for_equality(spec, spec1));
  BOOST_CHECK(boost::distance(spec.mappings(s)) == 3);
  BOOST_CHECK(std::find(spec.mappings(s).begin(), spec.mappings(s).end(), f) != spec.mappings(s).end());
  BOOST_CHECK(std::find(spec.mappings(s).begin(), spec.mappings(s).end(), g) != spec.mappings(s).end());
  BOOST_CHECK(std::find(spec.mappings(s0).begin(), spec.mappings(s0).end(), h) != spec.mappings(s0).end());
  BOOST_CHECK(boost::distance(spec1.mappings(s)) == 3);
  BOOST_CHECK(std::find(spec1.mappings(s).begin(), spec1.mappings(s).end(), f) != spec1.mappings(s).end());
  BOOST_CHECK(std::find(spec1.mappings(s).begin(), spec1.mappings(s).end(), g) != spec1.mappings(s).end());
  BOOST_CHECK(std::find(spec1.mappings(s0).begin(), spec1.mappings(s0).end(), h) != spec1.mappings(s0).end());

  function_symbol i("i", s0);
  spec.add_mapping(i);
  function_symbol_vector il(atermpp::make_vector(i));
  std::for_each(il.begin(), il.end(), boost::bind(&data_specification::add_mapping, &spec1, _1));
  compare_for_equality(spec, spec1);

  std::for_each(il.begin(), il.end(), boost::bind(&data_specification::remove_mapping, &spec, _1));
  spec1.remove_mapping(i);
  compare_for_equality(spec, spec1);
}

void test_equations()
{
  std::clog << "test_equations" << std::endl;
  basic_sort s("S");
  basic_sort s0("S0");
  function_sort s0s(atermpp::make_vector(reinterpret_cast<sort_expression&>(s0)), s);
  function_symbol f("f", s0s);
  variable x("x", s0);
  data_expression_vector xel(atermpp::make_vector(reinterpret_cast<data_expression&>(x)));
  application fx(f, boost::make_iterator_range(xel));
  variable_vector xl(atermpp::make_vector(x));
  data_equation fxx(xl, x, fx, x);

  data_specification spec;
  data_specification spec1;
  spec.add_sort(s);
  spec.add_sort(s0);

  spec1 = spec;
  BOOST_CHECK(compare_for_equality(spec, spec1));
  spec.add_equation(fxx);
  data_equation_vector fxxl(atermpp::make_vector(fxx));
  std::for_each(fxxl.begin(), fxxl.end(), boost::bind(&data_specification::add_equation, &spec1, _1));

  BOOST_CHECK(compare_for_equality(spec, spec1));

  data_equation fxf(xl, x, fx, f);
  data_equation_vector fxfl(atermpp::make_vector(fxf));
  spec.add_equation(fxf);
  std::for_each(fxfl.begin(), fxfl.end(), boost::bind(&data_specification::add_equation, &spec1, _1));

  BOOST_CHECK(compare_for_equality(spec, spec1));

  data_equation_vector result = find_equations(spec, f);
  BOOST_CHECK(result.size() == 2);
  BOOST_CHECK(std::find(result.begin(), result.end(), fxf) != result.end());
  BOOST_CHECK(std::find(result.begin(), result.end(), fxx) != result.end());
  std::for_each(fxfl.begin(), fxfl.end(), boost::bind(&data_specification::remove_equation, &spec, _1));
  spec1.remove_equation(fxf);
  BOOST_CHECK(compare_for_equality(spec, spec1));
}

void test_is_certainly_finite()
{
  std::clog << "test_is_certainly_finite" << std::endl;
  basic_sort s("S");
  basic_sort s0("S0");
  function_sort s0s0(atermpp::make_vector(static_cast<sort_expression&>(s0)), s0);
  function_symbol f("f", s);
  function_symbol g("g", s0s0);
  variable x("x", s0);
  application gx(g, boost::make_iterator_range(atermpp::make_vector(static_cast<data_expression&>(x))));
  data_specification spec;
  spec.add_sort(s);
  spec.add_sort(s0);
  spec.add_constructor(f);
  spec.add_constructor(g);

  BOOST_CHECK(spec.is_certainly_finite(s));
  BOOST_CHECK(!spec.is_certainly_finite(s0));

  spec.add_context_sort(sort_real::real_());
  BOOST_CHECK(spec.is_certainly_finite(sort_bool::bool_()));
  BOOST_CHECK(!spec.is_certainly_finite(sort_pos::pos()));
  BOOST_CHECK(!spec.is_certainly_finite(sort_nat::nat()));
  BOOST_CHECK(!spec.is_certainly_finite(sort_int::int_()));
  BOOST_CHECK(!spec.is_certainly_finite(sort_real::real_()));

  basic_sort s1("S1");
  basic_sort s2("S2");
  spec.add_constructor(function_symbol("h", make_function_sort(s1, s2)));
  spec.add_constructor(function_symbol("i", make_function_sort(s2, s1)));

  spec.add_alias(alias(basic_sort("a"), s));
  spec.add_alias(alias(basic_sort("a0"), s0));
  spec.add_alias(alias(basic_sort("a1"), s1));

  BOOST_CHECK(spec.is_certainly_finite(normalize_sorts(basic_sort("a"),spec)));
  BOOST_CHECK(!spec.is_certainly_finite(normalize_sorts(basic_sort("a0"),spec)));
  BOOST_CHECK(!spec.is_certainly_finite(normalize_sorts(basic_sort("a1"),spec)));

  using namespace sort_list;

  BOOST_CHECK(!spec.is_certainly_finite(list(s)));
  BOOST_CHECK(!spec.is_certainly_finite(list(s0)));

  using namespace sort_set;

  BOOST_CHECK(spec.is_certainly_finite(set_(s)));
  BOOST_CHECK(!spec.is_certainly_finite(set_(s0)));

  using namespace sort_bag;

  BOOST_CHECK(!spec.is_certainly_finite(bag(s)));
  BOOST_CHECK(!spec.is_certainly_finite(bag(s0)));
  BOOST_CHECK(spec.is_certainly_finite(make_function_sort(s,s)));
  BOOST_CHECK(!spec.is_certainly_finite(make_function_sort(s,s0)));
  BOOST_CHECK(!spec.is_certainly_finite(make_function_sort(s0,s)));

  // structured sort
  atermpp::vector< data::structured_sort_constructor_argument > arguments;

  arguments.push_back(data::structured_sort_constructor_argument(s));
  arguments.push_back(data::structured_sort_constructor_argument(s0));
  arguments.push_back(data::structured_sort_constructor_argument(s1));

  atermpp::vector< data::structured_sort_constructor > constructors;
  constructors.push_back(data::structured_sort_constructor("a",
                         boost::make_iterator_range(arguments.begin(), arguments.begin() + 1)));
  constructors.push_back(data::structured_sort_constructor("b",
                         boost::make_iterator_range(arguments.begin() + 1, arguments.begin() + 2)));
  constructors.push_back(data::structured_sort_constructor("b",
                         boost::make_iterator_range(arguments.begin() + 2, arguments.begin() + 3)));

  structured_sort struct1(boost::make_iterator_range(constructors.begin(), constructors.begin() + 1));
  structured_sort struct2(boost::make_iterator_range(constructors.begin() + 1, constructors.begin() + 2));
  structured_sort struct3(boost::make_iterator_range(constructors.begin() + 2, constructors.begin() + 3));
  structured_sort struct4(boost::make_iterator_range(constructors.begin() + 0, constructors.begin() + 3));
  spec.add_sort(struct1);
  spec.add_sort(struct2);
  spec.add_sort(struct3);
  spec.add_sort(struct4);
  BOOST_CHECK(spec.is_certainly_finite(struct1));
  BOOST_CHECK(!spec.is_certainly_finite(struct2));
  BOOST_CHECK(!spec.is_certainly_finite(struct3));
  BOOST_CHECK(!spec.is_certainly_finite(struct4));
}

void test_constructor()
{
  std::clog << "test_constructor" << std::endl;
  std::string SPEC =
    "sort                      \n"
    "  D     = struct d1 | d2; \n"
    "  Error = struct e;       \n"
    "                          \n"
    ;
  data_specification data = parse_data_specification(SPEC);
  ATermAppl a = data::detail::data_specification_to_aterm_data_spec(data);
  data_specification spec1(a);
}

template < typename Container, typename Expression >
bool search(Container const& container, Expression const& expression)
{
  return std::find(container.begin(), container.end(), expression) != container.end();
}

bool search_alias(const alias_vector& v, const sort_expression& s)
{
  for(alias_vector::const_iterator i = v.begin(); i != v.end(); ++i)
  {
    if(i->name() == s)
    {
      return true;
    }
  }
  return false;
}

void test_system_defined()
{
  std::clog << "test_system_defined" << std::endl;

  data_specification specification;

  BOOST_CHECK(!specification.constructors(sort_bool::bool_()).empty());

  specification = parse_data_specification(
                    "sort S;"
                    "map f: Set(S);");

  BOOST_CHECK(search(specification.sorts(), sort_set::set_(basic_sort("S"))));
  // BOOST_CHECK(search(specification.sorts(), sort_fset::fset(basic_sort("S")))); MUST BE CHECKED ALSO?

  specification = parse_data_specification(
                    "sort D = Set(Nat);"
                    "sort E = D;"
                    "sort F = E;");
  BOOST_CHECK(specification.constructors(::sort_set::set_(sort_nat::nat())) == specification.constructors(basic_sort("D")));
  BOOST_CHECK(specification.constructors(normalize_sorts(basic_sort("D"),specification)) ==
              specification.constructors(normalize_sorts(basic_sort("E"),specification)));
  BOOST_CHECK(specification.mappings(normalize_sorts(basic_sort("D"),specification)) ==
              specification.mappings(normalize_sorts(basic_sort("E"),specification)));
  BOOST_CHECK(specification.constructors(normalize_sorts(basic_sort("D"),specification)) ==
              specification.constructors(normalize_sorts(basic_sort("F"),specification)));

  data_specification copy = specification;

  // A data specification that is constructed using data_specification_to_aterm_data_spec is assumed not
  // not be type checked. This must be indicated explicitly.
  data_specification specification1=data_specification(detail::data_specification_to_aterm_data_spec(copy));
  specification1.declare_data_specification_to_be_type_checked();
  BOOST_CHECK(compare_for_equality(specification1,specification));

  specification = parse_data_specification(
                    "sort D = struct d(getBool : Bool)?is_d;"
                    "sort E = D;"
                    "sort F = E;");

  alias_vector aliases(specification.user_defined_aliases());
  BOOST_CHECK(search_alias(aliases, basic_sort("D")));
  BOOST_CHECK(search_alias(aliases, basic_sort("E")));
  BOOST_CHECK(search_alias(aliases, basic_sort("F")));
  BOOST_CHECK(specification.constructors(basic_sort("D")).size() == 1);

  BOOST_CHECK(specification.constructors(basic_sort("D")) == specification.constructors(basic_sort("E")));
  BOOST_CHECK(specification.mappings(basic_sort("D")) == specification.mappings(basic_sort("E")));
  BOOST_CHECK(specification.constructors(basic_sort("D")) == specification.constructors(basic_sort("F")));

  copy = specification;
  specification1=data_specification(detail::data_specification_to_aterm_data_spec(copy));
  specification1.declare_data_specification_to_be_type_checked();
  BOOST_CHECK(compare_for_equality(specification1, specification));

  // Check for the non presence of function sort
  BOOST_CHECK(specification.mappings(make_function_sort(basic_sort("D"), sort_bool::bool_())).empty());

  specification.add_mapping(function_symbol("f", make_function_sort(sort_bool::bool_(), sort_bool::bool_(), sort_nat::nat())));

  BOOST_CHECK(!specification.mappings(make_function_sort(sort_bool::bool_(), sort_bool::bool_(), sort_nat::nat())).empty());

  // Manually structured sort
  atermpp::vector< data::structured_sort_constructor_argument > arguments;

  // sort that references itself by a name
  arguments.push_back(data::structured_sort_constructor_argument(basic_sort("Q")));

  atermpp::vector< data::structured_sort_constructor > constructors;
  constructors.push_back(data::structured_sort_constructor("q",
                         boost::make_iterator_range(arguments.begin(), arguments.begin() + 1)));

  specification.add_alias(alias(basic_sort("Q"), data::structured_sort(constructors)));
}

void test_utility_functionality()
{
  data_specification spec;

  basic_sort s("S");
  basic_sort s0("S0");
  basic_sort a("a");
  function_sort s0s(atermpp::make_vector(sort_expression(s0)), s);
  function_symbol f("f", s);

  function_symbol g("g", s0s);
  function_symbol h("h", s0);

  {
    const atermpp::vector<sort_expression> sorts(spec.sorts());
    BOOST_CHECK(std::find(sorts.begin(), sorts.end(), s0) == sorts.end());
    BOOST_CHECK(std::find(sorts.begin(), sorts.end(), s) == sorts.end());
    function_symbol_vector constructors(spec.constructors());
    BOOST_CHECK(std::find(constructors.begin(), constructors.end(), f) == constructors.end());
    function_symbol_vector mappings(spec.mappings());
    BOOST_CHECK(std::find(mappings.begin(), mappings.end(), f) == mappings.end());
    BOOST_CHECK(std::find(mappings.begin(), mappings.end(), g) == mappings.end());
  }

  spec.add_sort(s0);
  spec.add_constructor(f);
  spec.add_mapping(g);

  {
    const atermpp::vector<sort_expression> sorts(spec.sorts());
    BOOST_CHECK(std::find(sorts.begin(), sorts.end(), s0) != sorts.end());
    BOOST_CHECK(std::find(sorts.begin(), sorts.end(), s) != sorts.end()); // Automatically added!
    function_symbol_vector constructors(spec.constructors());
    BOOST_CHECK(std::find(constructors.begin(), constructors.end(), f) != constructors.end());
    function_symbol_vector mappings(spec.mappings());
    BOOST_CHECK(std::find(mappings.begin(), mappings.end(), f) == mappings.end());
    BOOST_CHECK(std::find(mappings.begin(), mappings.end(), g) != mappings.end());
    BOOST_CHECK(std::find(mappings.begin(), mappings.end(), h) == mappings.end());
  }
  spec.add_mapping(h);

  spec.add_sort(s);
  spec.add_alias(alias(basic_sort("a"),s));

  const atermpp::vector<sort_expression> sorts(spec.sorts());
  function_symbol_vector constructors(spec.constructors());
  function_symbol_vector mappings(spec.mappings());


  BOOST_CHECK(std::find(sorts.begin(), sorts.end(), s0) != sorts.end());
  BOOST_CHECK(std::find(sorts.begin(), sorts.end(), normalize_sorts(a,spec)) != sorts.end());
  BOOST_CHECK(std::find(sorts.begin(), sorts.end(), s) != sorts.end());

  BOOST_CHECK(std::find(mappings.begin(), mappings.end(), f) == mappings.end());
  BOOST_CHECK(std::find(constructors.begin(), constructors.end(), f) != constructors.end());

  BOOST_CHECK(std::find(constructors.begin(), constructors.end(), g) == constructors.end());
  BOOST_CHECK(std::find(mappings.begin(), mappings.end(), g) != mappings.end());
  BOOST_CHECK(std::find(constructors.begin(), constructors.end(), g) == constructors.end());
  BOOST_CHECK(std::find(constructors.begin(), constructors.end(), h) == constructors.end());
  BOOST_CHECK(std::find(mappings.begin(), mappings.end(), h) != mappings.end());
  BOOST_CHECK(spec.constructors(a) == spec.constructors(s));
  BOOST_CHECK(spec.mappings(a) == spec.mappings(s));
}

void test_normalisation()
{
  using namespace mcrl2::data;
  using namespace mcrl2::data::sort_list;
  using namespace mcrl2::data::sort_set;
  using namespace mcrl2::data::sort_bag;

  data_specification specification;

  basic_sort A("A");

  specification.add_sort(A);

  basic_sort L("L");
  basic_sort S("S");
  basic_sort B("B");

  specification.add_alias(alias(L, list(A)));
  specification.add_alias(alias(S, set_(A)));
  specification.add_alias(alias(B, bag(A)));
  BOOST_CHECK(normalize_sorts(L,specification) == normalize_sorts(list(A),specification));
  BOOST_CHECK(normalize_sorts(list(L),specification) == normalize_sorts(list(list(A)),specification));
  BOOST_CHECK(normalize_sorts(S,specification) == normalize_sorts(set_(A),specification));
  BOOST_CHECK(normalize_sorts(list(S),specification) == normalize_sorts(list(set_(A)),specification));
  BOOST_CHECK(normalize_sorts(B,specification) == normalize_sorts(bag(A),specification));
  BOOST_CHECK(normalize_sorts(list(B),specification) == normalize_sorts(list(bag(A)),specification));

  specification = parse_data_specification(
                    "sort A = struct a(B);"
                    "sort B = struct b(A)|c;");

  atermpp::vector< structured_sort_constructor_argument > arguments;

  arguments.push_back(structured_sort_constructor_argument(basic_sort("B")));
  arguments.push_back(structured_sort_constructor_argument(basic_sort("A")));

  atermpp::vector< structured_sort_constructor > constructors;

  constructors.push_back(structured_sort_constructor("a", boost::make_iterator_range(arguments.begin(), arguments.begin() + 1)));
  constructors.push_back(structured_sort_constructor("b", boost::make_iterator_range(arguments.begin() + 1, arguments.end())));
  constructors.push_back(structured_sort_constructor("c"));

  structured_sort sA(data::structured_sort(boost::make_iterator_range(constructors.begin(), constructors.begin() + 1)));
  structured_sort sB(data::structured_sort(boost::make_iterator_range(constructors.begin() + 1, constructors.end())));

  const atermpp::vector<sort_expression> sorts(specification.sorts());
  BOOST_CHECK(std::find(sorts.begin(), sorts.end(), normalize_sorts(sA,specification)) != sorts.end());
  BOOST_CHECK(std::find(sorts.begin(), sorts.end(), normalize_sorts(sB,specification)) != sorts.end());

  BOOST_CHECK(normalize_sorts(sA,specification) == normalize_sorts(normalize_sorts(sA,specification),specification));
  BOOST_CHECK(normalize_sorts(sB,specification) == normalize_sorts(normalize_sorts(sB,specification),specification));

  // Check whether the sort expression struct f(struct f(A)|g) |g normalises to A
  // in the context of the specification sort A = struct f(A) | g;
  specification = parse_data_specification(
                    "sort A = struct f(A) | g;");

  atermpp::vector< structured_sort_constructor_argument > arguments1;
  arguments1.push_back(structured_sort_constructor_argument(basic_sort("A")));

  atermpp::vector< structured_sort_constructor > constructors1;
  constructors1.push_back(structured_sort_constructor("f", arguments1));
  constructors1.push_back(structured_sort_constructor("g"));

  sort_expression s1=structured_sort(constructors1); // s1 has the shape struct f(A)|g
  atermpp::vector< structured_sort_constructor_argument > arguments2;
  arguments2.push_back(structured_sort_constructor_argument(s1));

  atermpp::vector< structured_sort_constructor > constructors2;
  constructors2.push_back(structured_sort_constructor("f", arguments2));
  constructors2.push_back(structured_sort_constructor("g"));

  sort_expression s2=structured_sort(constructors2); // s2 has the shape f(struct f(A)|g) |g
  BOOST_CHECK(normalize_sorts(s2,specification)==basic_sort("A"));
}

void test_copy()
{
  std::clog << "test_copy" << std::endl;

  data_specification specification = parse_data_specification(
                                       "sort D = struct d(bla : Bool)?is_d;"
                                       "sort A = S;"
                                       "sort S;"
                                       "map f: Set(S);");

  function_symbol_vector constructors(specification.constructors());
  BOOST_CHECK(std::find(constructors.begin(), constructors.end(), sort_bool::true_()) != constructors.end());

  data_specification other;
  other = specification;

  BOOST_CHECK(other == specification);

  specification = data_specification();

  core::garbage_collect();

  BOOST_CHECK(normalize_sorts(basic_sort("A"),other) == normalize_sorts(basic_sort("S"),other));

  const atermpp::vector<sort_expression> sorts(specification.sorts());
  BOOST_CHECK(std::find(sorts.begin(), sorts.end(), basic_sort("A")) == sorts.end());
}

void test_specification()
{
  data_specification spec = parse_data_specification("sort D = struct d1|d2;");

  BOOST_CHECK(spec.constructors(basic_sort("D")).size() == 2);
}

void test_bke()
{
  std::cout << "test_bke" << std::endl;

  std::string BKE =
    "% This model is translated from the mCRL model used for analysing the Bilateral                \n"
    "% Key Exchange (BKE) protocol. The analysis is described in a paper with the                   \n"
    "% name 'Analysing the BKE-security protocol with muCRL', by Jan Friso Groote,                  \n"
    "% Sjouke Mauw and Alexander Serebrenik.                                                        \n"
    "%                                                                                              \n"
    "% The translation of the existing mCRL model into this mCRL2 model has been                    \n"
    "% performed manually.  The purpose was making use of the additional language                   \n"
    "% features of mCRL2 with respect to mCRL.                                                      \n"
    "%                                                                                              \n"
    "% The behaviour of this model should be bisimular with the original mcrl model                 \n"
    "% after renaming actions. Though this is not verified for all system                           \n"
    "% configurations below.                                                                        \n"
    "%                                                                                              \n"
    "% Eindhoven, June 11, 2008, Jeroen van der Wulp                                                \n"
    "                                                                                               \n"
    "% Agents. There are exactly three agents - A, B, E An order E < A < B is                       \n"
    "% imposed on agents to reduce the size of the state space.                                     \n"
    "sort Agent = struct A | B | E;                                                                 \n"
    "map  less : Agent # Agent -> Bool;                                                             \n"
    "var  a: Agent;                                                                                 \n"
    "eqn  less(A,a) = (a == B);                                                                     \n"
    "     less(B,a) = false;                                                                        \n"
    "     less(E,a) = (a != E);                                                                     \n"
    "                                                                                               \n"
    "sort Address = struct address(agent : Agent);                                                  \n"
    "map  bad_address : Address;                                                                    \n"
    "                                                                                               \n"
    "% A nonce is a random, unpredictable value which is used to make the                           \n"
    "% exchanged messages unique and thus helps to counter replay attacks.                          \n"
    "sort Nonce = struct nonce(value : Nat);                                                        \n"
    "                                                                                               \n"
    "% There are two kinds of keys used in the protocol: symmetric and                              \n"
    "% asymmetric ones (functional keys).                                                           \n"
    "% Symmetric keys have form K(n) where n is a natural number.                                   \n"
    "sort SymmetricKey = struct symmetric_key(value : Nat);                                         \n"
    "                                                                                               \n"
    "% Sort for representing asymmetric keys                                                        \n"
    "sort AsymmetricKey = struct public_key(Agent)?is_public |                                      \n"
    "                            secret_key(Agent)?is_secret |                                      \n"
    "                            hash(value : Nonce)?is_hash;                                       \n"
    "map  has_complementary_key: AsymmetricKey -> Bool;                                             \n"
    "     complementary_key    : AsymmetricKey -> AsymmetricKey;                                    \n"
    "var  a : Agent;                                                                                \n"
    "     n : Nonce;                                                                                \n"
    "eqn  has_complementary_key(public_key(a)) = true;                                              \n"
    "     has_complementary_key(secret_key(a)) = true;                                              \n"
    "     has_complementary_key(hash(n)) = false;                                                   \n"
    "     complementary_key(public_key(a)) = secret_key(a);                                         \n"
    "     complementary_key(secret_key(a)) = public_key(a);                                         \n"
    "                                                                                               \n"
    "sort Key = struct key(SymmetricKey)?is_symmetric | key(AsymmetricKey)?is_asymmetric;           \n"
    "map  has_complementary_key: Key -> Bool;                                                       \n"
    "     complementary_key : Key -> Key; % gets the complementary key if key is asymmetric         \n"
    "var  a,a1   : Agent;                                                                           \n"
    "     n,n1   : Nat;                                                                             \n"
    "     k,k1   : Key;                                                                             \n"
    "     ak,ak1 : AsymmetricKey;                                                                   \n"
    "eqn                                                                                            \n"
    "     % gets the complementary key if key is asymmetric                                         \n"
    "     complementary_key(key(ak)) = key(complementary_key(ak));                                  \n"
    "     has_complementary_key(key(ak)) = has_complementary_key(ak);                               \n"
    "                                                                                               \n"
    "sort Message = struct                                                                          \n"
    "        encrypt(Nonce, Address, AsymmetricKey)?is_message_1 |                                  \n"
    "        encrypt(AsymmetricKey, Nonce, SymmetricKey, AsymmetricKey)?is_message_2 |              \n"
    "        encrypt(AsymmetricKey, SymmetricKey)?is_message_3;                                     \n"
    "map  valid_message_1 : Message # AsymmetricKey -> Bool;                                        \n"
    "     valid_message_2 : Message # AsymmetricKey -> Bool;                                        \n"
    "     valid_message_3 : Message # SymmetricKey  -> Bool;                                        \n"
    "     used_key : Message -> Key;                  % key used to encrypt                         \n"
    "var  sk, sk1 : SymmetricKey;                                                                   \n"
    "     ak, ak1, ak2 : AsymmetricKey;                                                             \n"
    "     n, n1  : Nonce;                                                                           \n"
    "     m, m1   : Message;                                                                        \n"
    "     a, a1   : Address;                                                                        \n"
    "eqn  used_key(encrypt(n,a,ak)) = key(ak);                                                      \n"
    "     used_key(encrypt(ak,n1,sk,ak1)) = key(ak1);                                               \n"
    "     used_key(encrypt(ak,sk)) = key(sk);                                                       \n"
    "     valid_message_1(m, ak) = is_message_1(m) && (used_key(m) == key(ak));                     \n"
    "     valid_message_2(m, ak) = is_message_2(m) && (used_key(m) == key(ak));                     \n"
    "     valid_message_3(m, sk) = is_message_3(m) && (used_key(m) == key(sk));                     \n"
    "                                                                                               \n"
    "% Type for message sets; currently cannot use Set() because set iteration is not possible      \n"
    "sort MessageSet = List(Message);                                                               \n"
    "map  insert                : Message # MessageSet -> MessageSet;                               \n"
    "     select_crypted_by     : Key # MessageSet -> MessageSet;                                   \n"
    "     select_not_crypted_by : Key # MessageSet -> MessageSet;                                   \n"
    "     select                : (Message -> Bool) # MessageSet -> MessageSet;                     \n"
    "var  k,k1 : Key;                                                                               \n"
    "     m,m1 : Message;                                                                           \n"
    "     ms   : MessageSet;                                                                        \n"
    "     c    : Message -> Bool;                                                                   \n"
    "eqn                                                                                            \n"
    "     % inserts a message m, if it is not in the list                                           \n"
    "     insert(m,[]) = [m];                                                                       \n"
    "     m < m1  -> insert(m,m1|>ms) = m|>m1|>ms;                                                  \n"
    "     m == m1 -> insert(m,m1|>ms) = m1|>ms;                                                     \n"
    "     m1 < m  -> insert(m,m1|>ms) = m1|>insert(m,ms);                                           \n"
    "                                                                                               \n"
    "     % the set (as ordered list) of messages in m that are signed by sk                        \n"
    "     select_crypted_by(k,ms) = select(lambda x : Message.k == used_key(x),ms);                 \n"
    "     select_not_crypted_by(k,ms) = select(lambda x : Message.k != used_key(x),ms);             \n"
    "                                                                                               \n"
    "     select(c,[]) = [];                                                                        \n"
    "     select(c,m|>ms) = if(c(m),m|>r,r) whr r = select(c, ms) end;                              \n"
    "                                                                                               \n"
    "% The eavesdropper's knowledge consists of:                                                    \n"
    "%  * a list of addresses                                                                       \n"
    "%  * a list of nonces                                                                          \n"
    "%  * a list of keys (both symmetric and asymmetric)                                            \n"
    "%  * a list of messages of which the key is not known                                          \n"
    "sort Knowledge = struct                                                                        \n"
    "        knowledge(addresses : Set(Address),                                                    \n"
    "                  nonces : Set(Nonce),                                                         \n"
    "                  keys : Set(Key),                                                             \n"
    "                  messages : MessageSet);                                                      \n"
    "map  update_knowledge : Message # Knowledge -> Knowledge;                                      \n"
    "     propagate : MessageSet # Knowledge -> Knowledge;                                          \n"
    "     propagate : Key # Knowledge -> Knowledge;                                                 \n"
    "     add_key : Key # Knowledge -> Knowledge;                                                   \n"
    "     add_nonce : Nonce # Knowledge -> Knowledge;                                               \n"
    "     add_address : Address # Knowledge -> Knowledge;                                           \n"
    "var  m  : Message;                                                                             \n"
    "     as : Set(Address);                                                                        \n"
    "     ns : Set(Nonce);                                                                          \n"
    "     ks : Set(Key);                                                                            \n"
    "     ms : MessageSet;                                                                          \n"
    "     k  : Knowledge;                                                                           \n"
    "     sk : SymmetricKey;                                                                        \n"
    "     ak,hk : AsymmetricKey;                                                                    \n"
    "     ck : Key;                                                                                 \n"
    "     n, n1 : Nonce;                                                                            \n"
    "     a  : Address;                                                                             \n"
    "eqn                                                                                            \n"
    "     % adds keys to knowledge that are part of known messages encrypted with a new key         \n"
    "     has_complementary_key(ak) && complementary_key(key(ak)) in keys(k) ->                     \n"
    "         update_knowledge(encrypt(n,a,ak),k) =                                                 \n"
    "                propagate(key(ak), add_key(key(ak),                                            \n"
    "                        add_address(a, add_nonce(n, k))));                                     \n"
    "     has_complementary_key(ak) && complementary_key(key(ak)) in keys(k) ->                     \n"
    "         update_knowledge(encrypt(hk,n1,sk,ak),k) =                                            \n"
    "                propagate(key(sk), propagate(key(ak),                                          \n"
    "                      add_key(key(sk),add_key(key(ak),k))));                                   \n"
    "     key(sk) in keys(k) ->                                                                     \n"
    "         update_knowledge(encrypt(ak,sk),k) =                                                  \n"
    "                propagate(key(ak), add_key(key(ak),k));                                        \n"
    "                                                                                               \n"
    "     % adds a message that cannot be decrypted with any known key                              \n"
    "     ((is_symmetric(ck) && !(ck in keys(k))) ||                                                \n"
    "       (is_asymmetric(ck) && !(has_complementary_key(ck) &&                                    \n"
    "                (complementary_key(ck) in keys(k))))) whr ck = used_key(m) end ->              \n"
    "         update_knowledge(m,k) =                                                               \n"
    "                knowledge(addresses(k),nonces(k),keys(k),insert(m,messages(k)));               \n"
    "                                                                                               \n"
    "     % adds a key to knowledge                                                                 \n"
    "     add_key(ck,knowledge(as,ns,ks,ms)) = knowledge(as,ns,ks + {ck},ms);                       \n"
    "     add_nonce(n,knowledge(as,ns,ks,ms)) = knowledge(as,ns + {n},ks,ms);                       \n"
    "     add_address(a,knowledge(as,ns,ks,ms)) = knowledge(as + {a},ns,ks,ms);                     \n"
    "                                                                                               \n"
    "     % adds keys to knowledge that are part of messages encrypted with a key k                 \n"
    "     propagate([],k) = k;                                                                      \n"
    "     propagate(m|>ms,k) = propagate(ms, update_knowledge(m, k));                               \n"
    "     propagate(ck,knowledge(as,ns,ks,ms)) =                                                    \n"
    "        propagate(select_crypted_by(ck,ms),                                                    \n"
    "                knowledge(as,ns,ks,select_not_crypted_by(ck,ms)));                             \n"
    ;

  data_specification data_spec = parse_data_specification(BKE);
  const alias_vector& aliases = data_spec.user_defined_aliases();
  for (alias_vector::const_iterator i = aliases.begin(); i != aliases.end(); ++i)
  {
    std::cout << "alias " << data::pp(*i) << std::endl;
    sort_expression s = i->reference();
    if (is_structured_sort(s))
    {
      structured_sort_constructor_list constructors = structured_sort(s).constructors();
      for (structured_sort_constructor_list::const_iterator j = constructors.begin(); j != constructors.end(); ++j)
      {
        structured_sort_constructor_argument_list arguments = j->arguments();
        for (structured_sort_constructor_argument_list::const_iterator k = arguments.begin(); k != arguments.end(); ++k)
        {
          std::cout << "argument: " << data::pp(*k) << " " << *k << std::endl;
          atermpp::aterm_appl name = k->name();
          if (name != no_identifier())
          {
            std::cout << "name = " << name << std::endl;
          }
        }
      }
    }
  }
}

int test_main(int argc, char** argv)
{
  MCRL2_ATERMPP_INIT(argc, argv);

  test_bke();

  test_sorts();
  core::garbage_collect();

  test_constructors();
  core::garbage_collect();

  test_functions();
  core::garbage_collect();

  test_equations();
  core::garbage_collect();

  test_is_certainly_finite();
  core::garbage_collect();

  test_constructor();
  core::garbage_collect();

  test_system_defined();
  core::garbage_collect();

  test_utility_functionality();
  core::garbage_collect();

  test_normalisation();
  core::garbage_collect();

  test_copy();
  core::garbage_collect();

  test_specification();
  core::garbage_collect();
    
  return EXIT_SUCCESS;
}


