#include "libfilezilla/impersonation.hpp"
#include "libfilezilla/process.hpp"

#ifdef FZ_WINDOWS

#include "libfilezilla/glue/windows.hpp"

namespace fz {

namespace {
void reset_handle(HANDLE& handle)
{
	if (handle != INVALID_HANDLE_VALUE) {
		CloseHandle(handle);
		handle = INVALID_HANDLE_VALUE;
	}
}

bool uninherit(HANDLE& handle)
{
	if (handle != INVALID_HANDLE_VALUE) {
		HANDLE newHandle = INVALID_HANDLE_VALUE;

		if (!DuplicateHandle(GetCurrentProcess(), handle, GetCurrentProcess(), &newHandle, 0, FALSE, DUPLICATE_SAME_ACCESS)) {
			newHandle = INVALID_HANDLE_VALUE;
		}
		CloseHandle(handle);
		handle = newHandle;
	}

	return handle != INVALID_HANDLE_VALUE;
}

class pipe final
{
public:
	pipe() = default;

	~pipe()
	{
		reset();
	}

	pipe(pipe const&) = delete;
	pipe& operator=(pipe const&) = delete;

	bool create(bool local_is_input)
	{
		reset();

		SECURITY_ATTRIBUTES sa{};
		sa.bInheritHandle = TRUE;
		sa.nLength = sizeof(sa);

		BOOL res = CreatePipe(&read_, &write_, &sa, 0);
		if (res) {
			// We only want one side of the pipe to be inheritable
			if (!uninherit(local_is_input ? read_ : write_)) {
				reset();
			}
		}
		else {
			read_ = INVALID_HANDLE_VALUE;
			write_ = INVALID_HANDLE_VALUE;
		}
		return valid();
	}

	bool valid() const {
		return read_ != INVALID_HANDLE_VALUE && write_ != INVALID_HANDLE_VALUE;
	}

	void reset()
	{
		reset_handle(read_);
		reset_handle(write_);
	}

	HANDLE read_{INVALID_HANDLE_VALUE};
	HANDLE write_{INVALID_HANDLE_VALUE};
};

native_string escape_argument(native_string const& arg)
{
	native_string ret;

	// Treat newlines as whitespace just to be sure, even if MSDN doesn't mention it
	if (arg.find_first_of(fzT(" \"\t\r\n\v")) != native_string::npos) {
		// Quite horrible, as per MSDN:
		// Backslashes are interpreted literally, unless they immediately precede a double quotation mark.
		// If an even number of backslashes is followed by a double quotation mark, one backslash is placed in the argv array for every pair of backslashes, and the double quotation mark is interpreted as a string delimiter.
		// If an odd number of backslashes is followed by a double quotation mark, one backslash is placed in the argv array for every pair of backslashes, and the double quotation mark is "escaped" by the remaining backslash, causing a literal double quotation mark (") to be placed in argv.

		ret = fzT("\"");
		int backslashCount = 0;
		for (auto it = arg.begin(); it != arg.end(); ++it) {
			if (*it == '\\') {
				++backslashCount;
			}
			else {
				if (*it == '"') {
					// Escape all preceeding backslashes and escape the quote
					ret += native_string(backslashCount + 1, '\\');
				}
				backslashCount = 0;
			}
			ret += *it;
		}
		if (backslashCount) {
			// Escape all preceeding backslashes
			ret += native_string(backslashCount, '\\');
		}

		ret += fzT("\"");
	}
	else {
		ret = arg;
	}

	return ret;
}

native_string get_cmd_line(native_string const& cmd, std::vector<native_string>::const_iterator const& begin, std::vector<native_string>::const_iterator const& end)
{
	native_string cmdline = escape_argument(cmd);

	for (auto it = begin; it != end; ++it) {
		auto const& arg = *it;
		if (!arg.empty()) {
			cmdline += fzT(" ") + escape_argument(arg);
		}
	}

	return cmdline;
}
}

HANDLE get_handle(impersonation_token const& t);
class process::impl
{
public:
	impl() = default;
	~impl()
	{
		kill();
	}

	impl(impl const&) = delete;
	impl& operator=(impl const&) = delete;

	bool create_pipes()
	{
		return
			in_.create(false) &&
			out_.create(true) &&
			err_.create(true);
	}

	bool spawn(native_string const& cmd, std::vector<native_string>::const_iterator const& begin, std::vector<native_string>::const_iterator const& end, io_redirection redirect_mode, impersonation_token const* it = nullptr)
	{
		if (process_ != INVALID_HANDLE_VALUE) {
			return false;
		}

		STARTUPINFO si{};
		si.cb = sizeof(si);

		if (redirect_mode != io_redirection::none) {
			if (!create_pipes()) {
				return false;
			}

			si.dwFlags = STARTF_USESTDHANDLES;
			si.hStdInput = in_.read_;
			si.hStdOutput = out_.write_;
			si.hStdError = err_.write_;
		}

		auto cmdline = get_cmd_line(cmd, begin, end);

		PROCESS_INFORMATION pi{};

		auto cmdline_buf = cmdline.data();

		DWORD const flags = CREATE_UNICODE_ENVIRONMENT | CREATE_DEFAULT_ERROR_MODE | CREATE_NO_WINDOW;
		BOOL res;
		if (it) {
			 res = CreateProcessAsUser(get_handle(*it), cmd.c_str(), cmdline_buf, nullptr, nullptr, TRUE, flags, nullptr, nullptr, &si, &pi);
		}
		else {
			res = CreateProcess(cmd.c_str(), cmdline_buf, nullptr, nullptr, TRUE, flags, nullptr, nullptr, &si, &pi);
		}
		if (!res) {
			return false;
		}

		process_ = pi.hProcess;

		// We don't need to use these
		if (redirect_mode != io_redirection::none) {
			reset_handle(pi.hThread);
			reset_handle(in_.read_);
			reset_handle(out_.write_);
			reset_handle(err_.write_);
			if (redirect_mode == io_redirection::closeall) {
				reset_handle(in_.write_);
				reset_handle(out_.read_);
				reset_handle(err_.read_);
			}
		}

		return true;
	}

	void kill()
	{
		if (process_ != INVALID_HANDLE_VALUE) {
			in_.reset();
			if (WaitForSingleObject(process_, 500) == WAIT_TIMEOUT) {
				TerminateProcess(process_, 0);
			}
			reset_handle(process_);
			out_.reset();
			err_.reset();
		}
	}

	int read(char* buffer, unsigned int len)
	{
		DWORD read = 0;
		BOOL res = ReadFile(out_.read_, buffer, len, &read, nullptr);
		if (!res) {
#if FZ_WINDOWS
			// ERROR_BROKEN_PIPE indicated EOF.
			if (GetLastError() == ERROR_BROKEN_PIPE) {
				return 0;
			}
#endif
			return -1;
		}
		return read;
	}

	bool write(char const* buffer, unsigned int len)
	{
		while (len > 0) {
			DWORD written = 0;
			BOOL res = WriteFile(in_.write_, buffer, len, &written, nullptr);
			if (!res || written == 0) {
				return false;
			}
			buffer += written;
			len -= written;
		}
		return true;
	}

	HANDLE handle() const { return process_; }

private:
	HANDLE process_{INVALID_HANDLE_VALUE};

	pipe in_;
	pipe out_;
	pipe err_;
};

#else

#include "libfilezilla/glue/unix.hpp"
#include "libfilezilla/mutex.hpp"

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/wait.h>
#include <string.h>
#include <unistd.h>

#include <atomic>
#include <memory>
#include <vector>

#if FZ_MAC
#include "libfilezilla/local_filesys.hpp"

#include <CoreFoundation/CFArray.h>
#include <CoreFoundation/CFURL.h>
#include <CoreFoundation/CFBundle.h>

#include <ApplicationServices/ApplicationServices.h>
#endif

namespace fz {

namespace {
void reset_fd(int& fd)
{
	if (fd != -1) {
		close(fd);
		fd = -1;
	}
}

class pipe final
{
public:
	pipe() = default;

	~pipe()
	{
		reset();
	}

	pipe(pipe const&) = delete;
	pipe& operator=(pipe const&) = delete;

	bool create()
	{
		reset();

		int fds[2];
		if (!create_pipe(fds)) {
			return false;
		}

		read_ = fds[0];
		write_ = fds[1];

		return valid();
	}

	bool valid() const {
		return read_ != -1 && write_ != -1;
	}

	void reset()
	{
		reset_fd(read_);
		reset_fd(write_);
	}

	int read_{-1};
	int write_{-1};
};

void get_argv(native_string const& cmd, std::vector<native_string>::const_iterator const& begin, std::vector<native_string>::const_iterator const& end, std::vector<char*> & argV)
{
	argV.reserve(end - begin + 2);
	argV.emplace_back(const_cast<char*>(cmd.c_str()));
	for (auto it = begin; it != end; ++it) {
		argV.emplace_back(const_cast<char*>(it->c_str()));
	}
	argV.emplace_back(nullptr);
}

std::atomic<unsigned int> forkblocks_{};
mutex forkblock_mtx_;
}

class process::impl
{
public:
	impl() = default;
	~impl()
	{
		kill();
	}

	impl(impl const&) = delete;
	impl& operator=(impl const&) = delete;

	bool create_pipes()
	{
		return
			in_.create() &&
			out_.create() &&
			err_.create();
	}

	bool spawn(native_string const& cmd, std::vector<native_string>::const_iterator const& begin, std::vector<native_string>::const_iterator const& end, io_redirection redirect_mode, std::vector<int> const& extra_fds = std::vector<int>(), impersonation_token const* it = nullptr)
	{
		if (pid_ != -1) {
			return false;
		}

		if (redirect_mode != io_redirection::none && !create_pipes()) {
			return false;
		}

		std::vector<char*> argV;
		get_argv(cmd, begin, end, argV);

		scoped_lock l(forkblock_mtx_);
		pid_t pid = fork();
		if (pid < 0) {
			return false;
		}
		else if (!pid) {
			// We're the child.

			if (redirect_mode != io_redirection::none) {
				// Close uneeded descriptors
				reset_fd(in_.write_);
				reset_fd(out_.read_);
				reset_fd(err_.read_);

				// Redirect to pipe. The redirected descriptors don't have
				// FD_CLOEXEC set.
				// Note that even if redirect_mode is closeall, we still leave valid descriptors
				// at stdin/out/err as we do not want to have these re-used for other things.
				if (dup2(in_.read_, STDIN_FILENO) == -1 ||
					dup2(out_.write_, STDOUT_FILENO) == -1 ||
					dup2(err_.write_, STDERR_FILENO) == -1)
				{
					_exit(-1);
				}
			}

			// Clear FD_CLOEXEC on extra descriptors
			for (int fd : extra_fds) {
				int flags = fcntl(fd, F_GETFD);
				if (flags == -1) {
					_exit(1);
				}
				if (fcntl(fd, F_SETFD, flags & ~FD_CLOEXEC) != 0) {
					_exit(1);
				}
			}

			if (it && *it) {
				if (!set_process_impersonation(*it)) {
					_exit(1);
				}
			}

			// Execute process
			execv(cmd.c_str(), argV.data()); // noreturn on success

			_exit(-1);
		}
		else {
			// We're the parent
			pid_ = pid;

			// Close unneeded descriptors
			if (redirect_mode != io_redirection::none) {
				reset_fd(in_.read_);
				reset_fd(out_.write_);
				reset_fd(err_.write_);
				if (redirect_mode == io_redirection::closeall) {
					reset_fd(in_.write_);
					reset_fd(out_.read_);
					reset_fd(err_.read_);
				}
			}
		}

		return true;
	}

	void kill()
	{
		in_.reset();

		if (pid_ != -1) {
			::kill(pid_, SIGTERM);

			int ret;
			do {
			} while ((ret = waitpid(pid_, nullptr, 0)) == -1 && errno == EINTR);

			(void)ret;

			pid_ = -1;
		}

		out_.reset();
		err_.reset();
	}

	int read(char* buffer, unsigned int len)
	{
		int r;
		do {
			r = ::read(out_.read_, buffer, len);
		} while (r == -1 && (errno == EAGAIN || errno == EINTR));

		return r;
	}

	bool write(char const* buffer, unsigned int len)
	{
		while (len) {
			int written;
			do {
				written = ::write(in_.write_, buffer, len);
			} while (written == -1 && (errno == EAGAIN || errno == EINTR));

			if (written <= 0) {
				return false;
			}

			len -= written;
			buffer += written;
		}
		return true;
	}

	pipe in_;
	pipe out_;
	pipe err_;

	int pid_{-1};
};

#endif


process::process()
	: impl_(new impl)
{
}

process::~process()
{
	delete impl_;
}

bool process::spawn(native_string const& cmd, std::vector<native_string> const& args, io_redirection redirect_mode)
{
	return impl_ ? impl_->spawn(cmd, args.cbegin(), args.cend(), redirect_mode) : false;
}

#if FZ_WINDOWS || FZ_UNIX
bool process::spawn(impersonation_token const& it, native_string const& cmd, std::vector<native_string> const& args, io_redirection redirect_mode)
{
#if FZ_WINDOWS
	return impl_ ? impl_->spawn(cmd, args.cbegin(), args.cend(), redirect_mode, &it) : false;
#else
	return impl_ ? impl_->spawn(cmd, args.cbegin(), args.cend(), redirect_mode, {}, &it) : false;
#endif
}
#endif

#ifndef FZ_WINDOWS
bool process::spawn(native_string const& cmd, std::vector<native_string> const& args, std::vector<int> const& extra_fds, io_redirection redirect_mode)
{
	return impl_ ? impl_->spawn(cmd, args.cbegin(), args.cend(), redirect_mode, extra_fds) : false;
}

bool process::spawn(impersonation_token const& it, native_string const& cmd, std::vector<native_string> const& args, std::vector<int> const& extra_fds, io_redirection redirect_mode)
{
	return impl_ ? impl_->spawn(cmd, args.cbegin(), args.cend(), redirect_mode, extra_fds, &it) : false;
}

#endif

bool process::spawn(std::vector<native_string> const& command_with_args, io_redirection redirect_mode)
{
	if (command_with_args.empty()) {
		return false;
	}
	auto begin = command_with_args.begin() + 1;
	return impl_ ? impl_->spawn(command_with_args.front(), begin, command_with_args.end(), redirect_mode) : false;
}

void process::kill()
{
	if (impl_) {
		impl_->kill();
	}
}

int process::read(char* buffer, unsigned int len)
{
	return impl_ ? impl_->read(buffer, len) : -1;
}

bool process::write(char const* buffer, unsigned int len)
{
	return impl_ ? impl_->write(buffer, len) : false;
}

#if FZ_WINDOWS
HANDLE process::handle() const
{
	return impl_ ? impl_->handle() : INVALID_HANDLE_VALUE;
}
#endif

#if FZ_MAC
namespace {
template<typename T>
class cfref final
{
public:
	cfref() = default;
	~cfref() {
		if (ref_) {
			CFRelease(ref_);
		}
	}

	explicit cfref(T ref, bool fromCreate = true)
		: ref_(ref)
	{
		if (ref_ && !fromCreate) {
			CFRetain(ref_);
		}
	}

	cfref(cfref const& op)
		: ref_(op.ref)
	{
		if (ref_) {
			CFRetain(ref_);
		}
	}

	cfref& operator=(cfref const& op)
	{
		if (this != &op) {
			if (ref_) {
				CFRelease(ref_);
			}
			ref_ = op.ref_;
			if (ref_) {
				CFRetain(ref_);
			}
		}

		return *this;

	}

	explicit operator bool() const { return ref_ != nullptr; }

	operator T& () { return ref_; }
	operator T const& () const { return ref_; }

private:
	T ref_{};
};

typedef cfref<CFStringRef> cfsr;
typedef cfref<CFURLRef> cfurl;
typedef cfref<CFBundleRef> cfbundle;
typedef cfref<CFMutableArrayRef> cfma;

cfsr cfsr_view(std::string_view const& v)
{
	return cfsr(CFStringCreateWithBytesNoCopy(nullptr, reinterpret_cast<uint8_t const*>(v.data()), v.size(), kCFStringEncodingUTF8, false, kCFAllocatorNull), true);
}

int try_launch_bundle(std::vector<std::string> const& cmd_with_args)
{
	std::string_view cmd(cmd_with_args[0]);
	if (!cmd.empty() && cmd.back() == '/') {
		cmd = cmd.substr(0, cmd.size() - 1);
	}
	if (!ends_with(cmd, std::string_view(".app"))) {
		return -1;
	}

	if (local_filesys::get_file_type(cmd_with_args[0], true) != local_filesys::dir) {
		return -1;
	}

	// Treat it as a bundle


	cfurl bundle_url(CFURLCreateWithFileSystemPath(nullptr, cfsr_view(cmd), kCFURLPOSIXPathStyle, true));
	if (!bundle_url) {
		return 0;
	}

	cfbundle bundle(CFBundleCreate(nullptr, bundle_url));
	if (!bundle) {
		return 0;
	}

	// Require the bundle to be an application
	uint32_t type, creator;
	CFBundleGetPackageInfo(bundle, &type, &creator);
	if (type != 'APPL') {
		return 0;
	}

	cfma args(CFArrayCreateMutable(nullptr, 0, &kCFTypeArrayCallBacks));
	if (!args) {
		return 0;
	}

	for (size_t i = 1; i < cmd_with_args.size(); ++i) {
		cfurl arg_url;

		auto const& arg = cmd_with_args[i];

		if (!arg.empty() && arg.front() == '/') {
			auto t = local_filesys::get_file_type(cmd_with_args[i], true);
			if (t != local_filesys::unknown) {
				arg_url = cfurl(CFURLCreateWithFileSystemPath(nullptr, cfsr_view(arg), kCFURLPOSIXPathStyle, t == local_filesys::dir));
				if (!arg_url) {
					return 0;
				}
			}
		}

		if (!arg_url) {
			arg_url = cfurl(CFURLCreateWithString(nullptr, cfsr_view(arg), nullptr));
		}

		if (!arg_url) {
			return 0;
		}
		CFArrayAppendValue(args, arg_url);
	}

	LSLaunchURLSpec ls{};
	ls.appURL = bundle_url;
	ls.launchFlags = kLSLaunchDefaults;
	ls.itemURLs = args;
	if (LSOpenFromURLSpec(&ls, nullptr) != noErr) {
		return 0;
	}

	return 1;
}
}
#endif

bool spawn_detached_process(std::vector<native_string> const& cmd_with_args)
{
	if (cmd_with_args.empty() || cmd_with_args[0].empty()) {
		return false;
	}

#ifdef FZ_WINDOWS
	STARTUPINFO si{};
	si.cb = sizeof(si);

	auto begin = cmd_with_args.cbegin() + 1;
	auto cmdline = get_cmd_line(cmd_with_args.front(), begin, cmd_with_args.cend());

	PROCESS_INFORMATION pi{};

	auto cmdline_buf = cmdline.data();

	DWORD const flags = CREATE_UNICODE_ENVIRONMENT | CREATE_DEFAULT_ERROR_MODE | CREATE_NO_WINDOW;
	BOOL res = CreateProcess(cmd_with_args.front().c_str(), cmdline_buf, nullptr, nullptr, TRUE, flags, nullptr, nullptr, &si, &pi);
	if (!res) {
		return false;
	}

	reset_handle(pi.hProcess);
	reset_handle(pi.hThread);
	return true;
#else
	if (cmd_with_args[0][0] != '/') {
		return false;
	}

#if FZ_MAC
	// Special handling for application bundles if passed a single file name
	int res = try_launch_bundle(cmd_with_args);
	if (res != -1) {
		return res == 1;
	}
#endif

	std::vector<char *> argV;
	auto begin = cmd_with_args.cbegin() + 1;
	get_argv(cmd_with_args.front(), begin, cmd_with_args.cend(), argV);

	pid_t const parent = getppid();
	pid_t const ppgid = getpgid(parent);

	scoped_lock l(forkblock_mtx_);

	// We're using a pipe created with O_CLOEXEC to signal failure from execv.
	// Fortunately the forkblock avoids a deadlock if the cloexec flag isn't set
	// atomically and another threa execs in-between, as even in 2022, macOS
	// doesn't have pipe2.
	pipe errpipe;
	errpipe.create();

	pid_t pid = fork();
	if (!pid) {
		reset_fd(errpipe.read_);

		// We're the child
		pid_t inner_pid = fork();
		if (!inner_pid) {
			// Change the process group ID of the new process so that terminating the outer process does not terminate the child
			setpgid(0, ppgid);
			execv(cmd_with_args[0].c_str(), argV.data());

			if (errpipe.write_ != -1) {
				ssize_t w;
				do {
					w = ::write(errpipe.write_, "", 1);
				} while (w == -1 && (errno == EAGAIN || errno == EINTR));
			}

			_exit(-1);
		}
		else {
			_exit(0);
		}

		return false;
	}
	else {
		reset_fd(errpipe.write_);

		// We're the parent
		int ret;
		do {
		} while ((ret = waitpid(pid, nullptr, 0)) == -1 && errno == EINTR);

		if (ret == -1) {
			return false;
		}

		if (errpipe.read_ != -1) {
			ssize_t r;
			char tmp;
			do {
				r = ::read(errpipe.read_, &tmp, 1);
			} while (r == -1 && (errno == EAGAIN || errno == EINTR));
			if (r == 1) {
				// execv failed in the child.
				return false;
			}
		}

		return true;
	}
#endif
}

#if !FZ_WINDOWS
forkblock::forkblock()
{
	forkblock_mtx_.lock();
	++forkblocks_;
}

forkblock::~forkblock()
{
	--forkblocks_;
	forkblock_mtx_.unlock();
}

namespace {
bool forked_child_{};
void atfork_lock_forkblock()
{
	// Unsafe if fork is called from a signal handler
	if (!forked_child_) {
		forkblock_mtx_.lock();
	}
}

void atfork_unlock_forkblock()
{
	// Unsafe if fork is called from a signal handler
	if (!forked_child_) {
		forkblock_mtx_.unlock();
	}
}

void atfork_check_forkblocks()
{
	// Last line of defense.
	if (forkblocks_) {
		_exit(1);
	}
	forked_child_ = true;
}


int const atfork_registered = []() {
	return pthread_atfork(&atfork_lock_forkblock, &atfork_unlock_forkblock, &atfork_check_forkblocks);
}();
}
#endif

}
