/*

	Random event generator definitions

	This file contains definitions common to all random
	event generators.  All implement the interface defined
	here.

*/


typedef struct {
    uint16 type;		      /* Device type code (allows supporting
					 multiple devices with common driver
					 code). */
    uint16 port;		      /* Port ID.  Meaning depends on nature
					 of hardware, and may mean nothing at
					 all. */
    uint32 baud;		      /* Baud rate for devices connected to
					 a serial port.  This may have a
					 different meaning for devices
					 connected to, for example, a
					 parallel port or SCSI string. */
} DevOpts;

typedef struct {
    char *reg_name;		      /* Name of REG, corresponding to
					 the name on the REG statement
					 in .eggrc */
    uint16 reg_type;		      /* REG type code, corresponding
					 to type field in the DevOpts
					 structure. */

    /* The following functions are used to call the methods
       of the REG device driver. */

    /* Open a device. 

       DevOpts should contain any necessary information to determine how
       this is to be done.  The return value, if >= 0, is a device handle.
       Otherwise it indicates an error. */

    int32 (*reg_OpenDev)(DevOpts *opts);

    /* Evaluate speed.

       dd is the device handle returned by opening the device.	The
       function returns the estimated number of bits/second that the
       device will produce, which should be used as a limit in setting the
       data rate. */

    int32 (*reg_EvalSpeed)(int32 dd);

    /* Sample from a device.

       dd is the device handle returned by opening the device.	The number
       of bits requested will be sampled from the device immediately
       (waiting as needed to collect the data, with no discards) and the
       result will be returned. */

    int32 (*reg_Sample)(int32 dd, uint16 bits);

    /* Discard spare data from device.

       dd is the device handle returned by opening the device.	The input
       stream is consumed until a wait would be required.  The number of
       bits discarded is returned. */

    int32 (*reg_Discard)(int32 dd);

    /* Close a device.

       Closes a specified device handle. */

    int32 (*reg_CloseDev)(int32 dd);
} REG_driver;
