/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#ifndef __box
#define __box

#include <algorithm>

template <typename T> struct box {
  T left, bottom;
  T right, top;
  T width, height;
  T midx, midy;

  double width_1, height_1;

  box () {
    left = bottom = right = top = width = height = 0;
    width_1 = height_1 = 0.0;
  }

  box (T l, T b, T r, T t) {
    operator() (l, b, r, t);
  }

  inline void operator() (T l, T b, T r, T t) {
    left = l;
    bottom = b;
    right = r;
    top = t;
    calc ();
  }

  inline void calc () {
    if (left > right) std::swap (left, right);
    if (bottom > top) std::swap (bottom, top);
    width = right - left;
    height = top - bottom;
    midx = left + width / 2;
    midy = bottom + height / 2;

    if (width != 0) width_1 = 1.0 / width;
    if (height != 0) height_1 = 1.0 / height;

  }

  inline void move (T dx, T dy) {
    left += dx;
    right += dx;
    bottom += dy;
    top += dy;
    calc ();
  }

  inline void resize (T dx, T dy) {
    right += dx;
    top += dy;
    calc ();
  }

  inline void lower_corner (T x, T y) {
    left = x;
    bottom = y;
    right = left + width;
    top = bottom + height;
    calc ();
  }

};

template <typename T> inline bool inbox (const box<T>& b, const T& x, const T& y) {
  return ((x >= b.left) && (x <= b.right) && (y >= b.bottom) && (y <= b.top));
}

#endif

