/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set sw=2 sts=2 et cin: */
/*
 * This file is part of the MUSE Instrument Pipeline
 * Copyright (C) 2007-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <muse.h>
#include <string.h>

#define PRINT_USAGE(rc)                                                        \
  fprintf(stderr, "Usage: %s INFILE OUTFILE\n", argv[0]);                      \
  cpl_end(); return (rc);

/*----------------------------------------------------------------------------*/
/**
  @brief    Convert INM bad pixel tables from FITS extension 3 to to FITS images
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char **argv)
{
  cpl_init(CPL_INIT_DEFAULT);

  if (argc <= 2) {
    /* two filenames are needed */
    PRINT_USAGE(1);
  }

  cpl_table *table = cpl_table_load(argv[1], 3, 1);
  if (!table) {
    PRINT_USAGE(10);
  }
  int nrow = cpl_table_get_nrow(table);
  cpl_msg_info(__func__, "read %d columns from \"%s\"", nrow, argv[1]);

  /* offset all table coordinates by +1 to get normal FITS coordinates */
  cpl_table_add_scalar(table, "xdet", 1);
  cpl_table_add_scalar(table, "ydet", 1);
  /* convert type column to Euro3D flag */
  cpl_table_new_column(table, "euro3d", CPL_TYPE_INT);
  int n, nhot = 0, ndark = 0, ntrap = 0;
  for (n = 0; n < nrow; n++) {
    /* get first character of type string in this row */
    char type = cpl_table_get_string(table, "type", n)[0];
    switch (type) {
    case 'H':
      cpl_table_set(table, "euro3d", n, EURO3D_HOTPIXEL);
      nhot++;
      break;
    case 'D':
      cpl_table_set(table, "euro3d", n, EURO3D_DARKPIXEL);
      ndark++;
      break;
    case 'T':
      cpl_table_set(table, "euro3d", n, EURO3D_MUSE_TRAP);
      ntrap++;
      break;
    } /* switch */
  } /* for n (table rows) */
#if 0
  cpl_table_dump(table, 0, 100000, stdout);
  fflush(stdout);
#endif
  cpl_msg_info(__func__, "%d hot, %d dark, and %d trap pixel%s",
               nhot, ndark, ntrap, nhot+ndark+ntrap != 1 ? "s" : "");

  /* use image part to create output image */
  muse_image *image = muse_image_new();
  image->data = cpl_image_load(argv[1], CPL_TYPE_INT, 0, 1);
  int nx = cpl_image_get_size_x(image->data),
      ny = cpl_image_get_size_y(image->data);
  /* want an empty image to start with */
  image->dq = cpl_image_new(nx, ny, CPL_TYPE_INT);

  for (n = 0; n < nrow; n++) {
    int i = cpl_table_get_int(table, "xdet", n, NULL),
        j = cpl_table_get_int(table, "ydet", n, NULL),
        dq = cpl_table_get_int(table, "euro3d", n, NULL),
        err, val = cpl_image_get(image->dq, i, j, &err);
    cpl_image_set(image->dq, i, j, dq | val);
  } /* for n (table rows) */

  /* now trim the image to be able to compare it to the MASTER file */
  image->header = cpl_propertylist_load(argv[1], 1);
  muse_image *trimmed = muse_quadrants_trim_image(image); 
  cpl_image_save(trimmed->dq, argv[2], CPL_TYPE_INT, NULL, CPL_IO_CREATE);
  cpl_msg_info(__func__, "saved to \"%s\"", argv[2]);

  muse_image_delete(trimmed);
  cpl_table_delete(table);
  muse_image_delete(image);
  cpl_end();
  return 0;
}
